package org.lsst.ccs.messaging;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Tracks agent connection and disconnection on the buses.
 *
 * @author emarin
 */
public class AgentPresenceManager implements StatusMessageListener {

    private final Object agentsLock = new Object();
    final CopyOnWriteArrayList<AgentPresenceListener> listAPL = new CopyOnWriteArrayList<>();
    private final Map<AgentInfo, AgentPresenceManager.TimeoutTask> mapAgents = new HashMap<>();
    private static final int SUSPICION_LENGTH = 10;
    private final Timer timer = new Timer(true);
    protected static Logger log = Logger.getLogger("org.lsst.ccs.messaging.agentpresencemanager");
    /** Lock used by clients waiting for a particular Agent to come alive. */
    private final ReentrantLock agentWaitLock = new ReentrantLock(true);
    /** List of clients waiting for a particular Agent to come alive. Guarded by {@code stateWaitLock}. */
    private volatile LinkedHashMap<Condition,String> agentWaitList;

    class TimeoutTask extends TimerTask {
        
        private final AgentInfo agent;
        private final int broadcastPeriod;
        
        TimeoutTask(AgentInfo agent, int broadcastPeriod) {
            this.agent = agent;
            this.broadcastPeriod = broadcastPeriod;
        }

        int getBroadcastPeriod() {
            return broadcastPeriod;
        }

        @Override
        public void run() {
            removeAgent(agent);
        }
    }

    @Override
    public void onStatusMessage(StatusMessage s) {

        AgentInfo a = (s.getOriginAgentInfo());
        if (a == null) return;

        int broadCastPeriod = -1;
        if (s instanceof StatusHeartBeat) {
            StatusHeartBeat hb = (StatusHeartBeat) s;
            broadCastPeriod = hb.getStatusBroadcastPeriod();
        }
        if (s.getState().isInState(PhaseState.OFF_LINE)) {
            log.debug("remove agent on status end");
            removeAgent(a);
        } else {
            updateAgent(a, broadCastPeriod);
        }
    }

    /** Called on non-OFF_LINE agent status message. */
    private void updateAgent(AgentInfo a, int broadcastPeriod) {
        synchronized (agentsLock) {
            AgentPresenceManager.TimeoutTask task = mapAgents.get(a);
            if (task != null) { // Agent already known to the map : its timeouttask is updated
                task.cancel();
                if (broadcastPeriod == -1) {
                    broadcastPeriod = task.getBroadcastPeriod();
                }
                task = new AgentPresenceManager.TimeoutTask(a, broadcastPeriod);
                addAgent(a, task, false);
            } else { // New agent
                if (broadcastPeriod == -1) {
                    broadcastPeriod = 10;
                }
                task = new AgentPresenceManager.TimeoutTask(a, broadcastPeriod);
                addAgent(a, task, true);
            }
            log.debug("resetting timer for agent " + a.getName() + " to " + broadcastPeriod);
            timer.schedule(task, SUSPICION_LENGTH * 1000 * broadcastPeriod);
        }
    }

    /** Called to remove agent. */
    private void removeAgent(AgentInfo agent) {
        synchronized (agentsLock) {
            log.debug("removing agent " + agent.getName());
            AgentPresenceManager.TimeoutTask t = mapAgents.remove(agent);
            if (t != null) {
                t.cancel();
                disconnecting(agent);
            } else {
                log.debug("removing agent with null timer");
            }
        }
    }

    /** Called to add agent. */
    private void addAgent(AgentInfo agent, AgentPresenceManager.TimeoutTask task, boolean isNewAgent) {
        synchronized (agentsLock) {
            mapAgents.put(agent, task);
            if (isNewAgent) {
                connecting(agent);
            }
        }
    }

    /**
     *
     * @return The list with the currently connected Agents
     */
    public List<AgentInfo> listConnectedAgents() {
        synchronized (agentsLock) {
            return new ArrayList<>(mapAgents.keySet());
        }
    }

    /**
     * Warns the list of listeners that an agent has connected
     *
     * @param agent
     */
    private void connecting(AgentInfo agent) {
        agentWaitNotify(agent.getName());
        for (AgentPresenceListener l : listAPL) {
            l.connecting(agent);
        }
    }

    /**
     * Warns the list of listeners that an agent has disconnected
     *
     * @param agent
     */
    private void disconnecting(AgentInfo agent) {
        for (AgentPresenceListener l : listAPL) {
            l.disconnecting(agent);
        }
    }

    /**
     * This method is called by transport layers that provide disconnection
     * suspicion information
     *
     * @param agentName
     */
    void disconnecting(String agentName) {
        synchronized (agentsLock) {
            AgentInfo agentToRemove = null;
            for ( AgentInfo a : mapAgents.keySet() ) {
                if ( a.getName().equals(agentName) ) {
                    agentToRemove = a;
                    break;
                }
            }
            if ( agentToRemove != null ) {
                removeAgent(agentToRemove);
            }
        }        
    }

    public void addAgentPresenceListener(AgentPresenceListener l) {
        synchronized (agentsLock) {
            for (AgentInfo a : listConnectedAgents()) {
                l.connecting(a);
            }
            listAPL.add(l);
        }
    }

    public void removeAgentPresenceListener(AgentPresenceListener l) {
        listAPL.remove(l);
    }

    public boolean agentExists(String agentName) {
        for (AgentInfo a : listConnectedAgents()) {
            if (a.getName().equals(agentName)) {
                return true;
            }
        }
        return false;
    }

    /**
     * Wait for an Agent to be available on the buses.
     * @param name The name of the Agent
     * @param timeout The time to wait for the agent to become available.
     * @param unit The TimeUnit for the timeout.
     * @return True if the Agent is available within the provided timeout, false otherwise.
     * @throws InterruptedException 
     */
    public final boolean waitForAgent(String name, long timeout, TimeUnit unit) throws InterruptedException {
        long deadline = TimeUnit.MILLISECONDS.convert(timeout, unit) + System.currentTimeMillis();
        if (!agentWaitLock.tryLock(timeout, unit)) return false;
        try {
            if ( agentExists(name) ) {
                return true;
            }
            Condition condition = agentWaitLock.newCondition();
            if (agentWaitList == null) agentWaitList = new LinkedHashMap<>(4);
            agentWaitList.put(condition, name);
            while (agentWaitList != null && agentWaitList.containsKey(condition)) {
                timeout = deadline - System.currentTimeMillis();
                if (!condition.await(timeout, TimeUnit.MILLISECONDS)) return false;
            }
        } finally {
            agentWaitLock.unlock();
        }
        return true;
    }
    
    private void agentWaitNotify(final String name) {
        if (agentWaitList != null) {
            try {
                agentWaitLock.lockInterruptibly();
                if (agentWaitList == null) {
                    return;
                }
                Iterator<Map.Entry<Condition, String>> it = agentWaitList.entrySet().iterator();
                while (it.hasNext()) {
                    Map.Entry<Condition, String> entry = it.next();
                    String agentWaitName = entry.getValue();
                    if (name.equals(agentWaitName)) {
                        entry.getKey().signal();
                        it.remove();
                    }
                }
                if (agentWaitList.isEmpty()) {
                    agentWaitList = null;
                }
            } catch (InterruptedException x) {
            } finally {
                agentWaitLock.unlock();
            }
        }
    }
    
}
