package org.lsst.ccs.messaging;

import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * Entry point for receiving messages from the buses.
 * Its name is used as a unique identifier on the buses.
 * @author emarin
 */
public class MessagingAccessLayer {
    
    private final String name;
    private final EnumMap<Bus, BusAccess> busAccesses = new EnumMap<>(Bus.class);

    public MessagingAccessLayer(String name, BusAccess... buses){
        this.name = name;
        for (BusAccess busAccess : buses) {
            busAccesses.put(busAccess.getBus(), busAccess);
        }
    }

    /**
     * This name must be unique as it is used by the messaging layer to send or
     * dispatch messages.
     * This name is determined by the associated agent's name.
     * Its uniqueness is enforced when connecting with the messaging layer.
     * @return the name of this MessagingAccessLayer 
     */
    public String getName(){
        return name;
    }
    
    /**
     * This list of buses might be used by the messaging layer to determine on which buses
     * the associated agent has to be connected to.
     * @return The list of Buses the layer has to be registered on.
    */
    public Collection<BusAccess> getBusAccesses(){
        return busAccesses.values();
    }
    
    public BusAccess getBusAccess(Bus bus){
        return busAccesses.get(bus);
    }
    
    /**
     * Handler for processing received messages.
     * Whenever a message is received, {@code update(...)} methods of all forwarders
     * listed by {@code getForwarderList()} are called, then {@code processBusMessage(...)} is called.
     * 
     * @param <T> the type of the received message
     */
    public static class BusAccess<T extends BusMessage> {

        // The forwarder map must preserve insertion order to ensure the 
        // agent presence manager will be the first one notified for status events
        private final Map<MessageListener, BusMessageForwarder> forwarderMap = 
                Collections.synchronizedMap(new LinkedHashMap<>());
        private final Bus bus;
        
        public BusAccess(Bus bus) {
            this.bus = bus;
        }
        
        public Bus getBus(){
            return bus;
        }
        
        public void processBusMessage(T message){
            synchronized(forwarderMap) {
                Iterator<BusMessageForwarder> i = forwarderMap.values().iterator();
                while(i.hasNext()) {
                    i.next().update(message);
                }
            }
        }
        
        public void addForwarder(MessageListener l, BusMessageForwarder forwarder){
            forwarderMap.put(l, forwarder);
        }
        
        public void removeForwarder(MessageListener l){
            forwarderMap.remove(l);
        }

        public void processClusterDeserializationError(String address) {

        }
        
    }
    
    public static abstract class StatusBusAccess extends BusAccess<StatusMessage>{

        public StatusBusAccess() {
            super(Bus.STATUS);
        }
        
        public abstract void processDisconnectionSuspicion(String address);
        
        public abstract void processAnormalEvent(Exception ex);
        
    }
    
}
