package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.utilities.logging.Logger;

public class StateBundleAggregator implements StatusMessageListener {

	static Logger log = Logger.getLogger("lsst.ccs.bus");

	public StateBundleAggregator() {
	}

	Set<String> origins = new HashSet<String>();
	Map<String, StateBundle> states = new ConcurrentHashMap<String, StateBundle>();

	public static interface Observer {
		public void stateChanged(String sysName, StateBundle out, StateBundle changed);
	}

	// we could delegate the filter to the messaging layer through a filter
	// we do it internally to do it in a more dynamical way, and let
	// clients register us to the bus without knowing our internals

	public void addOrigin(String origin) {
		origins.add(origin);
	}

	@Override
	public void onStatusMessage(StatusMessage msg) {
		String origin = msg.getOriginAgentInfo().getName();
		if (!origins.contains(origin))
			return;
		StateBundle bundle = msg.getState();

		StateBundle oldBundle = states.get(origin);
		if (oldBundle != null)
			bundle = oldBundle.mergeState(bundle);
		// log.fine(" sba new: "+bundle);
		states.put(origin, bundle);

		if (oldBundle != null && !oldBundle.equals(bundle)) {
			StateBundle diff = bundle.diffState(oldBundle);
			StateBundle outDiff = oldBundle.diffState(bundle);
			log.debug("state of " + origin + " changed : " + diff);
			notifyObservers(origin, outDiff, diff);
		}

	}

	ArrayList<Observer> observers = new ArrayList<Observer>();

	public synchronized void addObserver(Observer o) {
		observers.add(o);
	}

	public synchronized void deleteObserver(Observer o) {
		observers.remove(o);
	}

	public void notifyObservers(String origin, StateBundle old, StateBundle change) {
		ArrayList<Observer> obs;
		synchronized (this) {
			obs = new ArrayList<Observer>(observers);
		}
		for (Observer o : obs) {
			o.stateChanged(origin, old, change);
		}

	}

	public StateBundle getState(String origin) {
		return states.get(origin);
	}

	public Map<String, StateBundle> getStates() {
		return Collections.unmodifiableMap(states);
	}

}
