package org.lsst.ccs.messaging;

import java.io.Serializable;
import java.time.Duration;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.function.Predicate;

import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * Utility class to synchronously or asynchronously invoke or listen for events
 * on the buses.
 *
 * Synchronous command execution will wait a given time before throwing a
 * {@code TimeoutException}. The timeout used when throwing an exception depends
 * on the method invoked, the ConcurrentMessagingUtils configuration and the
 * command invoked.
 * If a method that takes a timeout is invoked, that timeout is used, otherwise
 * a cascade approach is used to determine the timeout among the following:
 *
 * <ul>
 * <li>The default timeout set on the ConcurrentMessagingUtils object, either at
 * construction time or by invoking the method {@link #setDefaultTimeout(Duration) setDefaultTimeout}
 * method.</li>
 * <li>The timeout value that comes from the annotated {@code Command} that is invoked</li>
 * <li>The timeout returned as part of the {@code CommandAck} when the invoked
 * {@code Command} is accepted</li>
 * </ul>
 *
 * Of the above possible timeouts, the last one defined out of the presented order
 * is picked and used.
 *
 * If none of the above timeouts is defined, an Exception will be thrown.
 *
 * @author The LSST CCS Team
 */
public final class ConcurrentMessagingUtils {

    private final AgentMessagingLayer agentMessagingLayer;
    private final static Object NULL = new Object();
    private Duration defaultTimeout = null;

    /**
     * ConsurrentMessagingUtils constructor.
     *
     * @param agentMessagingLayer The {@code AgentMessagingLayer} used to send messages on the buses.
     */
    public ConcurrentMessagingUtils(AgentMessagingLayer agentMessagingLayer) {
        this(agentMessagingLayer, Duration.ofSeconds(5));
    }

    /**
     * Build a ConsurrentMessagingUtils object with a default timeout.
     *
     * @param agentMessagingLayer The {@code AgentMessagingLayer} used to send messages on the buses.
     * @param defaultTimeout The default timeout used when synchronous commands are invoked.
     */
    public ConcurrentMessagingUtils(AgentMessagingLayer agentMessagingLayer, Duration defaultTimeout) {
        this.agentMessagingLayer = agentMessagingLayer;
        this.defaultTimeout = defaultTimeout;
    }

    /**
     * Set the default timeout for this ConcurrentMessagingUtils object.
     * The default timeout will be used to determine the timeout to be used
     * when a synchronous method without a timeout is invoked.
     *
     * @param defaultTimeout The default timeout.
     */
    public void setDefaultTimeout(Duration defaultTimeout) {
        this.defaultTimeout = defaultTimeout;
    }

    /**
     * Get the default timeout for this ConcurrentMessagingUtils object.
     *
     * @return The default timeout Duration.
     */
    public Duration getDefaultTimeout() {
        return defaultTimeout;
    }

    /**
     * Send synchronously a command on the Buses without a timeout.
     * The timeout used will be the last one defined out of the following:
     * <ul>
     * <li>default timeout</li>
     * <li>timeout defined in the {@code Command} annotation</li>
     * <li>timeout returned as part of the {@code CommandAck}</li>
     * </ul>
     * If none of the above is defined, an exception will be thrown.
     *
     * @param command
     * The CommandRequest object to be sent on the buses.
     * @return The reply of the CommandRequest.
     * @throws Exception
     * If an exception was fired by the remote execution of the
     * command or the timeout expired or no valid timeout could
     * be found.
     *
     */
    public Object sendSynchronousCommand(CommandRequest command) throws Exception {
        return invokeIt(false, command, defaultTimeout, false);
    }

    /**
     * Send a command on the Buses and wait for the reply within the provided
     * timeout in milliseconds. If the command does not yield a reply within
     * the provided timeout a TimeoutException will be thrown.
     *
     * @param command
     * The CommandRequest object to be sent on the buses.
     * @param millisTimeout
     * Timeout in milliseconds. If the reply is not received within
     * the timeout a TimeoutException will be thrown.
     * @return The reply of the CommandRequest.
     * @throws Exception
     * If an exception was fired by the remote execution of the
     * command or the timeout expired.
     *
     * @deprecated Use method {@link #sendSynchronousCommand(org.lsst.ccs.bus.messages.CommandRequest, Duration)} instead.
     *
     */
    @Deprecated
    public Object sendSynchronousCommand(CommandRequest command,
            long millisTimeout) throws Exception {
        return invokeIt(false, command, Duration.ofMillis(millisTimeout), true);
    }

    /**
     * Send a command on the Buses and wait for the reply within the provided
     * Duration timeout. If the command does not yield a reply within
     * the provided timeout a TimeoutException will be thrown.
     *
     * @param command
     * The CommandRequest object to be sent on the buses.
     * @param timeout
     * The Duration timeout. If the reply is not received within
     * the timeout a TimeoutException will be thrown.
     * @return The reply of the CommandRequest.
     * @throws Exception
     * If an exception was fired by the remote execution of the
     * command or the timeout expired.
     *
     */
    public Object sendSynchronousCommand(CommandRequest command,
            Duration timeout) throws Exception {
        return invokeIt(false, command, timeout, true);
    }

    /**
     * Send a command on the buses and immediately return a Future that will
     * asynchronously listen for the command reply.
     *
     * @param command
     * The CommandRequest object to be sent on the buses.
     * @return A Future on the reply of the command execution. The future will
     * also contain any possible exception thrown during the command
     * execution.
     *
     */
    public Future<Object> sendAsynchronousCommand(CommandRequest command) {
        LinkedCommandOriginator commandOriginator = new LinkedCommandOriginator(
                false, agentMessagingLayer);
        LinkedFuture<Object> linkedFuture = new LinkedFuture(commandOriginator,
                false);
        linkedFuture.init();
        agentMessagingLayer.sendCommandRequest(command, commandOriginator);
        return linkedFuture;
    }

    /**
     * Send a CommandRequest on the buses and synchronously wait for the Ack to
     * come back within the provided timeout in milliseconds.
     *
     * @param command
     * The CommandRequest object to be sent on the buses.
     * @param millisTimeout
     * Timeout in milliseconds. If the ack is not received within the
     * timeout a TimeoutException will be thrown.
     * @return The CommandAck for the CommandRequest.
     * @throws Exception
     * If an exception was fired by the remote execution of the
     * command or the timeout expired.
     *
     * @deprecated Use method {@link #getAckForCommand(org.lsst.ccs.bus.messages.CommandRequest, Duration)} instead.
     *
     */
    @Deprecated
    public Object getAckForCommand(CommandRequest command, long millisTimeout)
            throws Exception {
        return invokeIt(true, command, Duration.ofMillis(millisTimeout), true);
    }

    /**
     * Send a CommandRequest on the buses and synchronously wait for the Ack to
     * come back within the provided Duration timeout.
     *
     * @param command
     * The CommandRequest object to be sent on the buses.
     * @param timeout
     * Duration timeout. If the {@code CommandAck} is not received within the
     * timeout a TimeoutException will be thrown.
     * @return The CommandAck for the CommandRequest.
     * @throws Exception
     * If an exception was fired by the remote execution of the
     * command or the timeout expired.
     *
     */
    public Object getAckForCommand(CommandRequest command, Duration timeout)
            throws Exception {
        return invokeIt(true, command, timeout, true);
    }

    private Object invokeIt(boolean ackOnly, CommandRequest command,
            Duration timeout, boolean isTimeoutUserProvided) throws Exception {
        
        long timeoutMillis = 0l;
        TimeUnit timeoutUnits = TimeUnit.MILLISECONDS;
       
        if ( timeout == null ) {
            if ( isTimeoutUserProvided ) {
                throw new IllegalArgumentException("Provided timeout cannot be null");
            } else {
                timeoutMillis = -1;                
            }
        } else {
            timeoutMillis = timeout.toMillis();
        }
        
        LinkedCommandOriginator commandOriginator = new LinkedCommandOriginator(
                ackOnly, agentMessagingLayer);
        SynchLinkedFuture<Object> linkedFuture = new SynchLinkedFuture(commandOriginator,
                false, isTimeoutUserProvided);
        linkedFuture.init();
        agentMessagingLayer.sendCommandRequest(command, commandOriginator);
        Object res = null;
        try {
            res = linkedFuture.get(timeoutMillis, timeoutUnits);
        } catch (TimeoutException exc) {
            throw exc;
        }
        if (res instanceof Exception) {
            throw (Exception) res;
        }
        return res;
    }

    /**
     * Get a Future on a StatusBusMessage. The content of the Future is filled
     * when the first StatusBusMessage that satisfies the
     * ScriptingStatusBusMessageFilter is received. When the Future is exercised
     * it will return the first ScriptingStatusBusMessage or an
     * ExecutionException will be thrown when the timeout is reached.
     *
     * @param filter
     * ScriptingStatusBusMessageFilter The message filter
     * @param timeout
     * Timeout in milliseconds, after which a
     * ScriptiongTimeoutException is thrown. This timeout is from the
     * time the method is invoked. If this timeout is reached an
     * exception will be thrown if/when the Future is exercised.
     * @return A Future on a ScriptingStatusBusMessage.
     * 
     * @deprecated use method {@link #startListeningForStatusBusMessage(java.util.function.Predicate,Duration)} instead 
     * 
     */
    @Deprecated
    public Future<StatusMessage> startListeningForStatusBusMessage(
            Predicate<BusMessage<? extends Serializable, ?>> filter, long timeout) {

        LinkedStatusBusListener innerListener = new LinkedStatusBusListener(
                filter, timeout, this.agentMessagingLayer);
        LinkedFuture future = new LinkedFuture<>(innerListener, true);
        future.init();
        this.agentMessagingLayer
                .addStatusMessageListener(innerListener, filter);

        return future;
    }

    /**
     * Get a Future on a StatusBusMessage. The content of the Future is filled
     * when the first StatusBusMessage that satisfies the
     * ScriptingStatusBusMessageFilter is received. When the Future is exercised
     * it will return the first ScriptingStatusBusMessage or an
     * ExecutionException will be thrown when the timeout is reached.
     *
     * @param filter
     * ScriptingStatusBusMessageFilter The message filter
     * @param timeout
     * Duration timeout, after which a
     * ScriptiongTimeoutException is thrown. This timeout is from the
     * time the method is invoked. If this timeout is reached an
     * exception will be thrown if/when the Future is exercised.
     * @return A Future on a ScriptingStatusBusMessage.
     */
    public Future<StatusMessage> startListeningForStatusBusMessage(
            Predicate<BusMessage<? extends Serializable, ?>> filter, Duration timeout) {

        LinkedStatusBusListener innerListener = new LinkedStatusBusListener(
                filter, timeout.toMillis(), this.agentMessagingLayer);
        LinkedFuture future = new LinkedFuture<>(innerListener, true);
        future.init();
        this.agentMessagingLayer
                .addStatusMessageListener(innerListener, filter);

        return future;
    }

    /**
     * Get a Future on a StatusBusMessage. The content of the Future is filled
     * when the first StatusBusMessage that satisfies the
     * ScriptingStatusBusMessageFilter is received. When the Future is exercised
     * it will return the first ScriptingStatusBusMessage.
     *
     * @param f
     * ScriptingStatusBusMessageFilter The message filter
     * @return A Future on a ScriptingStatusBusMessage.
     */
    public Future<StatusMessage> startListeningForStatusBusMessage(
            Predicate<BusMessage<? extends Serializable, ?>> f) {
        return startListeningForStatusBusMessage(f, Duration.ofMillis(-1));
    }

    class LinkedStatusBusListener extends LinkedTask<StatusMessage> implements
            StatusMessageListener {

        private final Predicate<BusMessage<? extends Serializable, ?>> filter;
        private final Timer timeoutTimer = new Timer("LinkedStatusBusListener");
        private boolean cleanedUp = false;
        private final AgentMessagingLayer agentMessagingLayer;
        private final long timeout;

        LinkedStatusBusListener(Predicate<BusMessage<? extends Serializable, ?>> filter, long timeout,
                AgentMessagingLayer agentMessagingLayer) {
            this.filter = filter;
            this.agentMessagingLayer = agentMessagingLayer;
            this.timeout = timeout;
        }

        @Override
        public void start() {
            if (timeout > 0) {
                timeoutTimer.schedule(new TimerTask() {
                    @Override
                    public void run() {
                        cancel();
                        TimeoutException ex = new TimeoutException(
                                "Timeout listening for filtered events "
                                + filter.toString());
                        getLinkedFuture().addToQueue(ex);
                    }
                }, timeout);
            }

        }

        @Override
        public Duration getTaskInternalTimeout() {
            return null;
        }

        @Override
        public void stop() {
            cancel();
        }

        @Override
        public void cancel() {
            if (!cleanedUp) {
                agentMessagingLayer.removeStatusMessageListener(this);
                cleanedUp = true;
            }
        }

        @Override
        public void onStatusMessage(StatusMessage bm) {
            if (!getLinkedFuture().isDone()) {
                timeoutTimer.cancel();
                getLinkedFuture().addToQueue(bm);
            }
        }

    }

    private class LinkedCommandOriginator extends LinkedTask<Object> implements
            CommandOriginator {

        private final boolean getAckOnly;
        private volatile Duration timeout = null;
        private boolean gotAck = false;
        private final Object ackLock = new Object();

        LinkedCommandOriginator(boolean ackOnly,
                AgentMessagingLayer agentMessagingLayer) {
            this.getAckOnly = ackOnly;
        }

        @Override
        public void cancel() {
        }

        @Override
        public void start() {
        }

        @Override
        public void stop() {
        }

        @Override
        public Duration getTaskInternalTimeout() {
            try {
                synchronized (ackLock) {
                    if (!gotAck) {
                        ackLock.wait();
                    }
                }
                return timeout;
            } catch (InterruptedException ie) {
                throw new RuntimeException("Problem getting task internal timeout ", ie);
            }
        }

        @Override
        public void processNack(CommandNack nack) {
            CommandRejectedException rejection = new CommandRejectedException(
                    nack);
            getLinkedFuture().addToQueue(rejection);
        }

        @Override
        public void processResult(CommandResult result) {
            if (getAckOnly) {
                return;
            }
            Object resultContent;
            //Try to collect the actual result first, and if that's not possible
            //get its encoded version
            try {
                resultContent = result.getResult();
            } catch (Exception e) {
                resultContent = result.getEncodedData();
            }
            getLinkedFuture().addToQueue(resultContent);
        }

        @Override
        public void processAck(CommandAck ack) {
            synchronized(ackLock) {
                timeout = ack.getTimeout();
                gotAck = true;
                ackLock.notifyAll();
            }
            if (getAckOnly) {
                getLinkedFuture().addToQueue(ack);

            }
        }
    }

    abstract class LinkedTask<T> {

        LinkedFuture<T> future = null;

        public abstract void cancel();

        public abstract void start();

        public abstract void stop();

        public abstract Duration getTaskInternalTimeout();

        void setLinkedFuture(LinkedFuture<T> future) {
            this.future = future;
            start();
        }

        LinkedFuture<T> getLinkedFuture() {
            return future;
        }

    }

    class LinkedFuture<T extends Object> implements Future<T> {

        private final LinkedTransferQueue<Object> queue = new LinkedTransferQueue<>();
        protected final LinkedTask<T> task;
        private boolean isCancelled = false;
        private final boolean throwException;
        private boolean initialized = false;

        LinkedFuture(LinkedTask<T> task, boolean throwException) {
            this.task = task;
            this.throwException = throwException;
        }

        protected void init() {
            if ( initialized ) {
                throw new RuntimeException("LinkedFuture must be initialized only once");
            }
            initialized = true;
            task.setLinkedFuture(this);
        }

        @Override
        public boolean isCancelled() {
            if ( ! initialized ) {
                throw new RuntimeException("LinkedFuture must be initialized first");
            }
            return isCancelled;
        }

        @Override
        public boolean isDone() {
            if ( ! initialized ) {
                throw new RuntimeException("LinkedFuture must be initialized first");
            }
            return !queue.isEmpty();
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            if ( ! initialized ) {
                throw new RuntimeException("LinkedFuture must be initialized first");
            }
            if (!isCancelled) {
                task.cancel();
                isCancelled = true;
            }
            return true;
        }

        @Override
        public T get() throws InterruptedException, ExecutionException {
            if ( ! initialized ) {
                throw new RuntimeException("LinkedFuture must be initialized first");
            }
            return processReply(queue.take());
        }

        @Override
        public T get(long timeout, TimeUnit unit) throws InterruptedException,
                ExecutionException, TimeoutException {
            if ( ! initialized ) {
                throw new RuntimeException("LinkedFuture must be initialized first");
            }
            Object reply = queue.poll(timeout, unit);
            if (reply == null) {
                throw new TimeoutException("Could not get reply within the specified timeout of " + timeout + " " + unit.toString());
            }
            return processReply(reply);

        }

        private T processReply(Object reply) throws InterruptedException,
                ExecutionException {
            if ( ! initialized ) {
                throw new RuntimeException("LinkedFuture must be initialized first");
            }
            if (reply instanceof Exception && throwException) {
                throw new ExecutionException("Execution Exception",
                        (Exception) reply);
            }
            return reply != NULL ? (T) reply : null;
        }

        void addToQueue(Object obj) {
            if ( ! initialized ) {
                throw new RuntimeException("LinkedFuture must be initialized first");
            }
            if (obj == null) {
                obj = NULL;
            }
            queue.offer(obj);
            task.stop();
        }

    }

    class SynchLinkedFuture<T extends Object> extends LinkedFuture {

        private final boolean isTimeoutUserProvided;
        
        SynchLinkedFuture(LinkedTask<T> task, boolean throwException, boolean isTimeoutUserProvided) {
            super(task, throwException);
            this.isTimeoutUserProvided = isTimeoutUserProvided;
        }

        @Override
        public T get(long timeout, TimeUnit unit) throws InterruptedException,
                ExecutionException, TimeoutException {
            long pollTimeout = timeout;
            TimeUnit pollTimeUnit = unit;
            if ( ! isTimeoutUserProvided ) {
                long start = System.currentTimeMillis();
                Duration internalTaskDuration = task.getTaskInternalTimeout();
                if ( internalTaskDuration != null ) {
                    pollTimeout = internalTaskDuration.toMillis();
                    pollTimeUnit = TimeUnit.MILLISECONDS;
                }
                pollTimeout -= System.currentTimeMillis() - start;
            }
            return (T) super.get(pollTimeout, pollTimeUnit);
        }
    }

}
