package org.lsst.ccs.messaging;

import java.io.Serializable;
import java.util.function.Predicate;

import org.lsst.ccs.bus.messages.BusMessage;

/**
 * A BusMessageFilter provides methods for creating and combining filters for
 * BusMessages. A BusMessageFilter can be used when registering a Bus Listener.
 * An example of creating a BusMessageFilter selecting for BusMessages from an
 * origin and of a given type is:
 * 
 * BusMessageFilter filter =
 * BusMessageFilter.messageOrigin("abc").and(BusMessageFilter
 * .messageClass(SomeMessage.class));
 * 
 * 
 * Deprecated : use BusMessageFilterFactory to create Predicate<BusMessage> objects
 * without encapsulation
 *
 */

@Deprecated
public abstract class BusMessageFilter implements Predicate<BusMessage<? extends Serializable, ?>> {

    /**
     * Create BusMessageFilter on the origin of a BusMessage. The filter will
     * select BusMessages from the provided origin.
     * 
     * @param origin
     *            The origin of the desired BusMessages
     * @return A BusMessageFilter on the origin of a BusMessage.
     * 
     */
    public static BusMessageFilter messageOrigin(String origin) {
        return new OriginBusMessageFilter(origin);
    }

    /**
     * Create BusMessageFilter on the class of the object embedded in a StatusMessage. The filter will
     * select StatusMessages that contain an object that can be assigned from the provided clazz.
     * 
     * @param clazz
     *            The clazz of the desired object embedded in a StatusMessages
     * @return A BusMessageFilter on the class of an object embedded in a StatusMessage.
     * 
     */
    public static BusMessageFilter embeddedObjectClass(Class clazz) {
        return new EmbeddedObjectClassBusMessageFilter(clazz);
    }
    
    
    /**
     * Create BusMessageFilter on the class of a BusMessage. The filter will
     * select BusMessages that can be assigned from the provided clazz.
     * 
     * @param clazz
     *            The clazz of the desired BusMessages
     * @return A BusMessageFilter on the origin of a BusMessage.
     * 
     */
    public static BusMessageFilter messageClass(Class clazz) {
        return new ClassBusMessageFilter(clazz);
    }

    /**
     * Check if a BusMessage passes this filter.
     * 
     * @param busMessage
     *            The BusMessage to test.
     * @return True if the BusMessage passes the BusMessageFilter.
     * 
     */
    public boolean accept(BusMessage busMessage) {
        return this.internalAccept(busMessage);
    }

    /**
     * Check if a BusMessage passes this filter. Implemented to call
     * accept(busMessage).
     * 
     * @param busMessage
     *            The BusMessage to test.
     * @return True if the BusMessage passes the BusMessageFilter.
     * 
     */
    @Override
    public boolean test(BusMessage busMessage) {
        return accept(busMessage);
    }

    public BusMessageFilter or(BusMessageFilter filter) {
        Predicate<BusMessage<? extends Serializable, ?>> f = filter;
        return new PredicateBasedMessageFilter(this.or(f), "("
                + internalToString() + " || " + filter.internalToString() + ")");
    }

    public BusMessageFilter and(BusMessageFilter filter) {
    	Predicate<BusMessage<? extends Serializable, ?>> f = filter;
        return new PredicateBasedMessageFilter(this.and(f), "("
                + internalToString() + " && " + filter.internalToString() + ")");
    }

    public BusMessageFilter negate() {

        // this syntax should work, and call the default negate!!!
        // return new
        // PredicateBasedMessageFilter(Predicate<BusMessage>.super.negate(),
        // "(!"
        // + internalToString() + ")");

    	Predicate<BusMessage<? extends Serializable, ?>> rawPred = (m) -> accept(m);
        return fromPredicate(rawPred.negate());
    }

    public static BusMessageFilter fromPredicate(Predicate<BusMessage<? extends Serializable, ?>> p) {
        return new PredicateBasedMessageFilter(p, "predicate");
    }

    protected abstract boolean internalAccept(BusMessage busMessage);

    protected abstract String internalToString();

    // to keep code using BusMessageFilters compatible with Predicate
    // implementation
    // todo : move all code to Predicate<BusMessage> ?

    private static class PredicateBasedMessageFilter extends BusMessageFilter {
    	Predicate<BusMessage<? extends Serializable, ?>> pred;
        String st;

        PredicateBasedMessageFilter(Predicate<BusMessage<? extends Serializable, ?>> p, String s) {
            pred = p;
            st = s;
        }

        @Override
        protected String internalToString() {
            return st;
        }

        @Override
        protected boolean internalAccept(BusMessage busMessage) {
            return pred.test(busMessage);
        }

    }

    private static class ClassBusMessageFilter extends BusMessageFilter {

        private final Class clazz;

        ClassBusMessageFilter(Class clazz) {
            this.clazz = clazz;
        }

        ClassBusMessageFilter(String clazz) {
            try {
                this.clazz = Class.forName(clazz);
            } catch (ClassNotFoundException e) {
                throw new RuntimeException(
                        "Could not properly initialize BusMessage class filter",
                        e);
            }
        }

        @Override
        public boolean internalAccept(BusMessage busMessage) {
            return clazz.isInstance(busMessage);
        }

        @Override
        protected String internalToString() {
            return " class instanceof " + clazz.getCanonicalName() + " ";
        }

    }

    
    private static class EmbeddedObjectClassBusMessageFilter extends BusMessageFilter {

        private final Class clazz;

        EmbeddedObjectClassBusMessageFilter(Class clazz) {
            this.clazz = clazz;
        }

        EmbeddedObjectClassBusMessageFilter(String clazz) {
            try {
                this.clazz = Class.forName(clazz);
            } catch (ClassNotFoundException e) {
                throw new RuntimeException(
                        "Could not properly initialize BusMessage class filter",
                        e);
            }
        }

        @Override
        public boolean internalAccept(BusMessage busMessage) {            
            return busMessage.getClassName().equals(clazz.getName());
        }

        @Override
        protected String internalToString() {
            return " embedded class instanceof " + clazz.getCanonicalName() + " ";
        }

    }
    
    private static class OriginBusMessageFilter extends BusMessageFilter {

        private final String origin;

        OriginBusMessageFilter(String origin) {
            this.origin = origin;
        }

        @Override
        public boolean internalAccept(BusMessage busMessage) {
            if (origin == null || "".equals(origin)) {
                return true;
            }
            return busMessage.getOriginAgentInfo().getName().equals(origin);
        }

        @Override
        protected String internalToString() {
            return " origin = " + origin + " ";
        }

    }

    @Override
    public String toString() {
        String result = "(" + internalToString() + ")";
        return result;
    }

}
