package org.lsst.ccs.messaging;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedTransferQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * Utility class to synchronously or asynchronously invoke or listen for events
 * on the buses.
 * 
 * @author The LSST CCS Team
 */
// TODO: make this class final after removing deprecated class
// org.lsst.ccs.ConcurrentMessagingUtils
public class ConcurrentMessagingUtils {

    private final AgentMessagingLayer agentMessagingLayer;
    private final static Object NULL = new Object();

    public ConcurrentMessagingUtils(AgentMessagingLayer agentMessagingLayer) {
        this.agentMessagingLayer = agentMessagingLayer;
    }

    /**
     * Send a command on the Buses and wait for the reply.
     *
     * @param command
     *            The CommandRequest object to be sent on the buses.
     * @param millisTimeout
     *            Timeout in milliseconds. If the reply is not received within
     *            the timeout a TimeoutException will be thrown.
     * @return The reply of the CommandRequest.
     * @throws Exception
     *             If an exception was fired by the remote execution of the
     *             command or the timeout expired.
     * 
     */
    public Object sendSynchronousCommand(CommandRequest command,
            long millisTimeout) throws Exception {
        return invokeIt(false, command, millisTimeout);
    }

    /**
     * Send a command on the buses and immediately return a Future that will
     * asynchronously listen for the command reply.
     *
     * @param command
     *            The CommandRequest object to be sent on the buses.
     * @return A Future on the reply of the command execution. The future will
     *         also contain any possible exception thrown during the command
     *         execution.
     * 
     */
    public Future<Object> sendAsynchronousCommand(CommandRequest command) {
        LinkedCommandOriginator commandOriginator = new LinkedCommandOriginator(
                false, agentMessagingLayer);
        LinkedFuture<Object> linkedFuture = new LinkedFuture(commandOriginator,
                false);
        agentMessagingLayer.sendCommandRequest(command, commandOriginator);
        return linkedFuture;
    }

    /**
     * Send a CommandRequest on the buses and synchronously wait for the Ack to
     * come back.
     * 
     * @param command
     *            The CommandRequest object to be sent on the buses.
     * @param millisTimeout
     *            Timeout in milliseconds. If the ack is not received within the
     *            timeout a TimeoutException will be thrown.
     * @return The CommandAck for the CommandRequest.
     * @throws Exception
     *             If an exception was fired by the remote execution of the
     *             command or the timeout expired.
     */
    public Object getAckForCommand(CommandRequest command, long millisTimeout)
            throws Exception {
        return invokeIt(true, command, millisTimeout);
    }

    private Object invokeIt(boolean ackOnly, CommandRequest command,
            long millisTimeout) throws Exception {
        LinkedCommandOriginator commandOriginator = new LinkedCommandOriginator(
                ackOnly, agentMessagingLayer);
        LinkedFuture<Object> linkedFuture = new LinkedFuture(commandOriginator,
                false);
        agentMessagingLayer.sendCommandRequest(command, commandOriginator);
        Object res = null;
        try {
            res = linkedFuture.get(millisTimeout, TimeUnit.MILLISECONDS);
        } catch (TimeoutException exc) {
            throw exc;
        }
        if (res instanceof Exception) {
            throw (Exception) res;
        }
        return res;
    }

    /**
     * Get a Future on a StatusBusMessage. The content of the Future is filled
     * when the first StatusBusMessage that satisfies the
     * ScriptingStatusBusMessageFilter is received. When the Future is exercised
     * it will return the first ScriptingStatusBusMessage or an
     * ExecutionException will be thrown when the timeout is reached.
     *
     * @param filter
     *            ScriptingStatusBusMessageFilter The message filter
     * @param timeout
     *            Timeout in milliseconds, after which a
     *            ScriptiongTimeoutException is thrown. This timeout is from the
     *            time the method is invoked. If this timeout is reached an
     *            exception will be thrown if/when the Future is exercised.
     * @return A Future on a ScriptingStatusBusMessage.
     */
    public Future<StatusMessage> startListeningForStatusBusMessage(
            BusMessageFilter filter, long timeout) {

        LinkedStatusBusListener innerListener = new LinkedStatusBusListener(
                filter, timeout, this.agentMessagingLayer);
        LinkedFuture future = new LinkedFuture<>(innerListener, true);
        this.agentMessagingLayer
                .addStatusMessageListener(innerListener, filter);

        return future;
    }

    /**
     * Get a Future on a StatusBusMessage. The content of the Future is filled
     * when the first StatusBusMessage that satisfies the
     * ScriptingStatusBusMessageFilter is received. When the Future is exercised
     * it will return the first ScriptingStatusBusMessage.
     *
     * @param filter
     *            ScriptingStatusBusMessageFilter The message filter
     * @return A Future on a ScriptingStatusBusMessage.
     */
    public Future<StatusMessage> startListeningForStatusBusMessage(
            BusMessageFilter filter) {
        return startListeningForStatusBusMessage(filter, -1);
    }

    class LinkedStatusBusListener extends LinkedTask<StatusMessage> implements
            StatusMessageListener {

        private final BusMessageFilter filter;
        private final Timer timeoutTimer = new Timer("LinkedStatusBusListener");
        private boolean cleanedUp = false;
        private final AgentMessagingLayer agentMessagingLayer;
        private final long timeout;

        LinkedStatusBusListener(BusMessageFilter filter, long timeout,
                AgentMessagingLayer agentMessagingLayer) {
            this.filter = filter;
            this.agentMessagingLayer = agentMessagingLayer;
            this.timeout = timeout;
        }

        @Override
        public void start() {
            if (timeout > 0) {
                timeoutTimer.schedule(new TimerTask() {
                    @Override
                    public void run() {
                        cancel();
                        TimeoutException ex = new TimeoutException(
                                "Timeout listening for filtered events "
                                        + filter.toString());
                        getLinkedFuture().addToQueue(ex);
                    }
                }, timeout);
            }

        }

        @Override
        public void stop() {
            cancel();
        }

        @Override
        public void cancel() {
            if (!cleanedUp) {
                agentMessagingLayer.removeStatusMessageListener(this);
                cleanedUp = true;
            }
        }

        @Override
        public void onStatusMessage(StatusMessage bm) {
            if (!getLinkedFuture().isDone()) {
                timeoutTimer.cancel();
                getLinkedFuture().addToQueue(bm);
            }
        }

    }

    private class LinkedCommandOriginator extends LinkedTask<Object> implements
            CommandOriginator {

        private final boolean getAckOnly;

        LinkedCommandOriginator(boolean ackOnly,
                AgentMessagingLayer agentMessagingLayer) {
            this.getAckOnly = ackOnly;
        }

        @Override
        public void cancel() {
        }

        @Override
        public void start() {
        }

        @Override
        public void stop() {
        }

        @Override
        public void processNack(CommandNack nack) {
            CommandRejectedException rejection = new CommandRejectedException(
                    nack);
            getLinkedFuture().addToQueue(rejection);
        }

        @Override
        public void processResult(CommandResult result) {
            if (getAckOnly) {
                return;
            }
            Object resultContent;
            //Try to collect the actual result first, and if that's not possible
            //get its encoded version
            try {
                resultContent = result.getResult();
            } catch (Exception e) {
                resultContent = result.getEncodedData();
            }
            getLinkedFuture().addToQueue(resultContent);
        }

        @Override
        public void processAck(CommandAck ack) {
            if (getAckOnly) {
                getLinkedFuture().addToQueue(ack);

            }
        }
    }

    abstract class LinkedTask<T> {

        LinkedFuture<T> future = null;

        public abstract void cancel();

        public abstract void start();

        public abstract void stop();

        void setLinkedFuture(LinkedFuture<T> future) {
            this.future = future;
            start();
        }

        LinkedFuture<T> getLinkedFuture() {
            return future;
        }

    }

    class LinkedFuture<T extends Object> implements Future<T> {

        private final LinkedTransferQueue<Object> queue = new LinkedTransferQueue<>();
        private final LinkedTask<T> task;
        private boolean isCancelled = false;
        private final boolean throwException;

        LinkedFuture(LinkedTask<T> task, boolean throwException) {
            this.task = task;
            this.throwException = throwException;
            init();
        }

        private void init() {
            task.setLinkedFuture(this);
        }

        @Override
        public boolean isCancelled() {
            return isCancelled;
        }

        @Override
        public boolean isDone() {
            return !queue.isEmpty();
        }

        @Override
        public boolean cancel(boolean mayInterruptIfRunning) {
            if (!isCancelled) {
                task.cancel();
                isCancelled = true;
            }
            return true;
        }

        @Override
        public T get() throws InterruptedException, ExecutionException {
            return processReply(queue.take());
        }

        @Override
        public T get(long timeout, TimeUnit unit) throws InterruptedException,
                ExecutionException, TimeoutException {
            Object reply = queue.poll(timeout,unit);
            if ( reply == null ) {
                throw new TimeoutException("Could not get reply within the specified timeout of "+timeout+" "+unit.toString());
            }
            return processReply(reply);
            
        }

        private T processReply(Object reply) throws InterruptedException,
                ExecutionException {
            if (reply instanceof Exception && throwException) {
                throw new ExecutionException("Execution Exception",
                        (Exception) reply);
            }
            return reply != NULL ? (T) reply : null;
        }

        void addToQueue(Object obj) {
            if (obj == null) {
                obj = NULL;
            }
            queue.offer(obj);
            task.stop();
        }

    }

}
