package org.lsst.ccs.messaging;

//import org.apache.log4j.Logger;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandReply;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.logging.Logger;

import java.io.IOException;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.EncodedDataStatus;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusData;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.messaging.MessagingAccessLayer.BusAccess;
import org.lsst.ccs.messaging.MessagingAccessLayer.StatusBusAccess;

/**
 * Calls a BusMessagingLayer and adds all that is necessary to handle messages
 * following application layer concerns.
 * <p>
 * Each is bound to a subsystem.
 * <p>
 * This class is meant to be subclassed if there is a need to setup complex
 * correlations between messages (or more generally for "application layer" concerns)
 * </P>
 * <B>beware</B> <TT>ThreadLocal</TT> objects may be changed in future releases: they may be
 * moved to <I>Context</I> objects carried through thread creations.
 */
class BusApplicationLayer {

    private final AgentInfo agentInfo;
    private final String subsystemName;
    private final BusMessagingLayer busMessagingLayer;
    private static final Logger log = Logger.getLogger("org.lsst.ccs.bus.layer");
    private final MessagingAccessLayer messagingAccessLayer;
    private final AgentPresenceManager agentPresenceManager;
    
    private final ForwarderToCommandExecutor commandExecutorForwarder = new ForwarderToCommandExecutor();
    private final ForwarderToCommandOriginator commandOriginatorForwarder = new ForwarderToCommandOriginator();

    /**
     * creates an entry point to communication for the subsystem.
     * One is negate supposed to interact directly with the <TT>BusMessagingLayer</TT>
     * afterwards except for administrative purposes (example: MembershipListeners).
     *
     * @param subsystemName rules of naming apply
     * @param busMessagingLayer transport layer
     * @throws NullPointerException if busMessagingLayer is null
     */
    BusApplicationLayer(AgentInfo agentInfo) {

        String protocolProperty = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.transport", "jgroups:udp_ccs:");
        String transportPropsProperty = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.transport.properties", "");
        try {
            busMessagingLayer = TransportManager.getConnection(protocolProperty, transportPropsProperty);
        } catch (TransportException exc) {
            //todo: change this exception
            throw new RuntimeException(exc);
        }
        this.agentInfo = agentInfo;
        subsystemName = agentInfo.getName();
        
        BusAccess<LogMessage> logBusAccess = new BusAccess<>(Bus.LOG);
        
        BusAccess<StatusMessage> statusBusAccess = new StatusBusAccess() {

            @Override
            public void processDisconnectionSuspicion(String address) {
                if (getAgentPresenceManager() != null) {
                    getAgentPresenceManager().disconnecting(address);
                }
            }

            @Override
            public void processAnormalEvent(Exception ex) {
                if (agentPresenceManager != null) {
                    agentPresenceManager.anormalEvent(ex);
                }
            }
        };
        
        BusAccess<CommandMessage> commandBusAccess = new BusAccess<CommandMessage>(Bus.COMMAND) {
            @Override
            public void processBusMessage(CommandMessage message) {
                super.processBusMessage(message);
                if (message instanceof CommandRequest) {
                    commandExecutorForwarder.update((CommandRequest) message);
                } else if (message instanceof CommandReply) {
                    commandOriginatorForwarder.update((CommandReply) message);
                }
            }
        };
        
        this.messagingAccessLayer = new MessagingAccessLayer(subsystemName, logBusAccess, statusBusAccess, commandBusAccess);
        agentPresenceManager = new AgentPresenceManager(getBusMessagingLayer() instanceof ProvidesDisconnectionInformation);

        addStatusListener(agentPresenceManager, BusMessageFilter.messageOrigin(this.agentInfo.getName()).negate());
        try {
            busMessagingLayer.connect(messagingAccessLayer);
        } catch (DuplicateAgentNameException|IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    final BusMessagingLayer getBusMessagingLayer() {
        return busMessagingLayer;
    }

    AgentInfo getAgentInfo() {
        return agentInfo;
    }

    /**
     * sends a command message to all destinations.
     * If origin of message is negate set then sets it using the current subsystem name.
     * If a Correlation Id is negate set then creates one.
     *
     * @param cmd
     * @throws IOException
     * @throws DestinationsException may be thrown if the transport layer is unable to find some of the
     * destination and has no broadcast policy in this case.
     * @see org.lsst.ccs.bus.BusMembershipListener#anormalEvent(Exception) for another way to signal
     * destinations exceptions
     */
    void sendCommand(CommandRequest cmd, CommandOriginator originator) throws IOException {
        String destination = BusMessagingLayer.parseDestination(cmd.getDestination());
        if (destination == null || destination.isEmpty()) {
            throw new RuntimeException("Invalid destination for command " + cmd);
        }
        if (!getAgentPresenceManager().agentExists(destination)) {
            DestinationsException exc = new DestinationsException(subsystemName, destination);
            log.fine("sending fail (closed)" + subsystemName + " to destination " + destination);
            throw exc;
        }
        commandOriginatorForwarder.addCommandOriginator(cmd.getCorrelationId(), originator);

        if (cmd.getOriginAgentInfo().getName() == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        busMessagingLayer.sendMessage(subsystemName, Bus.COMMAND, cmd);
    }

    /**
     * broadcasts a status message.
     * If origin is negate set then sets it with the current subsystem name.
     *
     * @param status
     * @throws IOException
     */
    void sendStatus(StatusMessage status) throws IOException {
        String origin = status.getOriginAgentInfo().getName();
        if (origin == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        busMessagingLayer.sendMessage(subsystemName, Bus.STATUS, status);
    }

    /**
     * broadcasts a log message.
     * If origin is negate set then sets it with the current subsystem name.
     *
     * @param evt
     * @throws IOException
     */
    void sendLog(LogMessage evt) throws IOException {
        String origin = evt.getOriginAgentInfo().getName();
        if (origin == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        busMessagingLayer.sendMessage(subsystemName, Bus.LOG, evt);

    }

    /**
     * sends a reply or an ack responding to a command.
     * <B>Beware</B> : if <TT>originalCommand</TT> or <TT>CorrelID</TT>
     * or <TT>destination</TT> are negate set in the parameter this code will try to populate
     * these fields with informations coming from <TT>ThreadLocal</TT> data so if
     * the initial Thread that received the command spawn children (or more generally
     * if replies or ack are generated in different Threads) then this facility will be defeated.
     *
     * @param cmd
     * @throws IOException
     * @throws DestinationsException may be thrown if the transport layer is unable to find some of the
     * destination and has no broadcast policy in this case.
     * @see org.lsst.ccs.bus.BusMembershipListener#anormalEvent(Exception) for another way to signal
     * destinations exceptions
     */
    void reply(CommandReply cmd) throws IOException {
        String origin = cmd.getOriginAgentInfo().getName();
        if (origin == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        // String destination = cmd.getDestination();
        busMessagingLayer.sendMessage(subsystemName, Bus.COMMAND, cmd);
    }

    class ForwarderToCommandExecutor implements BusMessageForwarder<CommandRequest> {

        private volatile CommandExecutor commandExecutor;

        void setCommandExecutor(CommandExecutor commandExecutor) {
            if (this.commandExecutor != null) {
                throw new RuntimeException("A CommandExecutor is already registered for this Agent. There can be only one!");
            } else {
                this.commandExecutor = commandExecutor;
            }
        }

        @Override
        public void update(CommandRequest message) {
            
            //Check that this message was really intended for this CommandExecutor
            String destination = message.getDestination();
            if (destination.contains("/")) {
                destination = destination.substring(0, destination.indexOf("/"));
            }
            if (!destination.equals(getAgentInfo().getName())) {
                return;
            }

            if (commandExecutor == null) {
                throw new RuntimeException("A CommandExecutor has not been registered for this Agent.");
            }
            try {
                commandExecutor.executeCommandRequest(message);
            } catch (Throwable throwable) {
                throwable.printStackTrace();
                log.error("on command :" + throwable);
            }

        }
    }

    /**
     * Instances of this class forward commands coming from the transport layer to
     * registered {@code CommandOriginator}s. If a {@code CommandResult} is received
     * before {@code CommandAck}, its processing by the {@code CommandOriginator}
     * is postponed. Note, however, that registered command listeners will still
     * be notified of all messages in the order the messages are received.
     */
    class ForwarderToCommandOriginator implements BusMessageForwarder<CommandReply> {
      
        private class RequestData {
            RequestData(CommandOriginator originator) {
                this.originator = originator;
            }
            CommandOriginator originator;
            CommandResult result;
            boolean ackReceived;
        }

        private final ConcurrentHashMap<UUID, RequestData> currentRequests = new ConcurrentHashMap<>();

        ForwarderToCommandOriginator() {
        }

        void addCommandOriginator(UUID originatorId, CommandOriginator originator) {
            currentRequests.put(originatorId, new RequestData(originator));
        }

        @Override
        public void update(CommandReply message) {

            String destination = message.getDestination();
            if (!destination.equals(getAgentInfo().getName())) {
                return;
            }
            UUID originatorId = message.getCorrelationId();
            
            synchronized (this) {
                RequestData reqData = currentRequests.get(originatorId);
                if (reqData == null) {
                    throw new RuntimeException("Could not find the originator of the CommandRequest " + message.getEncodedData());
                }
                if (message instanceof CommandAck) {
                    reqData.originator.processAck((CommandAck) message);
                    if (reqData.result == null) {
                        reqData.ackReceived = true;
                    } else {
                        reqData.originator.processResult(reqData.result);
                        currentRequests.remove(originatorId);
                    }
                } else {
                    if (message instanceof CommandNack) {
                        reqData.originator.processNack((CommandNack) message);
                        currentRequests.remove(originatorId);
                    } else if (message instanceof CommandResult) {
                        if (reqData.ackReceived) {
                            reqData.originator.processResult((CommandResult) message);
                            currentRequests.remove(originatorId);
                        } else {
                            reqData.result = (CommandResult) message;
                        }
                    }
                }
            }

        }
    }

    /**
     * instances of this class will forward commands coming from the transport layer
     * to the registered <TT>CommandMessageListener</TT>.
     * When the incoming message is a command to be executed the <TT>TreadLocal</TT>
     * data that links commands with their corresponding relpis or ack is populated
     * (see <TT>reply</TT> method documentation).
     */
    private class ForwarderToCommandListeners implements BusMessageForwarder<CommandMessage> {

        private final CommandMessageListener listener;
        private final BusMessageFilter filter;

        ForwarderToCommandListeners(CommandMessageListener listener, BusMessageFilter filter) {
            this.listener = listener;
            this.filter = filter;
        }

        @Override
        public void update(CommandMessage message) {
            if (filter == null || filter.accept(message)) {
                listener.onCommandMessage(message);
            }
        }
    }

    /**
     * instances of this class will forward directly status messages to statuslistener
     */
    private class ForwarderToStatusListeners implements BusMessageForwarder<StatusMessage> {

        private final StatusMessageListener listener;
        private final boolean isEncodedListener;
        private final boolean isKeyValueListener;
        private final boolean isDataListener;
        private final boolean isCrystallizedListener;
        private final BusMessageFilter filter;

        ForwarderToStatusListeners(StatusMessageListener listener, BusMessageFilter filter) {
            this.listener = listener;
            this.filter = filter;
            isEncodedListener = listener instanceof EncodedStatusListens;
            isKeyValueListener = listener instanceof KeyValueStatusListener;
            isDataListener = listener instanceof DataStatusListener;
            isCrystallizedListener = listener instanceof SerializedDataStatusListener;
        }

        @Override
        public void update(StatusMessage message) {
            if (filter == null || filter.accept(message)) {
                if (isEncodedListener && message instanceof StatusData) {
                    StatusData status = (StatusData) message;
                    String source = status.getOriginAgentInfo().getName();
                    long msgTimestamp = status.getTimeStamp();
                    String key = status instanceof EncodedDataStatus ? ((EncodedDataStatus) status).getKey() : "";
                    if (isDataListener) {
                        try {
                            Object value = status.getObject();
                            if (value != null) {
                                ((DataStatusListener) listener).onDataArrival(source, msgTimestamp, key, value);
                            }
                        } catch (Exception e) {
//                            log.warn(e.getMessage(), e);
                        }
                    }
                    if (isKeyValueListener) {
                        KeyValueDataList data = status.getEncodedData();
                        if (data != null) {
                            for (KeyValueData d : data) {
                                String complexKey = d.getKey();
//                                if ( !key.isEmpty() ) {
//                                    complexKey = key+"/"+complexKey;
//                                }
//                                if ( complexKey.endsWith("/") ) {
//                                    complexKey = complexKey.substring(0,complexKey.length()-1);
//                                }
                                Object obj = d.getValue();
                                if (obj != null) {
                                    ((KeyValueStatusListener) listener).onKeyValueStatusDecomposition(source, d.getTimestamp(), complexKey, obj, 333);
                                }
                            }
                        }
                    }
                    if (isCrystallizedListener) {
                        byte[] bytes = message.getSerializedObject();
                        if (bytes != null) {
                            ((SerializedDataStatusListener) listener).onSerializedDataArrival(source, msgTimestamp, key, bytes);
                        }
                    }
                }
                listener.onStatusMessage(message);
            }
        }
    }
//        @Override
//        public void update(StatusMessage message) {
//            if (filter == null || filter.accept(message)) {
//                if (isEncodedListener && message instanceof EncodedDataStatus) {
//                    EncodedDataStatus status = (EncodedDataStatus) message;
//                    String source = status.getOriginAgentInfo().getName();
//                    for (EncodedDataStatus dataStatus : status) {
//                        long timeStamp = dataStatus.getDataTimestamp();
//                        KVList list = dataStatus.getContent();
//                        for (KeyData keyData : list) {
//                            String key = keyData.getKey();
//                    if (isDataListener) {
//                                Optional<Object> optObj = keyData.getValue();
//                                if (optObj.isPresent()) {
//                                    ((DataStatusListener) listener).onDataArrival(source, timeStamp, key, optObj.get());
//
//                            }
//
//                        }
//                    if (isKeyValueListener) {
//                                int id = keyData.hashCode();
//                                List<KeyData> detailsList = keyData.getContentAsList();
//                                for (KeyData detaileddata : detailsList) {
//                                    String complexKey = detaileddata.getKey();
//                                    Optional<Object> optional = detaileddata.getValue();
//                                    if (optional.isPresent()) {
//                                        ((KeyValueStatusListener) listener).onKeyValueStatusDecomposition(source, timeStamp, complexKey, optional.get(), id);
//                            }
//                        }
//                    }
//                    if (isCrystallizedListener) {
//                                Optional<byte[]> crysta = keyData.getCrystallizedData();
//                                if (crysta.isPresent()) {
//                                    ((SerializedDataStatusListener) listener).onSerializedDataArrival(source, timeStamp, key, crysta.get());
//                        }
//                    }
//                }
//
//                    }
//
//                }
//                listener.onStatusMessage(message);
//            }
//        }
//    }

    /**
     * instances of this class will forward directly log messages to a log listener
     */
    private class ForwarderToLogListeners implements BusMessageForwarder<LogMessage> {

        private final LogMessageListener listener;
        private final BusMessageFilter filter;

        ForwarderToLogListeners(LogMessageListener listener, BusMessageFilter filter) {
            this.listener = listener;
            this.filter = filter;
        }

        @Override
        public void update(LogMessage message) {
            if (filter == null || filter.accept(message)) {
                listener.onLogMessage(message);
            }
        }
    }

    /**
     * Set the CommandExecutor for this Agent.
     */
    void setCommandExecutor(CommandExecutor executor) {
        commandExecutorForwarder.setCommandExecutor(executor);
    }

    /**
     * registers a CommandMessageListener (in fact a ForwarderToCommandListeners to the underlying transport)
     *
     * @param l
     */
    void addCommandListener(CommandMessageListener l, BusMessageFilter filter) {
        ForwarderToCommandListeners forwarder = new ForwarderToCommandListeners(l, filter);
        messagingAccessLayer.getBusAccess(Bus.COMMAND).addForwarder(l, forwarder);
    }

    /**
     * removes a CommandMessageListener: since this command is based on strict identity the listener should be exactly the same
     * as the one registered.
     *
     * @param l
     */
    void removeCommandListener(CommandMessageListener l) {
        messagingAccessLayer.getBusAccess(Bus.COMMAND).removeForwarder(l);
    }

    /**
     * registers a StatusMessageListener (in fact a ForwarderToStatusListeners to the underlying transport)
     *
     * @param l
     */
    void addStatusListener(StatusMessageListener l, BusMessageFilter filter) {
        ForwarderToStatusListeners forwarder = new ForwarderToStatusListeners(l, filter);
        messagingAccessLayer.getBusAccess(Bus.STATUS).addForwarder(l, forwarder);
    }

    void removeStatusListener(StatusMessageListener l) {
        messagingAccessLayer.getBusAccess(Bus.STATUS).removeForwarder(l);
    }

    /**
     * registers a LogMessageListener (in fact a ForwarderToLogListeners to the underlying transport)
     *
     * @param l
     */
    void addLogListener(LogMessageListener l, BusMessageFilter filter) {
        ForwarderToLogListeners forwarder = new ForwarderToLogListeners(l, filter);
        messagingAccessLayer.getBusAccess(Bus.LOG).addForwarder(l, forwarder);
    }

    void removeLogListener(LogMessageListener l) {
        messagingAccessLayer.getBusAccess(Bus.LOG).removeForwarder(l);
    }

    /**
     * closes the underlying transport layer, stops the listening threads,
     * after this call all other sending calls will fail.
     */
    void close() {
        busMessagingLayer.disconnect(messagingAccessLayer);
    }

    public AgentPresenceManager getAgentPresenceManager() {
        return agentPresenceManager;
    }
}
