package org.lsst.ccs.messaging;

import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.messages.StatusHeartBeat;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Tracks agent connection and disconnection on the buses.
 *
 * @author emarin
 */
public class AgentPresenceManager implements StatusMessageListener {

    private final Object agentsLock = new Object();
    final CopyOnWriteArrayList<AgentPresenceListener> listAPL = new CopyOnWriteArrayList<>();
    private final Map<AgentInfo, AgentPresenceManager.TimeoutTask> mapAgents = new HashMap<>();
    private final Map<String, AgentInfo.AgentType> mapAgentsPDI = new HashMap<>();
    private static final int SUSPICION_LENGTH = 3;
    private final Timer timer;
    private final boolean providesDisconnectionInformation;
    protected static Logger log = Logger.getLogger("org.lsst.ccs.messaging.agentpresencemanager");

    class TimeoutTask extends TimerTask {
        
        private final AgentInfo agent;
        private final int broadcastPeriod;
        
        TimeoutTask(AgentInfo agent, int broadcastPeriod) {
            this.agent = agent;
            this.broadcastPeriod = broadcastPeriod;
        }

        int getBroadcastPeriod() {
            return broadcastPeriod;
        }

        @Override
        public void run() {
            removeAgent(agent);
        }
    }

    public AgentPresenceManager(boolean providesDisconnectionInformation) {
        this.providesDisconnectionInformation = providesDisconnectionInformation;
        timer = providesDisconnectionInformation ? null : new Timer(true);
    }

    @Override
    public void onStatusMessage(StatusMessage s) {

        AgentInfo a = (s.getOriginAgentInfo());
        if (a == null) {
            return;
        }

        if (providesDisconnectionInformation) {
            addAgent(a);
        } else {
            int broadCastPeriod = -1;
            if (s instanceof StatusHeartBeat) {
                StatusHeartBeat hb = (StatusHeartBeat) s;
                broadCastPeriod = hb.getStatusBroadcastPeriod();
            }
            if (s.getState().isInState(PhaseState.OFF_LINE)) {
                log.debug("remove agent on status end");
                removeAgent(a);
            } else {
                updateAgent(a, broadCastPeriod);
            }
        }
    }

    /** Called on non-OFF_LINE agent status message if {@code providesDisconnectionInformation} is {@code false}. */
    private void updateAgent(AgentInfo a, int broadcastPeriod) {
        synchronized (agentsLock) {
            AgentPresenceManager.TimeoutTask task = mapAgents.get(a);
            if (task != null) { // Agent already known to the map : its timeouttask is updated
                task.cancel();
                if (broadcastPeriod == -1) {
                    broadcastPeriod = task.getBroadcastPeriod();
                }
                task = new AgentPresenceManager.TimeoutTask(a, broadcastPeriod);
                addAgent(a, task, false);
            } else { // New agent
                if (broadcastPeriod == -1) {
                    broadcastPeriod = 10;
                }
                task = new AgentPresenceManager.TimeoutTask(a, broadcastPeriod);
                addAgent(a, task, true);
            }
            log.debug("resetting timer for agent " + a.getName() + " to " + broadcastPeriod);
            timer.schedule(task, SUSPICION_LENGTH * 1000 * broadcastPeriod);
        }
    }

    /** Called to remove agent if {@code providesDisconnectionInformation} is {@code false}. */
    private void removeAgent(AgentInfo agent) {
        synchronized (agentsLock) {
            log.debug("removing agent " + agent.getName());
            AgentPresenceManager.TimeoutTask t = mapAgents.remove(agent);
            if (t != null) {
                t.cancel();
                disconnecting(agent);
            } else {
                log.debug("removing agent with null timer");
            }
        }
    }

    /** Called to remove agent if {@code providesDisconnectionInformation} is {@code true}. */
    private void removeAgent(String agentName) {
        log.debug("removing agent " + agentName);
        synchronized (agentsLock) {
            AgentInfo.AgentType agentType = mapAgentsPDI.remove(agentName);
            if (agentType != null) {
                disconnecting(new AgentInfo(agentName, agentType));
            }
        }
    }

    /** Called to add agent if {@code providesDisconnectionInformation} is {@code false}. */
    private void addAgent(AgentInfo agent, AgentPresenceManager.TimeoutTask task, boolean isNewAgent) {
        synchronized (agentsLock) {
            mapAgents.put(agent, task);
            if (isNewAgent) {
                connecting(agent);
            }
        }
    }

    /** Called to add agent if {@code providesDisconnectionInformation} is {@code true}. */
    private void addAgent(AgentInfo a) {
        synchronized (agentsLock) {
            if (mapAgentsPDI.get(a.getName()) == null) {
                log.fine("adding " + a + " to the apm agent list");
                //TODO: change this code to a.getType() when we deprecate AgentInfo in org.lsst.ccs.bus.messages
                mapAgentsPDI.put(a.getName(), AgentInfo.AgentType.valueOf(a.getType().name()));
                connecting(a);
            }
        }
    }

    /**
     * Use listConnectedAgents instead
     *
     * @return The list with the currently connected Agents
     * @deprecated
     */
    @Deprecated
    public List<org.lsst.ccs.bus.messages.AgentInfo> listAgents() {
        List<AgentInfo> list = listConnectedAgents();
        List<org.lsst.ccs.bus.messages.AgentInfo> res = new ArrayList<>();
        for (AgentInfo a : list) {
            res.add(a);
        }
        return res;
    }

    public List<AgentInfo> listConnectedAgents() {
        synchronized (agentsLock) {
            if (providesDisconnectionInformation) {
                List<AgentInfo> list = new ArrayList<>();
                Set<Map.Entry<String, AgentInfo.AgentType>> set = mapAgentsPDI.entrySet();
                for (Map.Entry<String, AgentInfo.AgentType> e : set) {
                    list.add(new AgentInfo(e.getKey(), e.getValue()));
                }
                return list;
            } else {
                return new ArrayList<>(mapAgents.keySet());
            }
        }
    }

    /**
     * Use listConnectedAgents() instead
     *
     * @return The list with the currently connected Agents
     * @deprecated
     */
    @Deprecated
    public List<org.lsst.ccs.bus.messages.AgentInfo> listAgents(int timeout) {
        List<AgentInfo> list = listConnectedAgents(timeout);
        List<org.lsst.ccs.bus.messages.AgentInfo> res = new ArrayList<>();
        for (AgentInfo a : list) {
            res.add(a);
        }
        return res;
    }

    /**
     * Use listConnectedAgents() instead
     *
     * @return The list with the currently connected Agents
     * @deprecated
     */
    @Deprecated
    public List<AgentInfo> listConnectedAgents(int timeout) {
        return listConnectedAgents();
    }

    /**
     * Warns the list of listeners that an agent has connected
     *
     * @param agent
     */
    private void connecting(AgentInfo agent) {
        for (AgentPresenceListener l : listAPL) {
            l.connecting(agent);
        }
    }

    /**
     * Warns the list of listeners that an agent has disconnected
     *
     * @param agent
     */
    private void disconnecting(AgentInfo agent) {
        for (AgentPresenceListener l : listAPL) {
            l.disconnecting(agent);
        }
    }

    /**
     * This method is called by transport layers that provide disconnection
     * suspicion information
     *
     * @param agentName
     */
    void disconnecting(String agentName) {
        if (providesDisconnectionInformation) {
            removeAgent(agentName);
        }
    }

    public void anormalEvent(Exception exc) {
//        if(exc instanceof DuplicateAgentNameException) {
//            throw new IllegalArgumentException(exc);
//        }
    }

    public void addAgentPresenceListener(AgentPresenceListener l) {
        synchronized (agentsLock) {
            for (AgentInfo a : listConnectedAgents()) {
                l.connecting(a);
            }
            listAPL.add(l);
        }
    }

    public void removeAgentPresenceListener(AgentPresenceListener l) {
        listAPL.remove(l);
    }

    public boolean agentExists(String agentName) {
        for (AgentInfo a : listConnectedAgents()) {
            if (a.getName().equals(agentName)) {
                return true;
            }
        }
        return false;
    }

}
