package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.concurrent.ConcurrentHashMap;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * Entry point of a CCS agent to the buses.
 * Its name is used as a unique identifier on the buses.
 * @author emarin
 */
public class MessagingAccessLayer {
    
    private final String name;
    private  HashMap<Bus, BusAccess> busAccesses = new HashMap<Bus,BusAccess>();
    
    public MessagingAccessLayer(String name, Bus... buses){
        this.name = name;
    }
    /**
     * This name must be unique as it is used by the messaging layer to send or
     * dispatch messages.
     * This name is determined by the associated agent's name.
     * Its uniqueness is enforced when connecting with the messaging layer.
     * @return the name of this MessagingAccessLayer 
     */
    public String getName(){
        return name;
    }
    
    /**
     * This list of buses might be used by the messaging layer to determine on which buses
     * the associated agent has to be connected to.
     * @return The list of Buses the layer has to be registered on.
    */
    public Collection<BusAccess> getBusAccesses(){
        return busAccesses.values();
    }
    
    public BusAccess getBusAccess(Bus bus){
        return busAccesses.get(bus);
    }
    
    public void addBusAccess(BusAccess busAccess){
        busAccesses.put(busAccess.getBus(), busAccess);
    }
    
    /**
     * This method is called by the underlying bus implementation when a message is received
     * The message is then sent to the list of registered listeners
     * @param <T> the type of the received message
     * @param bus the bus on which the message has been received
     * @param message The message received
     */
    public static class BusAccess<T extends BusMessage> {

        private final ConcurrentHashMap<MessageListener, BusMessageForwarder> forwarderMap = new ConcurrentHashMap<MessageListener, BusMessageForwarder>();
        private Bus<T> bus;
        
        public BusAccess(Bus bus) {
            this.bus = bus;
        }
        
        public Bus<T> getBus(){
            return bus;
        }
        
        public void processBusMessage(T message){
            // Does nothing by default
        }
        
        public void addForwarder(MessageListener l, BusMessageForwarder forwarder){
            forwarderMap.put(l, forwarder);
        }
        
        public void removeForwarder(MessageListener l){
            forwarderMap.remove(l);
        }
        
        public ArrayList<BusMessageForwarder> getForwarderList(){
            return new ArrayList(forwarderMap.values());
        }

    }
    
    public static abstract class StatusBusAccess extends BusAccess<StatusMessage>{

        public StatusBusAccess(Bus<StatusMessage> bus) {
            super(bus);
        }
        
        public abstract void processDisconnectionSuspicion(String address);
        
        public abstract void processAnormalEvent(Exception ex);
        
    }
    
}
