package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CopyOnWriteArrayList;
import org.lsst.ccs.bus.messages.AgentInfo;
import org.lsst.ccs.bus.messages.DistributionInfo;
import org.lsst.ccs.bus.messages.HeartBeatStatus;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.CCSVersions;
import org.lsst.ccs.bus.states.PhaseState;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * The AgentPresenceManager tracks agent connection and disconnection on the buses
 *
 * @author emarin
 */
public class AgentPresenceManager implements StatusMessageListener {

    final CopyOnWriteArrayList<AgentPresenceListener> listAPL = new CopyOnWriteArrayList<>();
    private final Map<AgentInfo, AgentPresenceManager.TimeoutTask> mapAgents = new HashMap<>();
    private final Map<String, DistributionInfo> distInfoMap = new HashMap<>();
    private final Map<String, AgentInfo.AgentType> mapAgentsPDI = new HashMap<>();
    private static final int SUSPICION_LENGTH = 3;
    private final Timer timer;
    private final boolean providesDisconnectionInformation;
    protected static Logger log = Logger.getLogger("org.lsst.ccs.messaging.agentpresencemanager");

    class TimeoutTask extends TimerTask {

        private final AgentInfo agent;
        private final int broadcastPeriod;

        TimeoutTask(AgentInfo agent, int broadcastPeriod) {
            this.agent = agent;
            this.broadcastPeriod = broadcastPeriod;
        }

        int getBroadcastPeriod() {
            return broadcastPeriod;
        }

        @Override
        public void run() {
            removeAgent(agent);
        }
    }

    public AgentPresenceManager(boolean providesDisconnectionInformation) {
        this.providesDisconnectionInformation = providesDisconnectionInformation;
        if (!this.providesDisconnectionInformation) {
            timer = new Timer(true); // Timer defined as a daemon
        } else {
            timer = null;
        }
    }

    @Override
    public void onStatusMessage(StatusMessage s) {
        AgentInfo a = (s.getOriginAgentInfo());
        if (distInfoMap.get(a.getName()) == null && s instanceof HeartBeatStatus) {
            distInfoMap.put(a.getName(), ((HeartBeatStatus) s).getDistributionInfo());
        }
        if (providesDisconnectionInformation) {
            addAgent(a);
        } else {
            if (s.getState().isInState(PhaseState.OFF_LINE)) {
                log.debug("remove agent on status end");
                removeAgent(a);
            } else {

                if (a != null) {
                    int broadCastPeriod = (s instanceof HeartBeatStatus) ? ((HeartBeatStatus) s).getStatusBroadcastPeriod() : -1;
                    updateAgent(a, broadCastPeriod);
                }
            }
        }
    }

    private synchronized void updateAgent(AgentInfo a, int tempBroadcastPeriod) {
        int statusBroadcastPeriod;
        AgentPresenceManager.TimeoutTask task = mapAgents.get(a);
        // Is this agent already known to the map ?
        if (task != null) { // Agent already known to the map : its timeouttask is updated
            task.cancel();
            statusBroadcastPeriod = (tempBroadcastPeriod == -1) ? task.getBroadcastPeriod() : tempBroadcastPeriod;
            task = new AgentPresenceManager.TimeoutTask(a, statusBroadcastPeriod);
            addAgent(a, task, false);
        } else { // New agent
            statusBroadcastPeriod = (tempBroadcastPeriod == -1) ? 10 : tempBroadcastPeriod;
            task = new AgentPresenceManager.TimeoutTask(a, statusBroadcastPeriod);
            addAgent(a, task, true);
        }
        log.debug("resetting timer for agent " + a.getName() + " to " + statusBroadcastPeriod);
        timer.schedule(task, SUSPICION_LENGTH * 1000 * statusBroadcastPeriod);
    }

    /**
     * !providesDisconnectionInformation
     *
     * @param agent
     */
    private synchronized void removeAgent(AgentInfo agent){
        log.debug("removing agent " + agent.getName());
        AgentPresenceManager.TimeoutTask t = mapAgents.remove(agent);
        distInfoMap.remove(agent.getName());
        if (t != null) {
            t.cancel();
            disconnecting(agent);
        } else {
            log.debug("removing agent with null timer");
        }
    }

    /**
     * providesDisconnectionInformation
     *
     * @param agentName
     */
    private synchronized void removeAgent(String agentName) {
        log.debug("removing agent " + agentName);
        AgentInfo.AgentType agentType = mapAgentsPDI.remove(agentName);
        distInfoMap.remove(agentName);
        if (agentType != null) {
            disconnecting(new AgentInfo(agentName, agentType));
        }
    }

    /**
     * !providesDisconnectionInformation
     *
     * @param agent
     * @param task
     * @param isNewAgent
     */
    private synchronized void addAgent(AgentInfo agent, AgentPresenceManager.TimeoutTask task, boolean isNewAgent) {
        mapAgents.put(agent, task);
        if (isNewAgent) {
            connecting(agent);
        }
    }

    /**
     * providesDisconnectionInformation
     *
     * @retun
     */
    private synchronized void addAgent(AgentInfo a) {
        if (a != null && mapAgentsPDI.get(a.getName()) == null) {
            log.fine("adding " + a + " to the apm agent list");
            mapAgentsPDI.put(a.getName(), a.getType());
            connecting(a);
        }
    }

    public List<AgentInfo> listAgents() {
        return listAgents(0);
    }

    public List<AgentInfo> listAgents(int timeout) {
        if (providesDisconnectionInformation) {
            List<AgentInfo> list = new ArrayList<AgentInfo>();
            Set<Map.Entry<String, AgentInfo.AgentType>> set = mapAgentsPDI.entrySet();
            for (Map.Entry<String, AgentInfo.AgentType> e : set) {
                list.add(new AgentInfo(e.getKey(), e.getValue()));
            }
            return list;
        } else {
            return new ArrayList<>(mapAgents.keySet());
        }
    }

    /**
     * Warns the list of listeners that an agent has connected
     * @param agent 
     */
    private void connecting(AgentInfo agent) {
        for (AgentPresenceListener l : listAPL) {
            l.connecting(agent);
        }
    }

    /**
     * Warns the list of listeners that an agent has disconnected
     * @param agent 
     */
    private void disconnecting(AgentInfo agent) {
        for (AgentPresenceListener l : listAPL) {
            l.disconnecting(agent);
        }
    }

    /**
     * This method is called by transport layers that provide disconnection
     * suspicion information
     * @param agentName 
     */
    void disconnecting(String agentName) {
        if (providesDisconnectionInformation) {
            removeAgent(agentName);
        }
    }

    public void anormalEvent(Exception exc) {
//        if(exc instanceof DuplicateAgentNameException) {
//            throw new IllegalArgumentException(exc);
//        }
    }
    
    public void addAgentPresenceListener(AgentPresenceListener l){
//        for (AgentInfo a : listAgents()){
//            l.connecting(a);
//        }

        listAPL.add(l);
    }

    public void removeAgentPresenceListener(AgentPresenceListener l) {
        listAPL.remove(l);
    }

    public boolean agentExists(String agentName) {
        for (AgentInfo a : listAgents()) {
            if (a.getName().equals(agentName)) {
                return true;
            }
        }
        return false;
    }

    public DistributionInfo getAgentDistributionInfo(String agentName) {
        return distInfoMap.get(agentName);
    }
    
    /**
     * Get the CCSVersions for the current CCS Environment.
     * 
     */
    public CCSVersions getCCSVersions() {
        CCSVersions ccsVersions = new CCSVersions();
        List<AgentInfo> agents = listAgents();
        for ( AgentInfo a : agents ) {
            DistributionInfo d = getAgentDistributionInfo(a.getName());
            if (  d != null ) {
                ccsVersions.add(a, d);
            }
        }
        return ccsVersions;
    }
    

}
