package org.lsst.ccs.messaging;

import org.lsst.ccs.utilities.logging.Logger;

import java.io.IOException;
import org.lsst.ccs.bus.messages.AgentInfo;
import org.lsst.ccs.bus.messages.CommandReply;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * This interface provides messaging methods for a component (eg an agent) to be
 * able to communicate on the buses.
 * Provided methods allow to add/remove message listeners on each of the buses,
 * to send message on each of the buses, to specify a unique CommandExecutor 
 * associated to the agent and a list of CommandOriginator.
 * 
 * @author LSST CCS Team
 */
public class AgentMessagingLayer {

    private final BusMessageFilter filterMessagesFromThisAgent;

    private final Logger curLogger = Logger.getLogger("org.lsst.ccs.bus") ;
    
    private final BusApplicationLayer layer ;

    private final AgentInfo agentInfo;
    

    
    protected AgentMessagingLayer(AgentInfo agentInfo) {
        if ( agentInfo == null ) {
            throw new RuntimeException("MessagingAccess cannot have null Agent");
        }
        this.agentInfo = agentInfo;
        this.layer = new BusApplicationLayer(agentInfo);
        MessagingAccessManager.addMessagingAccess(this);
        filterMessagesFromThisAgent = BusMessageFilter.messageOrigin(getAgentName()).negate();
    }
    
    public AgentInfo getAgentInfo() {
        return agentInfo;
    }

    public String getAgentName() {
        return agentInfo.getName();
    }
    
    protected BusApplicationLayer getApplicationLayer() {
        return layer;
    }

    public void shutdownBusAccess() {
        layer.close() ;
        MessagingAccessManager.removeMessagingAccess(this);
    }
    
    public synchronized AgentPresenceManager getAgentPresenceManager(){
        return layer.getAgentPresenceManager();
    }

    /**
     * Adds a listener on the Log bus.
     * By default the listener is negate passed messages published from this
 AgentMessagingLayer.
     * @param listener the listener to be added on the Log bus
     */
    public synchronized void addLogMessageListener(LogMessageListener listener) {
        addLogMessageListener(listener, filterMessagesFromThisAgent);
    }

    /**
     * Adds a listener on the Log bus with a filter.
     * The listener is passed bus messages that pass the filter.
     * If the filter is null, all messages are passed to the filter.
     * @param listener the listener to be added on the Log bus
     * @param filter The BusMessageFilter to be applied to the incoming Bus Messages
     */
    public synchronized void addLogMessageListener(LogMessageListener listener, BusMessageFilter filter) {
        layer.addLogListener(listener,filter);
    }

    /**
     * Adds a listener on the Status bus.
     * By default the listener is negate passed messages published from this
 AgentMessagingLayer.
     * @param listener the listener to be added on the Status bus
     */
    public synchronized void addStatusMessageListener(StatusMessageListener listener) {
        addStatusMessageListener(listener, filterMessagesFromThisAgent);
    }

    /**
     * Adds a listener on the Status bus with a filter.
     * The listener is passed bus messages that pass the filter.
     * If the filter is null, all messages are passed to the filter.
     * @param listener the listener to be added on the Status bus
     * @param filter The BusMessageFilter to be applied to the incoming Bus Messages
     */
    public synchronized void addStatusMessageListener(StatusMessageListener listener, BusMessageFilter filter) {
        layer.addStatusListener(listener,filter);
    }

    /**
     * Adds a listener on the Command bus.
     * By default the listener is negate passed messages published from this
 AgentMessagingLayer.
     * @param listener the listener to be added on the Command bus
     */
    public synchronized void addCommandMessageListener(CommandMessageListener listener) {
        addCommandMessageListener(listener, filterMessagesFromThisAgent);
    }

    /**
     * Adds a listener on the Command bus with a filter.
     * The listener is passed bus messages that pass the filter.
     * If the filter is null, all messages are passed to the filter.
     * @param listener the listener to be added on the Command bus
     * @param filter The BusMessageFilter to be applied to the incoming Bus Messages
     */
    public synchronized void addCommandMessageListener(CommandMessageListener listener, BusMessageFilter filter) {
        layer.addCommandListener(listener, filter);
    }

    /**
     * Removes a listener on the Log bus
     * @param listener the listener to be removed on the Log bus
     */
    public synchronized void removeLogMessageListener(LogMessageListener listener) {
        layer.removeLogListener(listener);
    }

    /**
     * Removes a listener on the Status bus
     * @param listener the listener to be removed on the Status bus
     */
    public synchronized void removeStatusMessageListener(StatusMessageListener listener) {
        layer.removeStatusListener(listener);
    }

    /**
     * Removes a listener on the Command bus
     * @param listener the listener to be removed on the Command bus
     */
    public synchronized void removeCommandMessageListener(CommandMessageListener listener) {
        layer.removeCommandListener(listener);
    }

    /**
     * Sends a Log Message on the Log Bus
     * @param msg The message to be sent on the Log bus
     */
    public synchronized void sendLogMessage(LogMessage msg) {
        msg.setOriginAgentInfo(getAgentInfo());
        try {
            layer.sendLogEvent(msg);
        } catch (IOException e) {
            //TODO:change that!!!
            throw new RuntimeException(e);
        }
    }

    /**
     * Sends a Status Message on the Status Bus
     * @param msg The message to be sent on the Status bus
     */
    public synchronized void sendStatusMessage(StatusMessage msg) {
        msg.setOriginAgentInfo(getAgentInfo());
        curLogger.debug("sending status " + msg);
        try {
            layer.sendStatus(msg);
        } catch (IOException e) {
            //TODO:change that!!!
            throw new RuntimeException(e);
        }
    }

    /**
     * Sends a Command Request on the Command Bus.
     * This message will be received by the CommandExecutor and the list of CommandMessageListener
     * @param cmd The CommandRequest to be sent on the Command bus
     * @param originator The component that has requested the execution of the command
     */
    public synchronized void sendCommandRequest(CommandRequest cmd, CommandOriginator originator) {
        cmd.setOriginAgentInfo(getAgentInfo());
        try {
            layer.sendCommand(cmd, originator);
        } catch (IOException e) {
            //TODO:change that!!!
            if(e instanceof DestinationsException) {
                //System.out.println(" Log WARN: " + e);
                curLogger.warn("destination problem", e) ;
                // toDO : wit until lock arbitrator
                // The exception is thrown again to address https://jira.slac.stanford.edu/browse/LSSTCCS-181
                // In the future we might want to have smarter logic to address any
                // LockArbitrator issues.
                throw new RuntimeException(e);
            } else {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Sends a Command Reply on the Command Bus.
     * This message will be received by the CommandOriginator and the list of CommandMessageListener
     * The reply can be a Nack, Ack, Error or Result
     * @param reply The CommandReply to be sent on the Command bus
     */
    public synchronized void sendCommandReply(CommandReply reply) {
        reply.setOriginAgentInfo(getAgentInfo());
        try {
            layer.reply(reply);
        } catch (IOException e) {
            //TODO:change that!!!
            throw new RuntimeException(e);
        }
    }

    /**
     * Defines the component able to execute an incoming command
     * @param executor 
     */
    protected synchronized void setCommandExecutor(CommandExecutor executor) {
        layer.setCommandExecutor(executor);
    }

}
