package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.bus.messages.StatusMessage;

/**
 * Entry point of a CCS agent to the buses.
 * Its name is used as a unique identifier on the buses.
 * @author emarin
 */
public class MessagingAccessLayer {
    
    private final String name;
    private final List<BusAccess> busAccesses = new ArrayList<BusAccess>();
    
    public MessagingAccessLayer(String name, Bus... buses){
        this.name = name;
        if (buses == null){
            buses = Bus.values();
        }
    }
    /**
     * This name must be unique as it is used by the messaging layer to send or
     * dispatch messages.
     * This name is determined by the associated agent's name.
     * Its uniqueness is enforced when connecting with the messaging layer.
     * @return the name of this MessagingAccessLayer 
     */
    public String getName(){
        return name;
    }
    
    /**
     * This list of buses might be used by the messaging layer to determine on which buses
     * the associated agent has to be connected to.
     * @return The list of Buses the layer has to be registered on.
    */
    public List<BusAccess> getBusAccesses(){
        return busAccesses;
    }
    
    public void addBusAccess(BusAccess busAccess){
        busAccesses.add(busAccess);
    }
    
    /**
     * This method is called by the underlying bus implementation when a message is received
     * The message is then sent to the list of registered listeners
     * @param <T> the type of the received message
     * @param bus the bus on which the message has been received
     * @param message The message received
     */
    public static abstract class BusAccess<T extends BusMessage> {

        private Bus<T> bus;
        public BusAccess(Bus bus) {
            this.bus = bus;
        }
        
        public Bus<T> getBus(){
            return bus;
        }
        
        public abstract void processBusMessage(T message);
    }
    
    public static abstract class StatusBusAccess extends BusAccess<StatusMessage>{

        public StatusBusAccess(Bus<StatusMessage> bus) {
            super(bus);
        }
        
        public abstract void processDisconnectionSuspicion(String address, String info);
        
        public abstract void processAnormalEvent(Exception ex);
    }
}
