package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.bus.messages.BusMessage;

/**
 * A BusMessageFilter provides methods for creating and combining filters for BusMessages.
 * A BusMessageFilter can be used when registering a Bus Listener.
 * An example of creating a BusMessageFilter selecting for BusMessages from an origin and
 * of a given type is:
 * 
 *  BusMessageFilter filter = BusMessageFilter.messageOrigin("abc").and(BusMessageFilter.messageClass(SomeMessage.class));
 *
 */
public abstract class BusMessageFilter {

    private final List<BusMessageFilter> ors = new ArrayList<>();
    private final List<BusMessageFilter> ands = new ArrayList<>();
    private boolean is = true;

    /**
     * Create BusMessageFilter on the origin of a BusMessage. The filter will
     * select BusMessages from the provided origin.
     * @param origin The origin of the desired BusMessages
     * @return A BusMessageFilter on the origin of a BusMessage.
     * 
     */
    public static BusMessageFilter messageOrigin(String origin) {
        return new OriginBusMessageFilter(origin);
    }

    /**
     * Create BusMessageFilter on the class of a BusMessage. The filter will
     * select BusMessages that can be assigned from the provided clazz.
     * @param clazz The clazz of the desired BusMessages
     * @return A BusMessageFilter on the origin of a BusMessage.
     * 
     */
    public static BusMessageFilter messageClass(Class clazz) {
        return new ClassBusMessageFilter(clazz);
    }

    /**
     * Create BusMessageFilter on the summary of a BusMessage. The filter will
     * select BusMessages that contain the provided String as part of their summary.
     * @param summary The String used to match the summary of a BusMessages
     * @return A BusMessageFilter on the summary of a BusMessage.
     * 
     */
    public static BusMessageFilter messageSummary(String summary) {
        return new SummaryBusMessageFilter(summary);
    }

    /**
     * Negate the selection logic of a BusMessageFilter.
     * @return A clone of the original BusMessageFilter with the negated logic.
     * 
     */
    public BusMessageFilter not() {
        try {
            BusMessageFilter r = (BusMessageFilter) this.clone();
            r.is = false;
            return r;
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * OR two BusMessageFilters.
     * @param filter The filter to OR with.
     * @return A clone of the original BusMessageFilter ORed to the provided one.
     * 
     */
    public BusMessageFilter or(BusMessageFilter filter) {
        try {
            BusMessageFilter r = (BusMessageFilter) this.clone();
            r.addOr(filter);
            return r;
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * AND two BusMessageFilters.
     * @param filter The filter to AND with.
     * @return A clone of the original BusMessageFilter ANDed to the provided one.
     * 
     */
    public BusMessageFilter and(BusMessageFilter filter) {
        try {
            BusMessageFilter r = (BusMessageFilter) this.clone();
            r.addAnd(filter);
            return r;
        } catch (CloneNotSupportedException e) {
            throw new RuntimeException(e);
        }
    }

    protected void addOr(BusMessageFilter filter) {
        ors.add(filter);
    }

    protected void addAnd(BusMessageFilter filter) {
        ands.add(filter);
    }

    /**
     * Check if a BusMessage passes this filter.
     * @param busMessage The BusMessage to test.
     * @return True if the BusMessage passes the BusMessageFilter.
     * 
     */
    public boolean accept(BusMessage busMessage) {
        boolean result = this.internalAccept(busMessage);
        if (!is) {
            result = !result;
        }
        for (BusMessageFilter or : ors) {
            result = result || or.accept(busMessage);
        }
        for (BusMessageFilter and : ands) {
            result = result && and.accept(busMessage);
        }
        return result;
    }

    protected abstract boolean internalAccept(BusMessage busMessage);

    protected abstract String internalToString();
    
    protected abstract BusMessageFilter internalClone();

    protected void updateClone(BusMessageFilter f) {
        for (BusMessageFilter or : ors) {
            f.addOr(or);
        }
        for (BusMessageFilter and : ands) {
            f.addAnd(and);
        }
        f.is = is;
    }

    @Override
    protected Object clone() throws CloneNotSupportedException {
        BusMessageFilter f = internalClone();
        updateClone(f);
        return f;
    }

    
    
    private static class SummaryBusMessageFilter extends BusMessageFilter {

        private final String summary;

        SummaryBusMessageFilter(String summary) {
            this.summary = summary;
        }

        @Override
        public boolean internalAccept(BusMessage busMessage) {
            if (summary == null || "".equals(summary)) {
                return true;
            }
            return busMessage.getSummary().contains(summary);
        }

        @Override
        protected String internalToString() {
            return " summary.contains(\"" + summary + "\") ";
        }

        @Override
        protected BusMessageFilter internalClone() {
            return new SummaryBusMessageFilter(summary);
        }        
    }

    private static class ClassBusMessageFilter extends BusMessageFilter {

        private final Class clazz;

        ClassBusMessageFilter(Class clazz) {
            this.clazz = clazz;
        }

        ClassBusMessageFilter(String clazz) {
            try {
                this.clazz = Class.forName(clazz);
            } catch (ClassNotFoundException e) {
                throw new RuntimeException("Could not properly initialize BusMessage class filter", e);
            }
        }

        @Override
        public boolean internalAccept(BusMessage busMessage) {
            return clazz.isInstance(busMessage);
        }

        @Override
        protected String internalToString() {
            return " class instanceof " + clazz.getCanonicalName() + " ";
        }
        
        @Override
        protected BusMessageFilter internalClone() {
            return new ClassBusMessageFilter(clazz);
        }        
    }

    private static class OriginBusMessageFilter extends BusMessageFilter {

        private final String origin;

        OriginBusMessageFilter(String origin) {
            this.origin = origin;
        }

        @Override
        public boolean internalAccept(BusMessage busMessage) {
            if (origin == null || "".equals(origin)) {
                return true;
            }
            return busMessage.getOriginAgentInfo().getName().equals(origin);
        }

        @Override
        protected String internalToString() {
            return " origin = " + origin + " ";
        }

        @Override
        protected BusMessageFilter internalClone() {
            return new OriginBusMessageFilter(origin);
        }        
    }

    @Override
    public String toString() {
        String result = "(" + internalToString() + ")";
        if (!is) {
            result = "( !" + result+" )";
        }
        for (BusMessageFilter or : ors) {
            result += "||" + or.toString();
        }
        for (BusMessageFilter and : ands) {
            result += "&&" + and.toString();
        }
        return result;
    }

}
