package org.lsst.ccs.messaging;

//import org.apache.log4j.Logger;
import org.lsst.ccs.bus.messages.CommandAck;
import org.lsst.ccs.bus.messages.CommandReply;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.logging.Logger;

import java.io.IOException;
import java.util.HashMap;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.Optional;
import org.lsst.ccs.bus.messages.AgentInfo;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.EncodedDataStatus;
import org.lsst.ccs.bus.messages.KVList;
import org.lsst.ccs.bus.messages.KeyData;
import org.lsst.ccs.bus.messages.CommandError;
import org.lsst.ccs.bus.messages.CommandMessage;
import org.lsst.ccs.bus.messages.CommandNack;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.CommandResult;
import org.lsst.ccs.bus.messages.LogMessage;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.messaging.MessagingAccessLayer.BusAccess;
import org.lsst.ccs.messaging.MessagingAccessLayer.StatusBusAccess;

/**
 * Calls a BusMessagingLayer and adds all that is necessary to handle messages
 * following application layer concerns.
 * <p>
 * Each is bound to a subsystem.
 * <p>
 * This class is meant to be subclassed if there is a need to setup complex
 * correlations between messages (or more generally for "application layer" concerns)
 * </P>
 * <B>beware</B> <TT>ThreadLocal</TT> objects may be changed in future releases: they may be
 * moved to <I>Context</I> objects carried through thread creations.
 */
class BusApplicationLayer {

    private final AgentInfo agentInfo;
    private final String subsystemName;
    private final BusMessagingLayer busMessagingLayer;
    private static Logger log = Logger.getLogger("org.lsst.ccs.bus.layer");
    private final MessagingAccessLayer messagingAccessLayer;
    private final AgentPresenceManager agentPresenceManager;
    
    /**
     * creates an entry point to communication for the subsystem.
     * One is not supposed to interact directly with the <TT>BusMessagingLayer</TT>
     * afterwards except for administrative purposes (example: MembershipListeners).
     *
     * @param subsystemName rules of naming apply
     * @param busMessagingLayer transport layer
     * @throws NullPointerException if busMessagingLayer is null
     */
    BusApplicationLayer(AgentInfo agentInfo) {

        String protocolProperty = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.transport", "jgroups:udp_ccs:");
        String transportPropsProperty = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.transport.properties", "");
        try {
            //busMessagingLayer = (BusMessagingLayer) Class.forName(clazzName).newInstance();
            this.busMessagingLayer = TransportManager.getConnection(protocolProperty, transportPropsProperty);
        } catch (TransportException exc) {
            //todo: change this exception
            throw new RuntimeException(exc);
        }
        this.agentInfo = agentInfo;
        this.subsystemName = agentInfo.getName();
        this.messagingAccessLayer = new MessagingAccessLayer(subsystemName);

        if (getBusMessagingLayer() instanceof ProvidesDisconnectionInformation){
            agentPresenceManager = new AgentPresenceManager(true);
            
        } else {
            agentPresenceManager = new AgentPresenceManager(false);
        }
        addStatusListener(agentPresenceManager,BusMessageFilter.messageOrigin(this.agentInfo.getName()).not());
        messagingAccessLayer.addBusAccess(new BusAccess<LogMessage>(Bus.LOG) {
            
            @Override
            public void processBusMessage(LogMessage message) {
                    for (BusMessageForwarder forwarder : logForwarderMap.values()){
                        forwarder.update(message);
                    }
            }
        });
        
        messagingAccessLayer.addBusAccess(new StatusBusAccess(Bus.STATUS) {
            
            @Override
            public void processBusMessage(StatusMessage message) {
                    for (BusMessageForwarder forwarder : statusForwarderMap.values()){
                        forwarder.update(message);
                    }
            }
           
            @Override
            public void processDisconnectionSuspicion(String address, String info){
                if (getAgentPresenceManager() != null ){
                    getAgentPresenceManager().disconnecting(address, info);
                }
            }
            
            @Override
            public void processAnormalEvent(Exception ex){
                if (agentPresenceManager != null){
                    agentPresenceManager.anormalEvent(ex);
                }
            }
        });
        
       messagingAccessLayer.addBusAccess(new BusAccess<CommandMessage>(Bus.COMMAND) {
            
            @Override
            public void processBusMessage(CommandMessage message) {
                    for (BusMessageForwarder forwarder : commandForwarderMap.values()){
                        forwarder.update(message);
                    }
                    if (message instanceof CommandRequest){
                        commandExecutorForwarder.update((CommandRequest)message);
                    }
                    else if (message instanceof CommandReply){
                        commandOriginatorForwarder.update((CommandReply)message);
                    }
                }
        });
        try {
            busMessagingLayer.connect(messagingAccessLayer);
        } catch (DuplicateAgentNameException ex) {
            throw new RuntimeException(ex);
        } catch (IOException ex){
            throw new RuntimeException(ex);
        }
    }

    final BusMessagingLayer getBusMessagingLayer() {
        return busMessagingLayer;
    }
    
    AgentInfo getAgentInfo() {
        return agentInfo;
    }

    /**
     * utility method: parse the destination string in Commands.
     * destination is a list of comma separated list of names(beware of spaces!).
     * rules: if there is a "*" in the destination list then the message is broadcast;
     * if there is a slash in a name only the first part of the name (before the slash)
     * is used for transport destination (so for instance "sft/carrousel" is sent to "sft").
     *
     * @param destination
     * @return an array of agent names or an empty array if broadcasting is requested
     */
    protected String[] parseDestination(String destination) {
        String[] dests = destination.split(",");
        for (int ix = 0; ix < dests.length; ix++) {
            String dest = dests[ix];
            if ("*".equals(dest)) {
                dests = new String[0];
                break;
            }
            if (dest.contains("/")) {
                dests[ix] = dest.substring(0, dest.indexOf("/"));
            }
        }
        return dests;
    }

    /**
     * sends a command message to all destinations.
     * If origin of message is not set then sets it using the current subsystem name.
     * If a Correlation Id is not set then creates one.
     *
     * @param cmd
     * @throws IOException
     * @throws DestinationsException may be thrown if the transport layer is unable to find some of the
     * destination and has no broadcast policy in this case.
     * @see org.lsst.ccs.bus.BusMembershipListener#anormalEvent(Exception) for another way to signal
     * destinations exceptions
     */
    void sendCommand(CommandRequest cmd, CommandOriginator originator) throws IOException {
        String originatorId = java.util.UUID.randomUUID().toString();
        cmd.setCommandOriginatorId(originatorId);
        commandOriginatorForwarder.addCommandOriginator(originatorId, originator);

        // There's only one destination since there's only one destination in CommandMessage
        String[] destinations = parseDestination(cmd.getDestination());
        if (cmd.getOriginAgentInfo().getName() == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        busMessagingLayer.sendMessage(subsystemName, Bus.COMMAND, cmd, destinations);
    }

    /**
     * broadcasts a status message.
     * If origin is not set then sets it with the current subsystem name.
     *
     * @param status
     * @throws IOException
     */
    void sendStatus(StatusMessage status) throws IOException {
        String origin = status.getOriginAgentInfo().getName();
        if (origin == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        busMessagingLayer.sendMessage(subsystemName, Bus.STATUS, status);
    }

    /**
     * broadcasts a log message.
     * If origin is not set then sets it with the current subsystem name.
     *
     * @param evt
     * @throws IOException
     */
    void sendLogEvent(LogMessage evt) throws IOException {
        String origin = evt.getOriginAgentInfo().getName();
        if (origin == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        busMessagingLayer.sendMessage(subsystemName, Bus.LOG, evt);

    }

    /**
     * sends a reply or an ack responding to a command.
     * <B>Beware</B> : if <TT>originalCommand</TT> or <TT>CorrelID</TT>
     * or <TT>destination</TT> are not set in the parameter this code will try to populate
     * these fields with informations coming from <TT>ThreadLocal</TT> data so if
     * the initial Thread that received the command spawn children (or more generally
     * if replies or ack are generated in different Threads) then this facility will be defeated.
     *
     * @param cmd
     * @throws IOException
     * @throws DestinationsException may be thrown if the transport layer is unable to find some of the
     * destination and has no broadcast policy in this case.
     * @see org.lsst.ccs.bus.BusMembershipListener#anormalEvent(Exception) for another way to signal
     * destinations exceptions
     */
    void reply(CommandReply cmd) throws IOException {
        String origin = cmd.getOriginAgentInfo().getName();
        if (origin == null) {
            throw new RuntimeException("Agent name must not be null");
        }
        String destination = cmd.getDestination();
        busMessagingLayer.sendMessage(subsystemName, Bus.COMMAND, cmd, destination);
    }

    ////////////// Listeners
    private final IdentityHashMap<CommandMessageListener, BusMessageForwarder> commandForwarderMap = new IdentityHashMap<>();
    private final IdentityHashMap<LogMessageListener, BusMessageForwarder> logForwarderMap = new IdentityHashMap<>();
    private final IdentityHashMap<StatusMessageListener, BusMessageForwarder> statusForwarderMap = new IdentityHashMap<>();
    private final ForwarderToCommandExecutor commandExecutorForwarder = new ForwarderToCommandExecutor();
    private final ForwarderToCommandOriginator commandOriginatorForwarder = new ForwarderToCommandOriginator();

    private class ForwarderToCommandExecutor implements BusMessageForwarder<CommandRequest> {

        private CommandExecutor commandExecutor = null;

        void removeCommandExecutor() {
            this.commandExecutor = null;
        }

        void setCommandExecutor(CommandExecutor commandExecutor) {
            if (this.commandExecutor != null) {
                throw new RuntimeException("A CommandExecutor is already registered for this Agent. There can be only one!");
            } else {
                this.commandExecutor = commandExecutor;
            }
        }

        @Override
        public void update(CommandRequest message) {
                //Check that this message was really intended for this CommandExecutor
                String destination = message.getDestination();
                if ( destination.contains("/") ) {
                    destination = destination.substring(0,destination.indexOf("/"));
                }
                if ( ! destination.equals(getAgentInfo().getName() ) ) {
                    return;
                }
                
                if (commandExecutor == null) {
                    throw new RuntimeException("A CommandExecutor has not been registered for this Agent.");
                }
                try {
                    commandExecutor.executeCommandRequest(message);
                } catch (Throwable throwable) {
                    throwable.printStackTrace();
                    log.error("on command :" + throwable);
                }
        }
    }

    /**
     * instances of this class will forward commands coming from the transport layer
     * to the registered <TT>CommandOriginator</TT>.
     * This is for replies after a CommandRequest has been sent.
     *
     */
    private class ForwarderToCommandOriginator implements BusMessageForwarder<CommandReply> {

        private final HashMap<String, CommandOriginator> currentOriginators = new HashMap<>();

        ForwarderToCommandOriginator() {
        }

        void addCommandOriginator(String originatorId, CommandOriginator originator) {
            currentOriginators.put(originatorId, originator);
        }

        @Override
        public void update(CommandReply message) {

            String destination = message.getDestination();
            if ( ! destination.equals(getAgentInfo().getName() ) ) {
                return;
            }
            String originatorId = message.getCommandOriginatorId();
            CommandOriginator commandOriginator = currentOriginators.get(originatorId);
            if (commandOriginator == null) {
                throw new RuntimeException("Could not find the origin of the CommandRequest " + message.getCommand());
            }

            if (message instanceof CommandAck ) {
                commandOriginator.processAck((CommandAck) message);
            } else {
                if (message instanceof CommandNack) {
                    commandOriginator.processNack((CommandNack) message);
                } else if (message instanceof CommandResult) {
                    commandOriginator.processResult((CommandResult) message);
                } else if (message instanceof CommandError) {
                    commandOriginator.processError((CommandError) message);
                }
                //Once used we remove the commandOriginator;
                currentOriginators.remove(originatorId);
            }
        }
    }

    /**
     * instances of this class will forward commands coming from the transport layer
     * to the registered <TT>CommandMessageListener</TT>.
     * When the incoming message is a command to be executed the <TT>TreadLocal</TT>
     * data that links commands with their corresponding relpis or ack is populated
     * (see <TT>reply</TT> method documentation).
     */
    private class ForwarderToCommandListeners implements BusMessageForwarder<CommandMessage> {

        private final CommandMessageListener listener;
        private final BusMessageFilter filter;

        ForwarderToCommandListeners(CommandMessageListener listener, BusMessageFilter filter) {
            this.listener = listener;
            this.filter = filter;
        }

        @Override
        public void update(CommandMessage message) {
            if (filter == null || filter.accept(message)) {
                listener.onCommandMessage(message);
            }
        }
    }

    /**
     * instances of this class will forward directly status messages to statuslistener
     */
    private class ForwarderToStatusListeners implements BusMessageForwarder<StatusMessage> {

        private final StatusMessageListener listener;
        private final boolean isEncodedListener;
        private final boolean isKeyValueListener;
        private final boolean isDataListener;
        private final boolean isCrystallizedListener;
        private final BusMessageFilter filter;

        ForwarderToStatusListeners(StatusMessageListener listener, BusMessageFilter filter) {
            this.listener = listener;
            this.filter = filter;
            isEncodedListener = listener instanceof EncodedStatusListens;
            isKeyValueListener = listener instanceof KeyValueStatusListener;
            isDataListener = listener instanceof DataStatusListener;
            isCrystallizedListener = listener instanceof SerializedDataStatusListener;
        }

        @Override
        public void update(StatusMessage message) {
            if (filter == null || filter.accept(message)) {
                if (isEncodedListener) {
                    if (message instanceof EncodedDataStatus) {
                        EncodedDataStatus status = (EncodedDataStatus) message;
                        String source = status.getOriginAgentInfo().getName();
                        for (EncodedDataStatus dataStatus : status) {
                            long timeStamp = dataStatus.getDataTimestamp();
                            KVList list = dataStatus.getContent();
                            for (KeyData keyData : list) {
                                String key = keyData.getKey();
                                if (isDataListener) {
                                    Optional<Object> optObj = keyData.getValue();
                                    if (optObj.isPresent()) {
                                        ((DataStatusListener) listener).onDataArrival(source, timeStamp, key, optObj.get());

                                    }

                                }
                                if (isKeyValueListener) {
                                    int id = keyData.hashCode();
                                    List<KeyData> detailsList = keyData.getContentAsList();
                                    for (KeyData detaileddata : detailsList) {
                                        String complexKey = detaileddata.getKey();
                                        Optional<Object> optional = detaileddata.getValue();
                                        if (optional.isPresent()) {
                                            ((KeyValueStatusListener) listener).onKeyValueStatusDecomposition(source, timeStamp, complexKey, optional.get(), id);
                                        }
                                    }
                                }
                                if (isCrystallizedListener) {
                                    Optional<byte[]> crysta = keyData.getCrystallizedData();
                                    if (crysta.isPresent()) {
                                        ((SerializedDataStatusListener) listener).onSerializedDataArrival(source, timeStamp, key, crysta.get());
                                    }
                                }
                            }

                        }
                    }

                } else {
                    listener.onStatusMessage(message);

                }
            }
        }
    }

    /**
     * instances of this class will forward directly log messages to a log listener
     */
    private class ForwarderToLogListeners implements BusMessageForwarder<LogMessage> {

        private final LogMessageListener listener;
        private final BusMessageFilter filter;

        ForwarderToLogListeners(LogMessageListener listener, BusMessageFilter filter) {
            this.listener = listener;
            this.filter = filter;
        }

        @Override
        public void update(LogMessage message) {
            if (filter == null || filter.accept(message)) {
                listener.onLogMessage(message);
            }
        }
    }

    /**
     * Set the CommandExecutor for this Agent.
     */
    void setCommandExecutor(CommandExecutor executor) {
        commandExecutorForwarder.setCommandExecutor(executor);
    }

    /**
     * Remove the CommandExecutor for this Agent.
     * This is meant to be invoked to clean things up, when the JVM is stopped,
     * So it's probably not needed.
     *
     */
    void removeCommandExecutor() {
        commandExecutorForwarder.removeCommandExecutor();
    }

    /**
     * registers a CommandMessageListener (in fact a ForwarderToCommandListeners to the underlying transport)
     *
     * @param l
     */
    void addCommandListener(CommandMessageListener l, BusMessageFilter filter) {
        ForwarderToCommandListeners forwarder = new ForwarderToCommandListeners(l, filter);
        commandForwarderMap.put(l, forwarder);
//        busMessagingLayer.addMessageListener(subsystemName, new ForwarderToCommandListeners(l, filter), Bus.COMMAND);
    }

    /**
     * removes a CommandMessageListener: since this command is based on strict identity the listener should be exactly the same
     * as the one registered.
     *
     * @param l
     */
    void removeCommandListener(CommandMessageListener l) {
        commandForwarderMap.remove(l);
//        if (forwarder != null) {
//            busMessagingLayer.removeMessageListener(subsystemName, forwarder, Bus.COMMAND);
//        } else {
//            //TODO: log
//        }
    }

    /**
     * registers a StatusMessageListener (in fact a ForwarderToStatusListeners to the underlying transport)
     *
     * @param l
     */
    void addStatusListener(StatusMessageListener l, BusMessageFilter filter) {
        ForwarderToStatusListeners forwarder = new ForwarderToStatusListeners(l, filter);
        statusForwarderMap.put(l, forwarder);
//        busMessagingLayer.addMessageListener(subsystemName, new ForwarderToStatusListeners(l, filter), Bus.STATUS);
    }

    void removeStatusListener(StatusMessageListener l) {
        statusForwarderMap.remove(l);
//        if (forwarder != null) {
//            busMessagingLayer.removeMessageListener(subsystemName, forwarder, Bus.STATUS);
//        } else {
//            //TODO: log
//        }

    }

    /**
     * registers a LogMessageListener (in fact a ForwarderToLogListeners to the underlying transport)
     *
     * @param l
     */
    void addLogListener(LogMessageListener l, BusMessageFilter filter) {
        ForwarderToLogListeners forwarder = new ForwarderToLogListeners(l, filter);
        logForwarderMap.put(l, forwarder);
//        busMessagingLayer.addMessageListener(subsystemName, new ForwarderToLogListeners(l, filter), Bus.LOG);
    }

    void removeLogListener(LogMessageListener l) {
        logForwarderMap.remove(l);
//        if (forwarder != null) {
//            busMessagingLayer.removeMessageListener(subsystemName, forwarder, Bus.LOG);
//        } else {
//            //TODO: log
//        }
    }

    /**
     * closes the underlying transport layer, stops the listening threads,
     * after this call all other sending calls will fail.
     */
    void close() {
        busMessagingLayer.disconnect(messagingAccessLayer);
    }
    
    public AgentPresenceManager getAgentPresenceManager(){
        return agentPresenceManager;
    }
}
