package org.lsst.ccs.messaging;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CopyOnWriteArrayList;
import org.lsst.ccs.bus.messages.AgentInfo;
import org.lsst.ccs.bus.messages.HeartBeatStatus;
import org.lsst.ccs.bus.messages.StatusForEnd;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * The AgentPresenceManager tracks agent connection and disconnection on the buses
 * @author emarin
 */
public class AgentPresenceManager implements StatusMessageListener {
    
    final CopyOnWriteArrayList<AgentPresenceListener> listAPL = new CopyOnWriteArrayList<>();
    private final Map<AgentInfo,AgentPresenceManager.TimeoutTask> mapAgents = new HashMap<AgentInfo,AgentPresenceManager.TimeoutTask>();
    private final Map<String,AgentInfo.AgentType> mapAgentsPDI = new HashMap<String,AgentInfo.AgentType>();
    private static final int SUSPICION_LENGTH = 3;
    private final Timer timer;
    private final boolean providesDisconnectionInformation;
    
    protected static Logger log = Logger.getLogger("org.lsst.ccs.messaging.agentpresencemanager");
    
    class TimeoutTask extends TimerTask{
        private final AgentInfo agent;
        private final int broadcastPeriod;
        
        TimeoutTask(AgentInfo agent, int broadcastPeriod){
            this.agent = agent;
            this.broadcastPeriod = broadcastPeriod;
        }
        
        int getBroadcastPeriod() {
            return broadcastPeriod;
        }
        
        @Override
        public void run() {
            removeAgent(agent);
        }
    }

    public AgentPresenceManager(boolean providesDisconnectionInformation){
        this.providesDisconnectionInformation = providesDisconnectionInformation;
        if (!this.providesDisconnectionInformation){
            timer = new Timer(true); // Timer defined as a daemon
        }
        else {
            timer = null;
        }
    }
    
    @Override
    public void onStatusMessage(StatusMessage s) {
        AgentInfo a = (s.getOriginAgentInfo());
        if (providesDisconnectionInformation){
            if (!(s instanceof StatusForEnd)){
                addAgent(a);
            }
        } else {
            if (s instanceof StatusForEnd){
                log.debug("remove agent on status end");
                removeAgent(a);
            } else {
                
                if (a != null){
                    int broadCastPeriod = (s instanceof HeartBeatStatus) ? ((HeartBeatStatus)s).getStatusBroadcastPeriod() : -1;
                    updateAgent(a, broadCastPeriod);
                }
            }
        }
    }
    
    private synchronized void updateAgent(AgentInfo a, int tempBroadcastPeriod){
        int statusBroadcastPeriod = -1;
        AgentPresenceManager.TimeoutTask task = mapAgents.get(a);
        // Is this agent already known to the map ?
        if (task != null){ // Agent already known to the map : its timeouttask is updated
            task.cancel();
            statusBroadcastPeriod = (tempBroadcastPeriod == -1 ) ? task.getBroadcastPeriod() : tempBroadcastPeriod;
            task = new AgentPresenceManager.TimeoutTask(a,statusBroadcastPeriod);
            addAgent(a, task, false);
        }
        else{ // New agent
            statusBroadcastPeriod = (tempBroadcastPeriod == -1) ? 10 : tempBroadcastPeriod;
            task = new AgentPresenceManager.TimeoutTask(a, statusBroadcastPeriod);
            addAgent(a, task, true);
        }
        log.debug("resetting timer for agent " + a.getName() + " to " + statusBroadcastPeriod);
        timer.schedule(task, SUSPICION_LENGTH*1000*statusBroadcastPeriod);
    }
    
    /**
     * !providesDisconnectionInformation
     * @param agent 
     */
    private synchronized void removeAgent(AgentInfo agent){
        log.debug("removing agent " + agent.getName());
        AgentPresenceManager.TimeoutTask t = mapAgents.remove(agent);
        if (t != null){
            t.cancel();
            disconnecting(agent);
        }else{
            log.debug("removing agent with null timer");
        }
    }
    
    /**
     * providesDisconnectionInformation
     * @param agentName 
     */
    public synchronized void removeAgent(String agentName){
        log.debug("removing agent " + agentName);
        AgentInfo.AgentType agentType = mapAgentsPDI.remove(agentName);
        if (agentType!= null){
            disconnecting(new AgentInfo(agentName, agentType));
        }
    }
    
    /**
     * !providesDisconnectionInformation
     * @param agent
     * @param task
     * @param isNewAgent 
     */
    private synchronized void addAgent(AgentInfo agent, AgentPresenceManager.TimeoutTask task, boolean isNewAgent){
        mapAgents.put(agent, task);
        if (isNewAgent){
            connecting(agent);
        }
    }
    
    /**
     * providesDisconnectionInformation
     * @return 
     */
    public synchronized void addAgent(AgentInfo a){
        if (a != null && mapAgentsPDI.get(a.getName()) == null){
            log.fine("adding " + a + " to the apm agent list");
            mapAgentsPDI.put(a.getName(), a.getType());
            connecting(a);
        }
    }
    
    public  List<AgentInfo> listAgents(){
        return listAgents(0); 
            }
    
    public  List<AgentInfo> listAgents(int timeout){
        if (providesDisconnectionInformation){
            List<AgentInfo> list = new ArrayList<AgentInfo>();
            Set<Map.Entry<String,AgentInfo.AgentType>> set = mapAgentsPDI.entrySet();
            for (Map.Entry<String,AgentInfo.AgentType> e : set){
                list.add(new AgentInfo(e.getKey(),e.getValue()));
            }
            return list; 
        } else {
            return new ArrayList<>(mapAgents.keySet());
        }
    }
    
    public void connecting(AgentInfo agent){
        for (AgentPresenceListener l : listAPL){
            l.connecting(agent);
        }
    }
    
    public void disconnecting(AgentInfo agent){
        for (AgentPresenceListener l : listAPL){
            l.disconnecting(agent);
        }
    }
    
    public void disconnecting(String agentName, String otherInfos) {
        if (providesDisconnectionInformation){
            removeAgent(agentName);
        }
    }
    
    public void anormalEvent(Exception exc) {
//        if(exc instanceof DuplicateAgentNameException) {
//            throw new IllegalArgumentException(exc);
//        }
    }
    
    public void addAgentPresenceListener(AgentPresenceListener l){
        listAPL.add(l);
    }

    public void removeAgentPresenceListener(AgentPresenceListener l){
        listAPL.remove(l);
    }

}
