package org.lsst.ccs.bus;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;

/**
 * This class listens for membership on the status bus, and notifies the registered
 * AgentPresenceListeners when a connection or disconnection is noticed
 * @author emarin
 */
class DefaultAgentPresenceManager extends AgentPresenceManager {
    
    private final Map<Agent,TimeoutTask> mapAgents = new HashMap<Agent,TimeoutTask>();
    private static final int SUSPICION_LENGTH = 3;
    private final Timer timer;

    class TimeoutTask extends TimerTask{
        private final Agent agent;
        private final int broadcastPeriod;
        
        TimeoutTask(Agent agent, int broadcastPeriod){
            this.agent = agent;
            this.broadcastPeriod = broadcastPeriod;
        }
        
        int getBroadcastPeriod() {
            return broadcastPeriod;
        }
        
        @Override
        public void run() {
            removeAgent(agent);
        }
    }
    
    public DefaultAgentPresenceManager(String name){
        super(name);
        timer = new Timer(true); // Timer defined as a daemon
    }
    
    @Override
    public void onStatus(BusMessage s) {
        if (s instanceof Status){
            Agent a = ((Status)s).getAgent();
            if (s instanceof StatusForEnd){
                log.debug("remove agent on status end");
                removeAgent(a);
            } else {
                
                if (a != null){
                    int broadCastPeriod = (s instanceof HeartBeatStatus) ? ((HeartBeatStatus)s).getStatusBroadcastPeriod() : -1;
                    updateAgent(a, broadCastPeriod);
                }
            }
        }
    }
    private synchronized void updateAgent(Agent a, int tempBroadcastPeriod){
        int statusBroadcastPeriod = -1;
        TimeoutTask task = mapAgents.get(a);
        // Is this agent already known to the map ?
        if (task != null){ // Agent already known to the map : its timeouttask is updated
            task.cancel();
            statusBroadcastPeriod = (tempBroadcastPeriod == -1 ) ? task.getBroadcastPeriod() : tempBroadcastPeriod;
            task = new TimeoutTask(a,statusBroadcastPeriod);
            addAgent(a, task, false);
        }
        else{ // New agent
            statusBroadcastPeriod = (tempBroadcastPeriod == -1) ? 10 : tempBroadcastPeriod;
            task = new TimeoutTask(a, statusBroadcastPeriod);
            addAgent(a, task, true);
        }
        log.debug("resetting timer for agent " + a.getName() + " to " + statusBroadcastPeriod);
        timer.schedule(task, SUSPICION_LENGTH*1000*statusBroadcastPeriod);
    }
    
    private synchronized void removeAgent(Agent agent){
        log.debug("removing agent " + agent.getName());
        TimeoutTask t = mapAgents.remove(agent);
        if (t != null){
            t.cancel();
            disconnecting(agent);
        }else{
            log.debug("removing agent with null timer");
        }
    }
    
    private synchronized void addAgent(Agent agent, TimeoutTask task, boolean isNewAgent){
        mapAgents.put(agent, task);
        if (isNewAgent){
            connecting(agent);
        }
    }
    
    public List<Agent> listAgent(int timeout){
        return new ArrayList<Agent>(mapAgents.keySet());
    }
    
    @Override
    public List<Agent> listAgents(int timeout) {
        return listAgent(timeout);
    }
    
    public void connecting(Agent agent){
        for (AgentPresenceListener l : listAPL){
            l.connecting(agent);
        }
    }
    
    public void disconnecting(Agent agent){
        for (AgentPresenceListener l : listAPL){
            l.disconnecting(agent);
        }
    }
    
    public void addAgentPresenceListener(AgentPresenceListener l){
        listAPL.add(l);
    }

    public void removeAgentPresenceListener(AgentPresenceListener l){
        listAPL.remove(l);
    }
}
