package org.lsst.ccs.bus;

//import org.apache.log4j.Logger;

import org.lsst.ccs.utilities.beanutils.Optional;
import org.lsst.ccs.utilities.logging.Logger;
import org.lsst.ccs.utilities.tracers.Tracer;

import java.io.IOException;
import java.util.IdentityHashMap;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Calls a BusMessagingLayer and adds all that is necessary to handle messages
 * following application layer concerns.
 * <p>
 * Each is bound to a subsystem.
 * <p>
 * This class is meant to be subclassed if there is a need to setup complex
 * correlations between messages (or more generally for "application layer" concerns)
 * </P>
 * <B>beware</B> <TT>ThreadLocal</TT> objects may be changed in future releases: they may be
 * moved to <I>Context</I> objects carried through thread creations.
 */
public class BusApplicationLayer {
    protected final String subsystemName;
    protected final BusMessagingLayer busMessagingLayer;
    protected static Logger log = Logger.getLogger("org.lsst.ccs.bus.layer");

    /* REMOVED : do not work
    protected ThreadLocal<String> localCorrelID = new ThreadLocal<String>();
    protected ThreadLocal<String> localOrigin = new ThreadLocal<String>();
    protected ThreadLocal<Command> localCommand = new ThreadLocal<Command>();
    */
    String token = java.util.UUID.randomUUID().toString(); // for arbitrator
    private final MembershipMultiplexor membershipMultiplexor = new MembershipMultiplexor();


    /**
     * creates an entry point to communication for the subsystem.
     * One is not supposed to interact directly with the <TT>BusMessagingLayer</TT>
     * afterwards except for administrative purposes (example: MembershipListeners).
     *
     * @param subsystemName     rules of naming apply
     * @param busMessagingLayer transport layer
     * @throws NullPointerException if busMessagingLayer is null
     */
    public BusApplicationLayer(String subsystemName, BusMessagingLayer busMessagingLayer) {
        this.subsystemName = subsystemName;
        this.busMessagingLayer = busMessagingLayer;
    }

    public String getToken() {
        return token;
    }

    /**
     * registers current subsystem to the Command bus
     *
     * @throws IOException
     */
    public void registerToCommand() throws IOException {
        busMessagingLayer.register(subsystemName, Bus.COMMAND);
    }

    /**
     * registers current subsystem to the Status bus
     *
     * @throws IOException
     */
    public void registerToStatus() throws IOException {
        busMessagingLayer.register(subsystemName, Bus.STATUS);
    }

    /**
     * registers current subsystem to the logging bus
     *
     * @throws IOException
     */
    public void registerToLog() throws IOException {
        busMessagingLayer.register(subsystemName, Bus.LOG);
    }

    /**
     * utility method: parse the destination string in Commands.
     * destination is a list of comma separated list of names(beware of spaces!).
     * rules: if there is a "*" in the destination list then the message is broadcast;
     * if there is a slash in a name only the first part of the name (before the slash)
     * is used for transport destination (so for instance "sft/carrousel" is sent to "sft").
     *
     * @param destination
     * @return an array of agent names or an empty array if broadcasting is requested
     */
    protected String[] parseDestination(String destination) {
        String[] dests = destination.split(",");
        for (int ix = 0; ix < dests.length; ix++) {
            String dest = dests[ix];
            if ("*".equals(dest)) {
                dests = new String[0];
                break;
            }
            if (dest.contains("/")) {
                dests[ix] = dest.substring(0, dest.indexOf("/"));
            }
        }
        return dests;
    }

    /**
     * sends a command message to all destinations.
     * If origin of message is not set then sets it using the current subsystem name.
     * If a Correlation Id is not set then creates one.
     *
     * @param cmd
     * @throws IOException
     * @throws DestinationsException may be thrown if the transport layer is unable to find some of the
     *                               destination and has no broadcast policy in this case.
     * @see org.lsst.ccs.bus.BusMembershipListener#anormalEvent(Exception) for another way to signal
     * destinations exceptions
     */
    public void sendCommand(Command cmd) throws IOException {
        String[] destinations = parseDestination(cmd.getDestination());
        cmd.setKey(token);
        if (cmd.getOrigin() == null) {
            cmd.setOrigin(subsystemName);
        }
        if (cmd.getCorrelId() == null) {
            // let's generate one
            cmd.setCorrelId(java.util.UUID.randomUUID().toString());
        }
        busMessagingLayer.sendMessage(subsystemName, Bus.COMMAND, cmd, destinations);
    }

    /**
     * broadcasts a status message.
     * If origin is not set then sets it with the current subsystem name.
     *
     * @param status
     * @throws IOException
     */
    public void sendStatus(Status status) throws IOException {
        if (status.getOrigin() == null) {
            status.setOrigin(subsystemName);
        }
        busMessagingLayer.sendMessage(subsystemName, Bus.STATUS, status);
    }

    /**
     * broadcasts a log message.
     * If origin is not set then sets it with the current subsystem name.
     *
     * @param evt
     * @throws IOException
     */
    public void sendLogEvent(LogEvent evt) throws IOException {
        if (evt.getOrigin() == null) {
            evt.setOrigin(subsystemName);
        }
        busMessagingLayer.sendMessage(subsystemName, Bus.LOG, evt);

    }

    /**
     * sends a reply or an ack responding to a command.
     * <B>Beware</B> : if <TT>originalCommand</TT> or <TT>CorrelID</TT>
     * or <TT>destination</TT> are not set in the parameter this code will try to populate
     * these fields with informations coming from <TT>ThreadLocal</TT> data so if
     * the initial Thread that received the command spawn children (or more generally
     * if replies or ack are generated in different Threads) then this facility will be defeated.
     *
     * @param cmd
     * @throws IOException
     * @throws DestinationsException may be thrown if the transport layer is unable to find some of the
     *                               destination and has no broadcast policy in this case.
     * @see org.lsst.ccs.bus.BusMembershipListener#anormalEvent(Exception) for another way to signal
     * destinations exceptions
     */
    public void reply(CommandAckOrReply cmd) throws IOException {
        /*  REMOVE
        if(cmd.getOriginalCommand() == null) {
           cmd.setOriginalCommand(localCommand.get()); 
        }
        */
        if (cmd.getOrigin() == null) {
            cmd.setOrigin(subsystemName);
        }
        /* REMOVE correlID is set by ctor of CommandAckOrReply
        if(cmd.getCorrelId() == null) {
           cmd.setCorrelId(localCorrelID.get());
        }
        */
        String destination = cmd.getDestination();
        /* REMOVE should be set by ctor of CommandAckOrReply
        if(destination == null) {
 		    destination= localOrigin.get() ;
            cmd.setDestination(destination);
        }
        */
        busMessagingLayer.sendMessage(subsystemName, Bus.COMMAND, cmd, destination);
    }

    ////////////// Listeners

    /*
    this will lead to a big bug because the same object can be Listening to different org.lsst.gruth.types of message
    IdentityHashMap<Object, BusMessageForwarder> forwarderMap = new IdentityHashMap<Object, BusMessageForwarder>() ;
    */
    IdentityHashMap<CommandListener, BusMessageForwarder> commandForwarderMap = new IdentityHashMap<>();
    IdentityHashMap<LogListener, BusMessageForwarder> logForwarderMap = new IdentityHashMap<>();
    IdentityHashMap<StatusListens, BusMessageForwarder> statusForwarderMap = new IdentityHashMap<>();

    void addMembershipListener(BusMembershipListener l) {
        // We implement this method for now using BusMessagingLayer's setMembershipListener
        // This avoids having to duplicated the functionality in each implementation of BusMessagingLayer
        membershipMultiplexor.addMembershipListener(l);
    }

    void removeMembershipListener(BusMembershipListener l) {
        membershipMultiplexor.removeMembershipListener(l);
    }
    /**
     * Allows multiple listeners to be registered, despite BusMessagingLayer supporting only
     * a single listener.
     */
    private class MembershipMultiplexor implements BusMembershipListener {
        // We use a CopyOnWriteArrayList in case a listener registers or 
        // unregisters while we are iterating to deliver messages 
        private final List<BusMembershipListener> listenerList = new CopyOnWriteArrayList<>();
        private boolean registered = false;
        
        private void addMembershipListener(BusMembershipListener l) {
            synchronized (this) {
                if (!registered) {
                   busMessagingLayer.setMembershipListener(this, Bus.COMMAND);
                   registered = true;
                }
            }
            listenerList.add(l);
        }

        private void removeMembershipListener(BusMembershipListener l) {
            listenerList.remove(l);
        }

        @Override
        public void connecting(String agentName, String otherInfos) {
            for (BusMembershipListener l : listenerList) {
                l.connecting(agentName, otherInfos);
            }
        }

        @Override
        public void disconnecting(String agentName, String otherInfos) {
            for (BusMembershipListener l : listenerList) {
                l.disconnecting(agentName, otherInfos);
            }
        }

        @Override
        public void anormalEvent(Exception exc) {
            for (BusMembershipListener l : listenerList) {
                l.anormalEvent(exc);
            }
        }       
    }
    

    /**
     * instances of this class will forward commands coming from the transport layer
     * to the registered <TT>CommandListener</TT>.
     * When the incoming message is a command to be executed the <TT>TreadLocal</TT>
     * data that links commands with their corresponding relpis or ack is populated
     * (see <TT>reply</TT> method documentation).
     */
    protected class ForwarderToCommand implements BusMessageForwarder {
        CommandListener listener;

        ForwarderToCommand(CommandListener listener) {
            this.listener = listener;
            commandForwarderMap.put(listener, this);
        }

        @Override
        public void update(BusMessage message) {
            if (message instanceof CommandReply) {
                listener.onReply((CommandReply) message);
            } else if (message instanceof CommandAck) {
                listener.onAck((CommandAck) message);
            } else if (message instanceof Command) {
                Command cmd = (Command) message;
                /* REMOVE
                localCommand.set(cmd);
                localOrigin.set(cmd.getOrigin());
                localCorrelID.set(cmd.getCorrelId());
        	   //System.out.println("origin : " + cmd.getOrigin() + " on " + Thread.currentThread());
                */
                try {
                    assert Tracer.trace("executing: " + cmd);
                    listener.onCommand(cmd);
                } catch (Throwable throwable) {
                    log.error("on command :" + throwable);
                }
            }

        }
    }

    /**
     * instances of this class will forward directly status messages to statuslistener
     */
    public class ForwarderToStatus implements BusMessageForwarder {
        final StatusListens listener;
        final boolean isEncodedListener;
        final boolean isKeyValueListener;
        final boolean isDataListener;
        final boolean isCrystallizedListener;

        public ForwarderToStatus(StatusListens listener) {
            this.listener = listener;
            isEncodedListener = listener instanceof EncodedStatusListens;
            isKeyValueListener = listener instanceof KeyValueStatusListener;
            isDataListener = listener instanceof DataStatusListener;
            isCrystallizedListener = listener instanceof SerializedDataStatusListener;
            statusForwarderMap.put(listener, this);
        }

        @Override
        public void update(BusMessage message) {
            if (message instanceof EncodedDataStatus && isEncodedListener) {
                EncodedDataStatus status = (EncodedDataStatus) message;
                String source = status.getOrigin();
                for (EncodedDataStatus dataStatus : status) {
                    long timeStamp = dataStatus.getDataTimestamp();
                    KVList list = dataStatus.getContent();
                    for (KeyData keyData : list) {
                        String key = keyData.getKey();
                        if (isDataListener) {
                            Optional<Object> optObj = keyData.getValue();
                            if (optObj.isPresent()) {
                                ((DataStatusListener) listener).onDataArrival(source, timeStamp, key, optObj.get());

                            }

                        }
                        if (isKeyValueListener) {
                            int id = keyData.hashCode();
                            List<KeyData> detailsList = keyData.getContentAsList();
                            for (KeyData detaileddata : detailsList) {
                                String complexKey = detaileddata.getKey();
                                Optional<Object> optional = detaileddata.getValue();
                                if (optional.isPresent()) {
                                    ((KeyValueStatusListener) listener).onKeyValueStatusDecomposition(source, timeStamp, complexKey, optional.get(), id);
                                }
                            }
                        }
                        if (isCrystallizedListener) {
                            Optional<byte[]> crysta = keyData.getCrystallizedData();
                            if (crysta.isPresent()) {
                                ((SerializedDataStatusListener) listener).onSerializedDataArrival(source, timeStamp, key, crysta.get());
                            }
                        }
                    }

                }


            } else {
                if (listener instanceof StatusListener) {
                    ((StatusListener) listener).onStatus(message);
                }
            }
        }
    }

    /**
     * instances of this class will forward directly log messages to a log listener
     */
    protected class ForwarderToLog implements BusMessageForwarder {
        LogListener listener;

        ForwarderToLog(LogListener listener) {
            this.listener = listener;
            logForwarderMap.put(listener, this);
        }

        @Override
        public void update(BusMessage message) {
            if (message instanceof LogEvent) {
                listener.onLog((LogEvent) message);
            }
        }
    }

    /**
     * registers a CommandListener (in fact a ForwarderToCommand to the underlying transport)
     *
     * @param l
     */
    public void addCommandListener(CommandListener l) {
        busMessagingLayer.addMessageListener(subsystemName, new ForwarderToCommand(l), Bus.COMMAND);
    }

    /**
     * removes a CommandListener: since this command is based on strict identity the listener should be exactly the same
     * as the one registered.
     *
     * @param l
     */
    public void removeCommandListener(CommandListener l) {
        BusMessageForwarder forwarder = commandForwarderMap.remove(l);
        if (forwarder != null) {
            busMessagingLayer.removeMessageListener(subsystemName, forwarder, Bus.COMMAND);
        } else {
            //TODO: log
        }
    }

    /**
     * registers a StatusListener (in fact a ForwarderToStatus to the underlying transport)
     *
     * @param l
     */
    public void addStatusListener(StatusListens l) {
        busMessagingLayer.addMessageListener(subsystemName, new ForwarderToStatus(l), Bus.STATUS);
    }

    public void removeStatusListener(StatusListens l) {
        BusMessageForwarder forwarder = statusForwarderMap.remove(l);
        if (forwarder != null) {
            busMessagingLayer.removeMessageListener(subsystemName, forwarder, Bus.STATUS);
        } else {
            //TODO: log
        }

    }

    /**
     * registers a LogListener (in fact a ForwarderToLog to the underlying transport)
     *
     * @param l
     */
    public void addLogListener(LogListener l) {
        busMessagingLayer.addMessageListener(subsystemName, new ForwarderToLog(l), Bus.LOG);
    }

    public void removeLogListener(LogListener l) {
        BusMessageForwarder forwarder = logForwarderMap.remove(l);
        if (forwarder != null) {
            busMessagingLayer.removeMessageListener(subsystemName, forwarder, Bus.LOG);
        } else {
            //TODO: log
        }

    }

    /**
     * lists the name of agents currently connected to the Command Bus
     * @return
     */
    public List<String> connectedToCommand() {
        return busMessagingLayer.getConnectedNames(Bus.COMMAND) ;
    }

    /**
     * closes the underlying transport layer, stops the listening threads,
     * after this call all other sending calls will fail.
     */
    public void close() {
        busMessagingLayer.closeFor(subsystemName);
    }
}
