package org.lsst.ccs.bus;

import org.apache.log4j.Logger;

import java.io.IOException;

/**
 * Calls a BusMessagingLayer and adds all that is necessary to handle messages
 * following application layer concerns.
 * <P>
 *  Each is bound to a subsystem.
 *  <P>
 *   This class is meant to be subclassed if there is a need to setup complex
 *   correlations between messages (or more generally for "application layer" concerns)
 *  </P>
 *  <B>beware</B> <TT>ThreadLocal</TT> objects may be changed in future releases: they may be
 *  moved to <I>Context</I> objects carried through thread creations.
 *
 */
public class BusApplicationLayer {
    protected final String subsystemName ;
    protected final BusMessagingLayer busMessagingLayer ;
    protected static Logger log = Logger.getLogger("lsst.ccs.subsystem");

    /*
     * TODO: change correlation and adresses in CommandAck and CommandReply
     */
    protected ThreadLocal<String> localCorrelID = new ThreadLocal<String>();
    protected ThreadLocal<String> localOrigin = new ThreadLocal<String>();
    protected ThreadLocal<Command> localCommand = new ThreadLocal<Command>();
    String token = java.util.UUID.randomUUID().toString(); // for arbitrator


    /**
     * creates an entry point to communication for the subsystem.
     * One is not supposed to interact directly with the <TT>BusMessagingLayer</TT>
     * afterwards except for administrative purposes (example: MembershipListeners).
     * @param subsystemName rules of naming apply
     * @param busMessagingLayer transport layer
     * @throws NullPointerException if busMessagingLayer is null
     */
    public BusApplicationLayer(String subsystemName, BusMessagingLayer busMessagingLayer) {
        this.subsystemName = subsystemName;
        this.busMessagingLayer = busMessagingLayer;
    }
    
    public String getToken() {
        return token ;
    }

    /**
     * registers current subsystem to the Command bus
     * @throws IOException
     */
    public void registerToCommand() throws IOException {
        busMessagingLayer.register(subsystemName, Bus.COMMAND);
    }

    /**
     * registers current subsystem to the Status bus
     * @throws IOException
     */
    public void registerToStatus() throws IOException {
        busMessagingLayer.register(subsystemName, Bus.STATUS);
    }

    /**
     * registers current subsystem to the logging bus
     * @throws IOException
     */
    public void registerToLog() throws IOException {
        busMessagingLayer.register(subsystemName, Bus.LOG);
    }

    /**
     * utility method: parse the destination string in Commands.
     * destination is a list of comma separated list of names(beware of spaces!).
     * rules: if there is a "*" in the destination list then the message is broadcast;
     * if there is a slash in a name only the first part of the name (before the slash)
     * is used for transport destination (so for instance "sft/carrousel" is sent to "sft").
     * @param destination
     * @return an array of agent names or an empty array if broadcasting is requested
     */
    protected String[] parseDestination(String destination) {
        String[] dests = destination.split(",");
        for(int ix = 0 ; ix < dests.length ;ix++){
            String dest = dests[ix] ;
            if("*".equals(dest)) {
                dests = new String[0] ;
                break ;
            }
            if(dest.contains("/")) {
                dests[ix] = dest.substring(0, dest.indexOf("/"));
            }
        }
        return dests ;
    }

    /**
     * sends a command message to all destinations.
     * If origin of message is not set then sets it using the current subsystem name.
     * If a Correlation Id is not set then creates one.
     * @param cmd
     * @throws IOException
     * @throws DestinationsException may be thrown if the transport layer is unable to find some of the
     * destination and has no broadcast policy in this case.
     * @see org.lsst.ccs.bus.BusMembershipListener#anormalEvent(Exception) for another way to signal
     * destinations exceptions
     */
    public void sendCommand(Command cmd) throws IOException {
        String[] destinations = parseDestination(cmd.getDestination()) ;
        cmd.setKey(token);
        if(cmd.getOrigin() == null) {
            cmd.setOrigin(subsystemName);
        }
        if (cmd.getCorrelId() == null) {
            // let's generate one
            cmd.setCorrelId(java.util.UUID.randomUUID().toString());
        }
        busMessagingLayer.sendMessage(subsystemName,Bus.COMMAND,cmd,destinations);
    }

    /**
     * broadcasts a status message.
     * If origin is not set then sets it with the current subsystem name.
     * @param status
     * @throws IOException
     */
    public void sendStatus(Status status) throws IOException {
        if(status.getOrigin() == null) {
            status.setOrigin(subsystemName);
        }
        busMessagingLayer.sendMessage(subsystemName,Bus.STATUS,status);
    }

    /**
     * broadcasts a log message.
     * If origin is not set then sets it with the current subsystem name.
     * @param evt
     * @throws IOException
     */
    public void sendLogEvent(LogEvent evt) throws IOException {
        if(evt.getOrigin() == null) {
            evt.setOrigin(subsystemName);
        }
        busMessagingLayer.sendMessage(subsystemName,Bus.LOG,evt);
        
    }

    /**
     *  sends a reply or an ack responding to a command.
     *  <B>Beware</B> : if <TT>originalCommand</TT> or <TT>CorrelID</TT>
     *  or <TT>destination</TT> are not set in the parameter this code will try to populate
     *  these fields with informations coming from <TT>ThreadLocal</TT> data so if
     *  the initial Thread that received the command spawn children (or more generally
     *  if replies or ack are generated in different Threads) then this facility will be defeated.
     * @param cmd
     * @throws IOException
     * @throws DestinationsException may be thrown if the transport layer is unable to find some of the
     * destination and has no broadcast policy in this case.
     * @see org.lsst.ccs.bus.BusMembershipListener#anormalEvent(Exception) for another way to signal
     * destinations exceptions
     */
    public void reply(CommandAckOrReply cmd) throws IOException {
        if(cmd.getOriginalCommand() == null) {
           cmd.setOriginalCommand(localCommand.get()); 
        }
        if(cmd.getOrigin() == null) {
            cmd.setOrigin(subsystemName);
        }
        if(cmd.getCorrelId() == null) {
           cmd.setCorrelId(localCorrelID.get());
        }
        String destination = cmd.getDestination() ;
        if(destination == null) {
 		    destination= localOrigin.get() ;
            cmd.setDestination(destination);
        }
        busMessagingLayer.sendMessage(subsystemName,Bus.COMMAND,cmd,destination);
    }
    
    ////////////// Listeners

    /**
     * instances of this class will forward commands coming from the transport layer
     * to the registered <TT>CommandListener</TT>.
     * When the incoming message is a command to be executed the <TT>TreadLocal</TT>
     * data that links commands with their corresponding relpis or ack is populated
     * (see <TT>reply</TT> method documentation).
     */
    protected class ForwarderToCommand implements BusMessageForwarder {
        CommandListener listener ;

        ForwarderToCommand(CommandListener listener) {
            this.listener = listener;
        }

        @Override
        public void update(BusMessage message) {
            if (message instanceof CommandReply) {
               listener.onReply((CommandReply) message);
            } else if (message instanceof CommandAck) {
                listener.onAck((CommandAck) message);
            }
            else if(message instanceof Command) {
                Command cmd = (Command) message ;
                //TODO : set a Context for Thread
                localCommand.set(cmd);
                localOrigin.set(cmd.getOrigin());
                localCorrelID.set(cmd.getCorrelId());
        	   //System.out.println("origin : " + cmd.getOrigin() + " on " + Thread.currentThread());
                try {
                listener.onCommand(cmd);
                } catch (Throwable throwable) {
                    log.error("on command :" + throwable);
                }
            }
                
        }
    }

    /**
     * instances of this class will forward directly status messages to statuslistener
     */
    protected class ForwarderToStatus implements BusMessageForwarder {
        StatusListener listener ;

        ForwarderToStatus(StatusListener listener) {
            this.listener = listener;
        }

        @Override
        public void update(BusMessage message) {
                listener.onStatus(message);
        }
    }

    /**
     * instances of this class will forward directly log messages to a log listener
     *
     */
    protected class ForwarderToLog implements BusMessageForwarder {
        LogListener listener ;

        ForwarderToLog(LogListener listener) {
            this.listener = listener;
        }

        @Override
        public void update(BusMessage message) {
            if(message instanceof LogEvent) {
                listener.onLog((LogEvent) message);
            }
        }
    }

    /**
     * registers a CommandListener (in fact a ForwarderToCommand to the underlying transport)
     * @param l
     */
    public void addCommandListener(CommandListener l) {
        busMessagingLayer.addMessageListener(subsystemName, new ForwarderToCommand(l), Bus.COMMAND);
    }

    /**
     * registers a StatusListener (in fact a ForwarderToStatus to the underlying transport)
     * @param l
     */
    public void addStatusListener(StatusListener l) {
        busMessagingLayer.addMessageListener(subsystemName, new ForwarderToStatus(l), Bus.STATUS);
    }

    /**
     * registers a LogListener (in fact a ForwarderToLog to the underlying transport)
     * @param l
     */
    public void addLogListener(LogListener l) {
        busMessagingLayer.addMessageListener(subsystemName, new ForwarderToLog(l),Bus.LOG);
    }

    /**
     * closes the underlying transport layer, stops the listening threads,
     * after this call all other sending calls will fail.
     */
    public void close() {
        busMessagingLayer.closeFor(subsystemName);
    }
}
