package org.lsst.ccs.bus.jms;

import org.lsst.ccs.bus.*;
import org.lsst.ccs.utilities.logging.Logger;

import javax.jms.Destination;
import javax.jms.JMSException;
import javax.jms.MessageConsumer;
import javax.jms.MessageListener;
import javax.jms.MessageProducer;
import javax.jms.ObjectMessage;
import javax.jms.Queue;
import javax.jms.Session;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;

/**
 * JMS implementation of the messaging factory
 * 
 * @author aubourg
 * 
 */
public class JMSMessagingFactory extends MessagingFactory {

	// Thread local pattern assumes reply will be send on same thread
	// will not handle deferred executions of commands.
	// thread local context is only set when receiving a message
	ThreadLocal<String> localCorrelID = new ThreadLocal<String>();
	ThreadLocal<Destination> localReplyTo = new ThreadLocal<Destination>();
	ThreadLocal<String> localOrigin = new ThreadLocal<String>();
	ThreadLocal<Command> localCommand = new ThreadLocal<Command>();
	String token = java.util.UUID.randomUUID().toString(); // for arbitrator

	// should be set to "MAIN" for the main control module that holds all locks
	// initially
	// TODO (at the moment default value for subsystem allows any command)
	@Override
	public String getToken() {
		return token;
	}

	protected static Logger log = Logger
			.getLogger("org.lsst.ccs.bus.jms.JMSMessagingFactory");

	@Override
	public void addCommandListener(final CommandListener l) {
		addCommandListener(l, null);
	}

	// TODO add executor service to handle multiple commands in parallel
	// would process the payload
	ExecutorService exec = Executors.newFixedThreadPool(5); // TODO configurable

	class MessageHandler implements Runnable {

		public MessageHandler(ObjectMessage msg, CommandListener l) {
			this.msg = msg;
			this.l = l;
		}

		ObjectMessage msg;
		CommandListener l;

		public void run() {
			Object payload;
			try {
				payload = ((ObjectMessage) msg).getObject();

				if (payload instanceof Command) {
					localCorrelID.set(msg.getJMSCorrelationID());
					localReplyTo.set(msg.getJMSReplyTo());
					localOrigin.set(((Command) payload).getOrigin());
					localCommand.set((Command) payload);
					l.onCommand((Command) payload);
				} else if (payload instanceof CommandReply) {
					localCorrelID.set(msg.getJMSCorrelationID());
					localReplyTo.set(null);
					localOrigin.set(null);
					l.onReply((CommandReply) payload);
				} else if (payload instanceof CommandAck) {
					localCorrelID.set(msg.getJMSCorrelationID());
					localReplyTo.set(null);
					localOrigin.set(null);
					l.onAck((CommandAck) payload);
				} else {
					log.warn("Message payload type "
							+ payload.getClass().getName() + " not handled "
							+ payload);
				}
			} catch (JMSException e) {
				log.error("Problem receiving message, class not handled?", e);
			}

		}
	}

	// in order to remove listeners, we need to keep track of their associated
	// consumer.

	Map<CommandListener, MessageConsumer> commandListenerMap = new ConcurrentHashMap<CommandListener, MessageConsumer>();

	@Override
	public void addCommandListener(final CommandListener l, String selector) {
		log.debug("adding command listener, selector='" + selector + "'");
		Session session = TopicSessionFactory.getSessionFactory()
				.getTopicSession();
		MessageConsumer sub = (selector == null) ? TopicSessionFactory
				.getSessionFactory().getCommandSubscriber(session)
				: TopicSessionFactory.getSessionFactory().getCommandSubscriber(
						session, selector);
		try {
			MessageListener ml = new MessageListener() {

				public void onMessage(javax.jms.Message msg) {
					if (msg instanceof ObjectMessage) {
						log.debug("MessageListener got jms message on command");

						MessageHandler mh = new MessageHandler(
								(ObjectMessage) msg, l);
						exec.execute(mh);

						// Object payload;
						// try {
						// payload = ((ObjectMessage) msg).getObject();
						//
						// if (payload instanceof Command) {
						// localCorrelID.set(msg.getJMSCorrelationID());
						// localReplyTo.set(msg.getJMSReplyTo());
						// localOrigin
						// .set(((Command) payload).getOrigin());
						// localCommand.set((Command) payload);
						// l.onCommand((Command) payload);
						// } else if (payload instanceof CommandReply) {
						// localCorrelID.set(msg.getJMSCorrelationID());
						// localReplyTo.set(null);
						// localOrigin.set(null);
						// l.onReply((CommandReply) payload);
						// } else {
						// log.warn("Message payload type "
						// + payload.getClass().getName()
						// + " not handled " + payload);
						// }
						// } catch (JMSException e) {
						// throw new RuntimeException(e);
						// }
					} else {
						log.warn("Message  type " + msg.getClass().getName()
								+ " not handled " + msg);

					}
				}
			};
			commandListenerMap.put(l, sub);
			sub.setMessageListener(ml);
			commandMessageListener = ml;
		} catch (JMSException e) {
			throw new RuntimeException(e);
		}
	}

	@Override
	public void removeCommandListener(CommandListener l) {
		MessageConsumer sub = commandListenerMap.get(l);
		// if sub is null it's either an error from the caller or could be a
		// race condition, to be handled : locking, retry ?
		if (sub == null)
			return;
		try {
			sub.setMessageListener(null);
			sub.close();
		} catch (JMSException e) {
			throw new RuntimeException(e);
		}
		commandListenerMap.remove(l);
	}

	MessageListener commandMessageListener; // TDOD

	// We keep it because we want to use it a the target for command replies
	// TODO : separate cleanly commands and command replies
	// check what we should do with the temporary queue
	// TODO : check if we don't keep a unique queue for a subsystem
	// instead of temp queues for each commands
	// backup solution is to send command replies on the command topic, which
	// works (was version 1 still commented out).
	@Override
	public void addStatusListener(final StatusListens l) {
		addStatusListener(l, null);
	}

	Map<StatusListens, MessageConsumer> statusListenerMap = new ConcurrentHashMap<StatusListens, MessageConsumer>();

	@Override
	public void addStatusListener(final StatusListens l, String selector) {
		log.debug("adding status listener, selector='" + selector + "'");
		Session session = TopicSessionFactory.getSessionFactory()
				.getTopicSession();
		MessageConsumer sub = TopicSessionFactory.getSessionFactory()
				.getStatusSubscriber(session, selector);
		MessageListener ml = new MessageListener() {
            //toDO: CHANGE THAT!!!!
            BusApplicationLayer.ForwarderToStatus forwarder =
                    new BusApplicationLayer("", null).new ForwarderToStatus(l);

			public void onMessage(javax.jms.Message msg) {
				if (msg instanceof ObjectMessage) {
					log.debug("MessageListener got jms message on status");
					Object payload;
					try {
						payload = ((ObjectMessage) msg).getObject();
					} catch (JMSException e) {
						log.error(
								"Got Object Message, could not deserialize: missing class?",
								e);
						return;
						// throw new RuntimeException(e);
					}
					if (payload instanceof Status) {
                        forwarder.update((BusMessage)payload);
					} else {
						log.warn("Message payload type "
								+ payload.getClass().getName()
								+ " not handled " + payload);

					}
				} else {
					log.warn("Message  type " + msg.getClass().getName()
							+ " not handled " + msg);

				}
			}
		};
		try {
			statusListenerMap.put(l, sub);
			sub.setMessageListener(ml);
		} catch (JMSException e) {
			log.error(e);
			throw new RuntimeException(e);
		}
	}

	@Override
	public void removeStatusListener(StatusListens l) {
		MessageConsumer sub = statusListenerMap.get(l);
		// if sub is null it's either an error from the caller or could be a
		// race condition, to be handled : locking, retry ?
		if (sub == null)
			return;
		try {
			sub.setMessageListener(null);
			sub.close();
		} catch (JMSException e) {
			throw new RuntimeException(e);
		}
		statusListenerMap.remove(l);
	}

	@Override
	public void addLogListener(LogListener l) {
		addLogListener(l, null);
	}

	Map<LogListener, MessageConsumer> logListenerMap = new ConcurrentHashMap<LogListener, MessageConsumer>();

	@Override
	public void addLogListener(final LogListener l, String selector) {
		log.debug("adding log listener, selector='" + selector + "'");
		Session session = TopicSessionFactory.getSessionFactory()
				.getTopicSession();
		MessageConsumer sub = TopicSessionFactory.getSessionFactory()
				.getLogSubscriber(session, selector);
		MessageListener ml = new MessageListener() {

			public void onMessage(javax.jms.Message msg) {
				if (msg instanceof ObjectMessage) {
					log.debug("MessageListener got jms message on log");
					Object payload;
					try {
						payload = ((ObjectMessage) msg).getObject();
					} catch (JMSException e) {
						throw new RuntimeException(e);
					}
					if (payload instanceof LogEvent) {
						l.onLog((LogEvent) payload);
					} else {
						log.warn("Message payload type "
								+ payload.getClass().getName()
								+ " not handled " + payload);

					}
				} else {
					log.warn("Message  type " + msg.getClass().getName()
							+ " not handled " + msg);

				}
			}
		};
		try {
			logListenerMap.put(l, sub);
			sub.setMessageListener(ml);
		} catch (JMSException e) {
			log.error(e);
			throw new RuntimeException(e);
		}

	}

	@Override
	public void removeLogListener(LogListener l) {
		MessageConsumer sub = logListenerMap.get(l);
		// if sub is null it's either an error from the caller or could be a
		// race condition, to be handled : locking, retry ?
		if (sub == null)
			return;
		try {
			sub.setMessageListener(null);
			sub.close();
		} catch (JMSException e) {
			throw new RuntimeException(e);
		}
		logListenerMap.remove(l);
	}

	@Override
	public void sendCommand(Command cmd) {
		cmd.setKey(token);
		// Hack added by bamade
		if (null == cmd.getOrigin()) {
			cmd.setOrigin(MessagingFactory.getInstance().getSubsystemName());
		}
		if (cmd.getCorrelId() == null) {
			cmd.setCorrelId(localCorrelID.get());
		}
		if (cmd.getCorrelId() == null) {
			// let's generate one
			cmd.setCorrelId(java.util.UUID.randomUUID().toString());
		}
		Session sess = TopicSessionFactory.getSessionFactory()
				.getTopicSession();
		MessageProducer send = TopicSessionFactory.getSessionFactory()
				.getCommandPublisher(sess);
		ObjectMessage m;

		try {
			m = sess.createObjectMessage();
			m.setObject(cmd);
			m.setJMSType(cmd.getMessageType());
			m.setStringProperty("destination", cmd.getDestination());
			m.setJMSPriority(convertToJMSPriority(cmd.getPriorityLevel()));
			if (cmd.getCorrelId() != null) {
				m.setJMSCorrelationID(cmd.getCorrelId());
			}
			log.debug("sending command " + cmd);

			sendMessage(m, sess, send);
			sess.close();
		} catch (JMSException e) {
			log.error(e);
			throw new RuntimeException(e);
		}
	}

	public static int convertToJMSPriority(PriorityLevel level) {
		switch (level) {
		case DEBUG:
			return 0;
		case INFO:
			return 4;
		case OPERATION:
			return 5;
		case CRITICAL:
			return 9;
		}
		return 4; // should not happen
	}

	@Override
	public void sendStatus(Status s) {
		// hack added by bamade
		if (null == s.getOrigin()) {
			s.setOrigin(MessagingFactory.getInstance().getSubsystemName());
		}

		Session sess = TopicSessionFactory.getSessionFactory()
				.getTopicSession();
		MessageProducer send = TopicSessionFactory.getSessionFactory()
				.getStatusPublisher(sess);
		ObjectMessage m;
		try {
			m = sess.createObjectMessage();
			m.setObject(s);
			m.setJMSType(s.getMessageType()); // "lsst.status", "lsst.alarm"...
			m.setJMSTimestamp(System.currentTimeMillis());// or source timestamp
			// ??
			m.setJMSPriority(convertToJMSPriority(s.getPriorityLevel())); // 0 =
			// LOW,
			// 9
			// =
			// HIGH
			m.setStringProperty("destination", "*");
			log.debug("sending status " + s);
			sendMessage(m, sess, send);
			sess.close();
		} catch (JMSException e) {
			log.error(e);
			throw new RuntimeException(e);
		}
	}

	@Override
	public void sendLogEvent(LogEvent evt) {
		Session sess = TopicSessionFactory.getSessionFactory()
				.getTopicSession();
		MessageProducer send = TopicSessionFactory.getSessionFactory()
				.getLogPublisher(sess);
		ObjectMessage m;
		try {
			m = sess.createObjectMessage();
			m.setObject(evt);
			m.setJMSType(evt.getMessageType());
			m.setStringProperty("destination", "*");
			m.setJMSPriority(convertToJMSPriority(evt.getPriorityLevel()));
			log.debug("sending log " + evt);

			sendMessage(m, sess, send);
			sess.close();
		} catch (JMSException e) {
			log.error(e);
			throw new RuntimeException(e);
		}

	}

	protected void sendMessage(ObjectMessage m, Session sess,
			MessageProducer sender) {
		try {
			if (m.getJMSCorrelationID() == null) {
				m.setJMSCorrelationID(localCorrelID.get());
			}
			m.setJMSMessageID("0"); /* TODO handle messageid */
			m.setJMSPriority(4); // 0-4 normal, 4-9 expedited

			// uncomment to send replies on the command topic

			m.setJMSReplyTo(TopicSessionFactory.getSessionFactory()
					.getCommandTopic());

			// and then comment out what follows

			// TODO close the consumer and delete the queue when reply
			// is received !!

			// create temp queues for replies could be resource-consuming
			// check and if yes revert to sending replies on command bus
			//
			// Session qs = QueueSessionFactory.getSessionFactory()
			// .getQueueSession();
			// Queue replyQueue = qs.createTemporaryQueue();
			// MessageConsumer c = qs.createConsumer(replyQueue);
			// c.setMessageListener(commandMessageListener);
			// m.setJMSReplyTo(replyQueue);

			// TODO should we provide rather a way of waiting
			// for the reply by polling the reply queue ?
			// need for a scheme to wait for reply...

			m.setJMSTimestamp(System.currentTimeMillis());

			sender.send(m);
			sender.close();
		} catch (JMSException e) {
			log.error("Error sending message", e);
			throw new RuntimeException(e);
		} finally {
			try {
				sess.close();
			} catch (JMSException e) {
				log.error("Error closing JMS session", e);
				throw new RuntimeException(e);
			}
		}

	}

	@Override
	public boolean isReplyRequested() {
		return localReplyTo.get() != null;
	}

	@Override
	public void noAutoReply() {
		localReplyTo.set(null);
	}

	@Override
	public void reply(CommandAckOrReply reply) {
		Destination dest = localReplyTo.get();
		String destSystem = localOrigin.get();

		if (reply.getCorrelId() == null) {
			reply.setCorrelId(localCorrelID.get());
		}

		reply.setOriginalCommand(localCommand.get());
		// hack added by bamade
		if (null == reply.getOrigin()) {
			reply.setOrigin(MessagingFactory.getInstance().getSubsystemName());
		}

		log.debug("sending reply " + reply + " to " + dest + "[" + destSystem
				+ "]");

		// is it a queue or a topic ?
		// with latest jms possible to factor out part of code wih Sessions and
		// MessagePublishers
		// TODO in progress

		if (dest instanceof Queue) {
			Session sess = QueueSessionFactory.getSessionFactory()
					.getQueueSession();
			MessageProducer sender = QueueSessionFactory.getSessionFactory()
					.getCommandReplySender(sess, dest);
			ObjectMessage m;
			try {
				m = sess.createObjectMessage();
				m.setObject(reply);
				if (reply instanceof CommandReply)
					m.setJMSType("lsst.commandreply");
				else
					m.setJMSType("lsst.commandack");
				m.setStringProperty("destination", destSystem);
				m.setJMSCorrelationID(reply.getCorrelId());
				sender.send(m);
				sender.close();
				sess.close();
			} catch (JMSException e) {
				log.error(e);
			}
		} else {
			Session sess = TopicSessionFactory.getSessionFactory()
					.getTopicSession();
			MessageProducer sender = TopicSessionFactory.getSessionFactory()
					.getCommandReplyPublisher(sess, dest);
			ObjectMessage m;
			try {
				m = sess.createObjectMessage();
				m.setObject(reply);
				if (reply instanceof CommandReply)
					m.setJMSType("lsst.commandreply");
				else
					m.setJMSType("lsst.commandack");
				m.setStringProperty("destination", destSystem);
				m.setJMSCorrelationID(reply.getCorrelId());
				sender.send(m);
				sender.close();
				sess.close();
			} catch (JMSException e) {
				log.error(e);
			}

		}

	}
}
