package org.lsst.ccs.messaging.jgroups;

import java.awt.Dimension;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.BufferedReader;
import java.io.FileInputStream;
import java.io.InputStreamReader;
import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.jgroups.Address;
import org.jgroups.JChannel;
import org.jgroups.Message;
import org.jgroups.ReceiverAdapter;
import org.jgroups.View;
import org.jgroups.conf.XmlConfigurator;
import org.jgroups.protocols.pbcast.GMS;
import org.jgroups.stack.MembershipChangePolicy;
import org.jgroups.stack.Protocol;
import org.jgroups.tests.Probe;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * JGroups tester.
 *
 * @author onoprien
 */
public class Tester {

// -- Fields : -----------------------------------------------------------------
    
    static private final AtomicInteger ID = new AtomicInteger();
    static private final String CLUSTER_NAME = "TEST";
    
    private String name;
    private int rank;
    private String xmlConfigurationFile = "udp_ccs.xml";
    private Properties jgroupsProperties;
    private final boolean showGUI;
    
    private JChannel channel;
    private JFrame frame;
    private int level = Level.INFO.intValue();
    
    private Random random = new Random();

// -- Life cycle : -------------------------------------------------------------
    
    public Tester(String name, boolean showGUI) {
        
        this.name = name == null ? "Node_"+ Integer.toString(ID.getAndIncrement()) : name;
        this.showGUI = showGUI;
        
        String propertiesConfigurationFile = xmlConfigurationFile.replace(".xml", ".properties");
        jgroupsProperties = BootstrapResourceUtils.getBootstrapProperties(propertiesConfigurationFile, this.getClass());
        String sysPref = "system.property.";
        for (String key : jgroupsProperties.stringPropertyNames()) {
            if (key.startsWith(sysPref)) {
                String shortKey = key.substring(sysPref.length());
                if (System.getProperty(shortKey) == null) {
                    System.setProperty(shortKey, jgroupsProperties.getProperty(key));
                }
            }
        }
        
//        StringBuilder sbb = new StringBuilder();
//        sbb.append("jgroupsProperties:\n");
//        for (String key : jgroupsProperties.stringPropertyNames()) {
//            sbb.append(key).append("=").append(jgroupsProperties.getProperty(key)).append("\n");
//        }
//        sbb.append("\nSystem Properties:\n");
//        for (String key : System.getProperties().stringPropertyNames()) {
//            sbb.append(key).append("=").append(System.getProperty(key)).append("\n");
//        }
//        sbb.append("\n");
//        System.out.println(sbb);
    }
    
    public Tester() {
        this(null, false);
    }
    
    public Tester(boolean showGUI) {
        this(null, showGUI);
    }
    
    private void start() {
        if (showGUI) {
            SwingUtilities.invokeLater(this::makeGUI);
        }
    }
    
    private void stop() {
        if (frame != null) frame.dispose();
        channel.close();
    }
    
    private void makeGUI() {
        
        frame = new JFrame(name);
        frame.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosed(WindowEvent e) {
                stop();
            }
        });
        Box root = Box.createVerticalBox();
        frame.add(new JScrollPane(root));
        
        // Tester settings:
        
        Box box = Box.createHorizontalBox();
        
        box.add(new JLabel("Output level: "));
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        Level[] levels = {Level.SEVERE, Level.WARNING, Level.INFO, Level.FINE, Level.FINER, Level.FINEST};
        JComboBox<Level> combo = new JComboBox<>(levels);
        combo.setSelectedItem(Level.INFO);
        combo.addActionListener(e -> {
            JComboBox<Level> c = (JComboBox<Level>) e.getSource();
            level = ((Level)c.getSelectedItem()).intValue();
        });
        box.add(combo);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        box.add(Box.createHorizontalGlue());
        root.add(box);
        root.add(Box.createRigidArea(new Dimension(5,5)));
        
        // Create and connect channel
        
        box = Box.createHorizontalBox();
        
        JButton b = new JButton("Create channel");
        b.addActionListener(e -> {
            ((JButton)e.getSource()).setEnabled(false);
            createChannel();
        });
        box.add(b);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        b = new JButton("Connect");
        b.addActionListener(e -> {
            if (e.getActionCommand().equals("Connect")) {
                ((JButton)e.getSource()).setText("Disconnect");
                connectChannel(null);
            } else {
                ((JButton)e.getSource()).setText("Connect");
                disconnectChannel();
            }
        });
        box.add(b);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        box.add(Box.createHorizontalGlue());
        root.add(box);
        root.add(Box.createRigidArea(new Dimension(5,5)));
        
        // Send messages and print view
        
        box = Box.createHorizontalBox();
        
        b = new JButton("Send");
        b.addActionListener(e -> sendMessage());
        box.add(b);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        b = new JButton("View");
        b.addActionListener(e -> logView());
        box.add(b);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        box.add(Box.createHorizontalGlue());
        root.add(box);
        root.add(Box.createRigidArea(new Dimension(5,5)));
        
        // Probe :
        
        box = Box.createHorizontalBox();
        
        box.add(new JLabel("Probe"));
        box.add(Box.createRigidArea(new Dimension(5,5)));
        JTextField f = new JTextField();
        f.addActionListener(e -> probe(e.getActionCommand()));
        box.add(f);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        box.add(Box.createHorizontalGlue());
        root.add(box);
        root.add(Box.createRigidArea(new Dimension(5,5)));
        
        // Finish
        
        root.add(Box.createVerticalGlue());
        frame.pack();
        frame.setVisible(true);
    }
    
// -- Commands : ---------------------------------------------------------------
    
    private String setName(String name) {
        if (channel != null) return "Too late, the channel has already been created.";
        this.name = name == null ? "Node_"+ Integer.toString(ID.getAndIncrement()) : name;
        return "Changed name to "+ this.name;
    }
    
    private String setRank(int rank) {
        if (channel != null) return "Too late, the channel has already been created.";
        if (rank > 15) return "Rank should be less than 16";
        this.rank = rank;
        return "Changed rank to "+ this.rank;
    }
    
    private String setConfig(String config) {
        if (channel != null) return "Too late, the channel has already been created.";
        if (!(config.startsWith("/") || config.startsWith("c:"))) {
            config = System.getProperty("user.home") + "/" + config;
        }
        xmlConfigurationFile = config;
        return "Changed config to "+ xmlConfigurationFile;
    }
    
    private String loadProperties(String config) throws Exception {
        if (channel != null) return "Too late, the channel has already been created.";
        if (!(config.startsWith("/") || config.startsWith("c:"))) {
            config = System.getProperty("user.home") + "/" + config;
        }
        if (!config.endsWith(".properties")) {
            config = config +".properties";
        }
        Properties prop = new Properties();
        prop.load(new FileInputStream(config));
        prop.forEach((key, value) -> {
            String newKey = (String)key;
            if (!newKey.startsWith("org.lsst.ccs.jgroups.")) {
                newKey = "org.lsst.ccs.jgroups."+ newKey;
            }
            jgroupsProperties.put(newKey, value);
        });
        return "Loaded from "+ config;
    }
    
    private String setProperty(String key, String value) {
        if (!key.startsWith("org.lsst.ccs.jgroups.")) {
            key = "org.lsst.ccs.jgroups." + key;
        }
        jgroupsProperties.put(key, value);
        return "Set "+ key +"="+ value;
    }
    
    private String setLogLevel(String level, String protocol) {
        if (channel == null) return "Channel does not exist.";
        if (protocol == null) {
            channel.stack().setLevel(level);
            return "Stack level: "+ channel.stack().getLevel();
        } else {
            Protocol prot = channel.stack().findProtocol(protocol);
            if (prot == null) {
                return "No such protocol: "+ protocol;
            } else {
                prot.setLevel(level);
                return prot.getName() +" level: "+ prot.getLevel();
            }
        }
    }
    
    private String createChannel() {
        if (channel != null) return "Too late, the channel has already been created.";
        
        try {
            XmlConfigurator configurator = JGroupsBusMessagingLayer.createXmlConfiguratorForBus(xmlConfigurationFile, jgroupsProperties, CLUSTER_NAME);
            channel = new JChannel(configurator);

            if (!(rank < 0 || "true".equalsIgnoreCase(jgroupsProperties.getProperty("ccs.jg.disable_rank")))) {
                channel.addAddressGenerator(new CCSAddress.Generator(rank));
            }
            
            channel.setName(name);
            String text = "Channel configuration:\n"+ channel.getProperties();
            log(text, Level.FINE);
            channel.setReceiver(new MessageReceiver());
            return "Created channel "+ name +" with rank "+ rank;
        } catch (Exception x) {
            return "Error creating channel"+ x;
        }
    }
    
    private String connectChannel(String clusterName) {
        if (channel == null) return "Channel does not exist.";
        if (channel.isConnected()) return "Channel is already connected.";
        if (channel.isConnecting()) return "Channel is already connecting.";
        if (clusterName == null) clusterName = CLUSTER_NAME;
        try {
            channel.connect(clusterName);
            return "Connected to cluster "+ clusterName;
        } catch (Exception x) {
            return "Error connecting channel. "+ x;
        }
    }
    
    private String disconnectChannel() {
        if (channel == null) return "Channel does not exist.";
        if (!channel.isConnected()) return "Channel is not connected.";
        try {
            String cluster = channel.clusterName();
            channel.disconnect();
            return "Disconnected "+ name +" from cluster "+ cluster;
        } catch (Exception x) {
            return "Error disconnecting channel. "+ x;
        }
    }
    
    private String sendMessage() {
        if (channel == null) return "Channel does not exist.";
        if (!channel.isConnected()) return "Channel is not connected.";
        try {
            channel.send(null, "default payload");
            return "Sent from "+ name +" to "+ channel.clusterName();
        } catch (Exception x) {
            return "Error sending message. "+ x;
        }
    }
    
    private String invokeOnChannel(String method) {
        if (channel == null) return "Channel does not exist.";
        try {
            return JChannel.class.getMethod(method).invoke(channel).toString();
        } catch (NoSuchMethodException x) {
            return "No such method: "+ method;
        } catch (IllegalAccessException x) {
            return "IllegalAccessException";
        } catch (InvocationTargetException x) {
            return "InvocationTargetException: "+ x;
        }
    }
    
    private String testMembership() {
        if (channel == null || !channel.isConnected()) return "Channel does not exist or is not connected.";
        List<Address> adds = channel.getView().getMembers();
        int n = adds.size();
        if (n < 3) return "Just "+ n +" nodes, wouldn't be much of a test";
        Set<Address> all = new HashSet<>(adds);
        if (all.size() != n) return "Equal addresses, list = "+ adds.toString() +", set = "+ all.toString();
        MembershipChangePolicy mp = ((GMS)channel.getProtocolStack().findProtocol(GMS.class)).getMembershipChangePolicy();
        for (int i=0; i<100; i++) {
            ArrayList<Address> aa = new ArrayList<>(all);
            Collections.shuffle(aa);
            int k = random.nextInt(n-1);
            ArrayList<Address> a1 = new ArrayList<>(aa.subList(0, k+1));
            ArrayList<Address> a2 = new ArrayList<>(aa.subList(k+1, n));
            Collection<Collection<Address>> subviews = new ArrayList<>(2);
            subviews.add(a1);
            subviews.add(a2);
            Set<Address> merged = new HashSet<>(mp.getNewMembership(subviews));
            if (merged.size() != n) {
                return "Found problem:\nV1: ["+ a1 +"]\nV2: ["+ a2 +"]\nMerged: ["+ merged +"]";
            }
        }
        return "No problems found";
    }
    
    private void logView() {
        log(channel.getViewAsString(), Level.OFF);
    }
    
    private String view() {
        try {
            return channel.getViewAsString();
        } catch (Exception x) {
            return "Error: "+ x;
        }
    }
    
    private void probe(String text) {
        String[] ss = text.split("\\s");
        log(Arrays.toString(ss), Level.FINE);
        try {
            Probe.main(ss);
        } catch (Exception x) {
            log(x, Level.WARNING);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void log(Object what, Level level) {
        JGroupsBusMessagingLayer.getLogger().severe(String.valueOf(what));
    }
    
    private String processCommand(String s) {
        try {
            String[] ss = s.split("\\s+");
            if (ss.length == 0) return null;
            switch (ss[0]) {
                case "setName":
                    return setName(ss[1]);
                case "setRank":
                    return setRank(Integer.parseInt(ss[1]));
                case "setConfig":
                    return setConfig(ss[1]);
                case "loadProp":
                    return loadProperties(ss[1]);
                case "setProp":
                    return setProperty(ss[1], ss[2]);
                case "setLevel":
                    switch (ss.length) {
                        case 2:
                            return setLogLevel(ss[1], null);
                        case 3:
                            return setLogLevel(ss[1], ss[2]);
                        default:
                            return "Wrong number of arguments, should be (level, protocol)";
                    }
                case "create":
                    return createChannel();
                case "connect":
                    return connectChannel(ss.length == 1 ? null : ss[1]);
                case "disconnect":
                    return disconnectChannel();
                case "view":
                    return view();
                case "send":
                    return sendMessage();
                case "invoke":
                    return invokeOnChannel(ss[1]);
                case "testMember":
                    return testMembership();
                default:
                    return "   Commands:\n"+
                            "setName name\n"+
                            "setRank int\n"+
                            "setConfig location\n"+
                            "loadProp file\n"+
                            "setProp key value\n"+
                            "setLevel level protocol (fatal, error, warn, info, debug, trace (capitalization not relevant))\n"+
                            "create\n"+
                            "connect clusterName\n"+
                            "disconnect\n"+
                            "view\n"+
                            "send\n"+
                            "invoke method\n"+
                            "testMember"
                            ;
            }
        } catch (Exception x) {
            return "Error: "+ x;
        }
    }

    
// -- Receiver class : ---------------------------------------------------------
    
    private class MessageReceiver extends ReceiverAdapter {

        @Override
        public void unblock() {
            log("unblock()", Level.FINEST);
        }

        @Override
        public void block() {
            log("block()", Level.FINEST);
        }

        @Override
        public void suspect(Address mbr) {
            log("suspect() "+ mbr, Level.FINEST);
        }

        @Override
        public void viewAccepted(View view) {
            log("viewAccepted() "+ view, Level.FINEST);
        }

        @Override
        public void receive(Message msg) {
            log("receive() "+ msg.getSrc(), Level.FINEST);
        }
        
    }
    
    
// -- Running : ----------------------------------------------------------------
    
    static public void main(String... args)  throws Exception {
        
        if (args.length == 0) {

//            JOptionPane.showMessageDialog(null, "New node "+ Arrays.deepToString(args)); // uncomment to enable debugger attachement

            Tester node = new Tester(true);
            node.start();

            node = new Tester(true);
            node.start();

            node = new Tester(true);
            node.start();
        
        } else {
            
            console(args);
            
        }
        
        System.exit(0);
    }
//    
//    static public void console(String... args)  throws Exception {
//        Console console = System.console();
//        System.out.println(" console = "+ console);
//        Tester tester = new Tester();
//        String command;
//        while (!"exit".equalsIgnoreCase(command = console.readLine("> "))) {
//            String response = tester.processCommand(command);
//            if (response != null) {
//                System.out.println(response);
//            }
//        }
//    }

    
    static public void console(String... args)  throws Exception {
        JGroupsBusMessagingLayer.getLogger().severe("Starting JGroups tester");
        BufferedReader reader = new BufferedReader(new InputStreamReader(System.in)); 
        Tester tester = new Tester();
        String command;
        while (!"exit".equalsIgnoreCase(command = reader.readLine())) {
            String response = tester.processCommand(command);
            if (response != null) {
                JGroupsBusMessagingLayer.getLogger().severe(response);
            }
        }
    }
    
}
