package org.lsst.ccs.messaging.jgroups;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.jgroups.Address;
import org.jgroups.JChannel;
import org.jgroups.MergeView;
import org.jgroups.Message;
import org.jgroups.ReceiverAdapter;
import org.jgroups.View;
import org.jgroups.conf.ProtocolConfiguration;
import org.jgroups.conf.XmlConfigurator;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.messaging.BusMessageForwarder;
import org.lsst.ccs.messaging.BusMessagingLayer;
import org.lsst.ccs.messaging.DuplicateAgentNameException;
import org.lsst.ccs.messaging.MessagingAccessLayer;
import org.lsst.ccs.messaging.MessagingAccessLayer.BusAccess;
import org.lsst.ccs.messaging.MessagingLayer;
import org.lsst.ccs.messaging.TransportStateException;
import org.lsst.ccs.messaging.util.AbstractDispatcher;
import org.lsst.ccs.messaging.util.Dispatcher;
import org.lsst.ccs.messaging.ClusterMembershipListener;
import org.lsst.ccs.messaging.HasClusterMembershipNotifications;

/**
 * JGroups based implementation of {@link MessagingLayer}.
 */
public class JGroupsBusMessagingLayer implements BusMessagingLayer, HasClusterMembershipNotifications {

    public static final String DEFAULT = "default";
    public static final String DEFAULT_UDP_PROTOCOL = "jgroups:udp_ccs";
    public static final String DEFAULT_DISPATCHER = "org.lsst.ccs.messaging.util.MultiQueueDispatcher";
    public static final String DEFAULT_DISPATCHER_PROP = 
            "inThreads=1,1,8" +"&"+
            
            "duration/in/LOG/START=300,2000" +"&"+
            "duration/in/STATUS/START=300,2000" +"&"+
            "duration/in/COMMAND/START=300,2000" +"&"+
            "duration/in/LOG/WAIT=500,2000" +"&"+
            "duration/in/STATUS/WAIT=500,2000" +"&"+
            "duration/in/COMMAND/WAIT=500,2000" +"&"+
            "duration/in/LOG/RUN=1000, 2000" +"&"+
            "duration/in/STATUS/RUN=3000,5000" +"&"+
            "duration/in/COMMAND/RUN=1000,2000" +"&"+
            "duration/in/LOG/SUBMIT=300,500" +"&"+
            "duration/in/STATUS/SUBMIT=300,500" +"&"+
            "duration/in/COMMAND/SUBMIT=300,500" +"&"+
            
            "duration/out/LOG/START=300,2000" +"&"+
            "duration/out/STATUS/START=300,2000" +"&"+
            "duration/out/COMMAND/START=300,2000" +"&"+
            "duration/out/LOG/WAIT=500,2000" +"&"+
            "duration/out/STATUS/WAIT=500,2000" +"&"+
            "duration/out/COMMAND/WAIT=500,2000" +"&"+
            "duration/out/LOG/RUN=500, 2000" +"&"+
            "duration/out/STATUS/RUN=500,2000" +"&"+
            "duration/out/COMMAND/RUN=500,2000" +"&"+
            "duration/out/LOG/SUBMIT=300,500" +"&"+
            "duration/out/STATUS/SUBMIT=300,500" +"&"+
            "duration/out/COMMAND/SUBMIT=300,500" +"&"+
            
            "queue/in/LOG=100,500" +"&"+
            "queue/in/STATUS=100,1000" +"&"+
            "queue/in/COMMAND=100,500" +"&"+
            "queue/out/LOG=100,500" +"&"+
            "queue/out/STATUS=100,500" +"&"+
            "queue/out/COMMAND=100,500";
    
    private final Properties jgroupsProperties;
    private String xmlConfigurationFile = "udp_ccs.xml";
    private static final String propertyKeyForJGroups = "org.lsst.ccs.jgroups.";
    private static final String propertyKeyForAllBuses = propertyKeyForJGroups + "ALL.";
    private static final Logger logger = Logger.getLogger("org.lsst.ccs.messaging.jgroups");
        
    private volatile String agentName;
    private final EnumMap<Bus,CCSBusHandler> busHandlers = new EnumMap<>(Bus.class);
    private final List<ClusterMembershipListener> clusterMembershipListeners = new CopyOnWriteArrayList<>();
    private final Dispatcher dispatcher;
    
// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructor.
     * 
     * @param protocolString jgroups*:[URL]:[message dispatcher configuration]
     */
    JGroupsBusMessagingLayer(String protocolString) {
        String[] protocolInfos = protocolString.split(":");
        String rawResource = protocolInfos[1];
        if (rawResource.length() > 0) {
            xmlConfigurationFile = String.format("/%s.xml", rawResource);
        }
        
        // The properties file corresponding to the XML file
        
        String propertiesConfigurationFile = xmlConfigurationFile.replace(".xml", ".properties");
        jgroupsProperties = BootstrapResourceUtils.getBootstrapProperties(propertiesConfigurationFile, this.getClass());
        String sysPref = "system.property.";
        for (String key : jgroupsProperties.stringPropertyNames()) {
            if (key.startsWith(sysPref)) {
                String shortKey = key.substring(sysPref.length());
                if (System.getProperty(shortKey) == null) {
                    System.setProperty(shortKey, jgroupsProperties.getProperty(key));
                }
            }
        }
        
        // Create Dispatcher
        
        String dispatcherClass, dispatcherConfig;
        if (protocolInfos.length < 3 || protocolInfos[2].trim().isEmpty()) {
            dispatcherClass = DEFAULT_DISPATCHER;
            dispatcherConfig = "";
        } else {
            String[] mpsPar = protocolInfos[2].split("\\?");
            dispatcherClass = mpsPar[0].trim();
            if (dispatcherClass.equalsIgnoreCase(DEFAULT)) dispatcherClass = DEFAULT_DISPATCHER;
            if (mpsPar.length > 1) {
                dispatcherConfig = mpsPar[1].trim();
                if (dispatcherConfig.equalsIgnoreCase(DEFAULT)) dispatcherConfig = DEFAULT_DISPATCHER_PROP;
            } else {
                dispatcherConfig = "";
            }
            
        }
        try {
            if (logger.isLoggable(Level.FINE)) {
                StringBuilder sb = new StringBuilder("Message Dispatcher: ");
                sb.append(dispatcherClass).append("\n");
                if (!dispatcherConfig.isEmpty()) {
                    sb.append("Parameters:\n");
                    for (String s : dispatcherConfig.split("\\&")) {
                        sb.append(s).append("\n");
                    }
                }
                logger.fine(sb.toString());
            }
            Class<?> clazz = getClass().getClassLoader().loadClass(dispatcherClass);
            dispatcher = (Dispatcher) clazz.getConstructor(String.class).newInstance(dispatcherConfig);
        } catch (ClassNotFoundException | NoSuchMethodException | InstantiationException | IllegalAccessException | InvocationTargetException x) {
            logger.log(Level.SEVERE, "Failed to initialize message Dispatcher", x);
            throw new RuntimeException(x);
        }
    }

    /**
     * Creates an XmlConfiguration object for a given bus.
     * This object is used to initialize a JChannel. The XmlConfiguration object is created from an xml file that is loaded from
     * the bootstrap environment. The properties object is then scanned to find relevant values to modify XmlConfiguration object.
     *
     * The property keys are of the form
     * org.lsst.ccs.jgroups.[BusIdentifier].[ProtocolName].[ProtocolPropertyName]=[ProtocolPropertyValue]
     *
     * Where: - BusIdentifier can be either COMMAND, LOG, STATUS if the property
     * applied to an individual bus or ALL if it applies to all the buses. -
     * ProtocolName is a valid JGroups protocol name as specified at
     * http://www.jgroups.org/manual/html/protlist.html - ProtocolPropertyName
     * is a property for the given protocol - ProtocolPropertyValue is the value
     * to be given to it
     *
     * If the ProtocolName does not exist in the original xml file, a new
     * ProtocolConfiguration object will be added to the XmlConfiguration.
     *
     * @param xmlConfigurationFile Name of the xml file to be loaded.
     * @param properties Properties object to be used to overwrite the xml file.
     * @param bus Bus for which the properties are scanned.
     * @return XmlConfigurator object to be used to create the JChannel.
     */
    static XmlConfigurator createXmlConfiguratorForBus(String xmlConfigurationFile, Properties properties, String bus) {

        InputStream xmlConfigurationFileInputStream = BootstrapResourceUtils.getBootstrapResource(xmlConfigurationFile, JGroupsBusMessagingLayer.class);
        if (xmlConfigurationFileInputStream == null) {
            throw new IllegalArgumentException(xmlConfigurationFile +" not found as a resource");
        }

        XmlConfigurator configurator = null;
        try {
            configurator = XmlConfigurator.getInstance(xmlConfigurationFileInputStream);
        } catch (IOException ioe) {
            throw new RuntimeException("Could not create JGroups xml configuration object for "+ xmlConfigurationFile, ioe);
        }

        List<ProtocolConfiguration> protocolConfigurationList = configurator.getProtocolStack();

        // First look for properties that applies to all BUSES, then for bus
        // specific properties.
        String[] propertyKeys = new String[]{propertyKeyForAllBuses, propertyKeyForJGroups + bus.toUpperCase() + "."};

        for (String propertyKey : propertyKeys) {
            // When looping over all the keys of a property object we have to
            // keep in mind that the
            // properties might be chained. The following is a utility method to
            // get all keys in a
            // Properties object, including its parents. See
            // https://jira.slac.stanford.edu/browse/LSSTCCS-227
            Set<Object> keys = BootstrapResourceUtils.getAllKeysInProperties(properties);
            for (Object key : keys) {
                String property = (String) key;
                if (property.startsWith(propertyKey)) {

                    // Remove the property key so that only
                    // PROTOCOL.propertyName is left
                    String shortProperty = property.replace(propertyKey, "");
                    int divider = shortProperty.lastIndexOf(".");
                    String protocolPropertyName = shortProperty.substring(divider + 1);
                    String protocolName = shortProperty.substring(0, divider);
                    String protocolPropertyValue = properties.getProperty(property);

                    boolean found = false;
                    for (ProtocolConfiguration protocolConfiguration : protocolConfigurationList) {
                        if (protocolConfiguration.getProtocolName().equals(protocolName)) {
                            protocolConfiguration.getProperties().put(protocolPropertyName, protocolPropertyValue);
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        HashMap<String, String> hash = new HashMap<>();
                        if (!("add".equalsIgnoreCase(protocolPropertyName) && "true".equalsIgnoreCase(protocolPropertyValue))) {
                            hash.put(protocolPropertyName, protocolPropertyValue);
                        }
                        ProtocolConfiguration newProtocol = new ProtocolConfiguration(protocolName, hash);
                        protocolConfigurationList.add(newProtocol);
                    }
                }
            }
        }
        return configurator;
    }

    @Override
    public void close() throws IOException {
        dispatcher.shutdown();
        for (CCSBusHandler bh : busHandlers.values()) {
            bh.close();
        }
    }


// -- Implementing MessagingLayer and HasClusterDisconnectionNotifications : ---

    private class BusHandler extends ReceiverAdapter {

        private final JChannel channel;
        private final BusAccess accessLayer;
        private volatile View lastView;

        BusHandler(BusAccess accessLayer, JChannel chan) {
            this.accessLayer = accessLayer;
            channel = chan;
            lastView = new View(channel.getAddress(),0,Collections.singletonList(channel.getAddress()));
        }

        @Override
        public void receive(Message message) {
            String agent = message.getSrc().toString();
            long time = System.currentTimeMillis();
            Dispatcher.Task task;
            try {
                BusMessage busMessage = message.getObject();
                if (busMessage == null) {
                    throw new RuntimeException("Null payload! This could be due to inconsistent communication protocols.");
                }
                task = new AbstractDispatcher.Task(busMessage, time) {
                    @Override
                    public void run() {
                        accessLayer.processBusMessage(busMessage);
                    }
                };
            } catch (RuntimeException x) {
                task = new AbstractDispatcher.Task(null, time) {
                    @Override
                    public void run() {
                        accessLayer.processClusterDeserializationError(agent, x);
                    }
                };
            }
            dispatcher.in(task, getBus(), agent);
        }

        @Override
        public void viewAccepted(View view) {
            logger.fine(accessLayer.getBus() +" view accepted : " + view.getMembers());

            boolean isMerge =  view instanceof MergeView;
            long deltaId = Math.abs(view.getViewId().getId() - lastView.getViewId().getId());            
            //If it's a MergeView and the deltaId is greater than one, then we are dealing
            //either with a hung process rejoining or we have a cluster merge from the side
            //of the cluster without the coordinator.
            //Either way we remove all the existing members in the view to force a new connection
            //in the AgentPresenceManager; this is to force the updating of the in-memory data
            boolean needsClusterReset = isMerge && deltaId > 1;
            
            List<Address> leftMembers = View.leftMembers(lastView, view);
            List<Address> joinedMembers = View.newMembers(lastView, view);
            if ( getBus() == Bus.STATUS ) {
                if ( needsClusterReset ) {
                    leftMembers.addAll(lastView.getMembers());
                    leftMembers.remove(channel.getAddress());
                    joinedMembers.addAll(leftMembers);
                    logger.fine("Cluster Reset for agent"+channel.getAddressAsString()+" new joined members : " + joinedMembers);
                }
            }
            
            lastView = view;
            int nConnected = joinedMembers.size();
            int nDisconnected = leftMembers.size();
            
            if (getBus() == Bus.STATUS) {
                if ( nDisconnected > 0 ) {
                    ArrayList<String> disconnectedAgents = new ArrayList<>(nDisconnected);
                    for (Address address : leftMembers) {
                        disconnectedAgents.add(address.toString());
                    }
                    logger.fine("Firing notification of cluster disconnection: ====== " + disconnectedAgents);
                    dispatcher.in(() -> {
                        for (ClusterMembershipListener l : clusterMembershipListeners) {
                            try {
                                l.membersLeft(disconnectedAgents);
                            } catch (Exception x) {
                                logger.log(Level.WARNING, "Exception while notifying listeners of agent disconnection", x);
                            }
                        }
                    }, getBus(), disconnectedAgents.toArray(new String[0]));
                }
                if ( nConnected > 0 ) {
                    ArrayList<String> connectedAgents = new ArrayList<>(nConnected);
                    for (Address address : joinedMembers) {
                        connectedAgents.add(address.toString());
                    }

                    logger.fine("Firing notification of cluster connection: ====== " + connectedAgents);
                    dispatcher.in(() -> {
                        for (ClusterMembershipListener l : clusterMembershipListeners) {
                            try {
                                l.membersJoined(connectedAgents);
                            } catch (Exception x) {
                                logger.log(Level.WARNING, "Exception while notifying listeners of agent connection", x);
                            }
                        }
                    }, getBus(), connectedAgents.toArray(new String[0]));
                }
            }
            logger.fine("Updated current view : "+ lastView.getMembers());
        }
        
        void close() {
            channel.close();
        }
        
        void send(BusMessage message) {
            Dispatcher.Task task = new AbstractDispatcher.Task(message, message.getCCSTimeStamp().getUTCInstant().toEpochMilli()) {
                @Override
                public void run() {
                    try {
                        channel.send(null, message);
                    } catch (Exception exc) {   
                        throw new RuntimeException("Failed sending message "+ message.getClass().getCanonicalName() +" with data "+ message.getClassName(),exc);
                    }
                }                
            };
            dispatcher.out(task, getBus(), Dispatcher.Order.NORM);
        }

        Bus getBus() {
            return accessLayer.getBus();
        }
    }

    @Override
    public void connect(MessagingAccessLayer accessLayer) throws DuplicateAgentNameException, IOException {
        logger.fine("### Registering " + accessLayer.getName() +" on the buses");

        agentName = accessLayer.getName();
        if (agentName == null || "".equals(agentName)) {
            throw new IllegalArgumentException("Invalid agent name "+ agentName);
        }
        
        dispatcher.initialize();
        
        for (BusAccess ba : accessLayer.getBusAccesses()) {
            Bus bus = ba.getBus();
            try {
                
                XmlConfigurator configurator = createXmlConfiguratorForBus(xmlConfigurationFile, jgroupsProperties, bus.toString());
                JChannel channel = new JChannel(configurator);
                logger.fine("######################################################\n"
                        + "JGroup Configuration for bus: "+ bus.toString() + "\n"
                        + channel.getProtocolStack().printProtocolSpecAsXML() + "\n"
                        + "######################################################");
                channel.setName(agentName);
                
                String disable_rank = System.getProperty("ccs.jg.disable_rank");
                if (!("true".equalsIgnoreCase(disable_rank))) {
                    channel.addAddressGenerator(new CCSAddress.Generator(accessLayer.getAgentInfo()));
                }
                
                channel.connect(bus.toString());                
                View view = channel.getView();
                if (isDuplicate(view, channel.getAddress())) {
                    channel.close();
                    throw new DuplicateAgentNameException(agentName, "Channel with same name already exists in view");
                }

                String[] propertiesArray = channel.getProperties().split(";");
                String bind_addr = null;
                for (int i = 0; i < propertiesArray.length; i++) {
                    if (propertiesArray[i].startsWith("bind_addr=")) {
                        bind_addr = propertiesArray[i];
                        break;
                    }
                }
                logger.fine("*** new view " + channel.getClusterName() + " " + view + " " + bind_addr);
                
                CCSBusHandler bh = new CCSBusHandler(ba, channel, bus);
                bh.getBusHandler().viewAccepted(view);
                busHandlers.put(bus, bh);
                channel.setReceiver(bh.getBusHandler());
                bh.open();
            } catch (Exception e) {
                throw new IOException(e);
            }
        }
    }
    
    private class CCSBusHandler {
        private final BusHandler busHandler;
        private volatile Boolean isOpen;
        private final Object openLock = new Object();
        private final Bus bus;
        
        CCSBusHandler(BusAccess ba, JChannel ch, Bus bus) {
            busHandler = new BusHandler(ba,ch);
            this.bus = bus;
        }
        
        BusHandler getBusHandler() {
            return busHandler;
        }
        
        void open() {
            synchronized(openLock) {
                isOpen = Boolean.TRUE;
                openLock.notify();
            }
        }        
        
        void close() {
            isOpen = Boolean.FALSE;
            busHandler.close();
        }
        
        void waitForOpen() {
            if (isOpen) {
                return;
            }
            synchronized(openLock) {
                if ( !isOpen ) {
                    throw new TransportStateException("JGroups bus "+bus+" not connected");
                }
                if ( isOpen == null) {
                    try {
                        openLock.wait();
                    } catch (InterruptedException ie) {
                        throw new RuntimeException(ie);
                    }
                }
            }
        }
        
    }

    @Override
    public void disconnect(MessagingAccessLayer accessLayers) {
        try {
            close();
        } catch (IOException x) {
        }
    }

    @Override
    public boolean hasInternalHeartbeat() {
        return true;
    }

    @Override
    public void addClusterMembershipListener(ClusterMembershipListener listener) {
        clusterMembershipListeners.add(listener);
    }

    @Override
    public void removeClusterMembershipListener(ClusterMembershipListener listener) {
        clusterMembershipListeners.remove(listener);
    }

    @Override
    public <T extends BusMessage> void sendMessage(String senderAgent, Bus bus, T message) {
        busHandlers.get(bus).waitForOpen();
        if (senderAgent == null) {
            throw new IllegalArgumentException("no sender agent");
        }
        if (bus == null) {
            throw new IllegalArgumentException("no bus");
        }
        if (message == null) {
            throw new IllegalArgumentException("no message");
        }
        busHandlers.get(bus).getBusHandler().send(message);
    }


// -- Getters : ----------------------------------------------------------------

    @Override
    public Set<String> getRegisteredLocalAgents(Bus... buses) {
        return agentName == null ? Collections.emptySet() : Collections.singleton(agentName);
    }

    private boolean isDuplicate(View view, Address address) {
        for (Address a : view.getMembers()) {
            if (address.equals(a)) {
                continue;
            }
            if (a.toString().equals(address.toString())) {
                return true;
            }
        }
        return false;
    }
    
    static public Logger getLogger() {
        return logger;
    }

    public Dispatcher getDispatcher() {
        return dispatcher;
    }
    

// -- Dummy implementations for useless BusMessagingLayer : --------------------
    
    @Override
    public void register(String agentName, Bus... buses) throws IOException, DuplicateAgentNameException {
    }

    @Override
    public void addMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        throw new UnsupportedOperationException("BusMessagingLayer called");
    }

    @Override
    public void removeMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        throw new UnsupportedOperationException("BusMessagingLayer called");
    }

    @Override
    public void closeFor(String agentName, Bus... buses) {
        try {
            close();
        } catch (IOException x) {
        }
    }
    
}
