package org.lsst.ccs.messaging.jgroups.tester;

import java.awt.Dimension;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.logging.Level;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JComboBox;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.SwingUtilities;
import org.jgroups.Address;
import org.jgroups.JChannel;
import org.jgroups.Message;
import org.jgroups.ReceiverAdapter;
import org.jgroups.View;
import org.jgroups.tests.Probe;

/**
 *
 * @author onoprien
 */
public class Node {

// -- Fields : -----------------------------------------------------------------
    
    static private final AtomicInteger ID = new AtomicInteger();
    static private final String CLUSTER_NAME = "DEF_CLUSTER";
    
    private final String name;
    private final boolean showGUI;
    
    private JChannel channel;
    private JFrame frame;
    private int level = Level.INFO.intValue();

// -- Life cycle : -------------------------------------------------------------
    
    public Node() {
        this(Integer.toString(ID.getAndIncrement()), true);
    }
    
    public Node(String name, boolean showGUI) {
        this.name = name;
        this.showGUI = showGUI;
    }
    
    private void start() {
        if (showGUI) {
            SwingUtilities.invokeLater(this::makeGUI);
        }
    }
    
    private void stop() {
        if (frame != null) frame.dispose();
        channel.close();
    }
    
    private void makeGUI() {
        
        frame = new JFrame(name);
        frame.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosed(WindowEvent e) {
                stop();
            }
        });
        Box root = Box.createVerticalBox();
        frame.add(new JScrollPane(root));
        
        // Node settings:
        
        Box box = Box.createHorizontalBox();
        
        box.add(new JLabel("Output level: "));
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        Level[] levels = {Level.SEVERE, Level.WARNING, Level.INFO, Level.FINE, Level.FINER, Level.FINEST};
        JComboBox<Level> combo = new JComboBox<>(levels);
        combo.setSelectedItem(Level.INFO);
        combo.addActionListener(e -> {
            JComboBox<Level> c = (JComboBox<Level>) e.getSource();
            level = ((Level)c.getSelectedItem()).intValue();
        });
        box.add(combo);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        box.add(Box.createHorizontalGlue());
        root.add(box);
        root.add(Box.createRigidArea(new Dimension(5,5)));
        
        // Create and connect channel
        
        box = Box.createHorizontalBox();
        
        JButton b = new JButton("Create channel");
        b.addActionListener(e -> {
            ((JButton)e.getSource()).setEnabled(false);
            createChannel();
        });
        box.add(b);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        b = new JButton("Connect");
        b.addActionListener(e -> {
            if (e.getActionCommand().equals("Connect")) {
                ((JButton)e.getSource()).setText("Disconnect");
                connectChannel();
            } else {
                ((JButton)e.getSource()).setText("Connect");
                disconnectChannel();
            }
        });
        box.add(b);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        box.add(Box.createHorizontalGlue());
        root.add(box);
        root.add(Box.createRigidArea(new Dimension(5,5)));
        
        // Send messages and print view
        
        box = Box.createHorizontalBox();
        
        b = new JButton("Send");
        b.addActionListener(e -> sendMessage());
        box.add(b);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        b = new JButton("View");
        b.addActionListener(e -> logView());
        box.add(b);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        box.add(Box.createHorizontalGlue());
        root.add(box);
        root.add(Box.createRigidArea(new Dimension(5,5)));
        
        // Probe :
        
        box = Box.createHorizontalBox();
        
        box.add(new JLabel("Probe"));
        box.add(Box.createRigidArea(new Dimension(5,5)));
        JTextField f = new JTextField();
        f.addActionListener(e -> probe(e.getActionCommand()));
        box.add(f);
        box.add(Box.createRigidArea(new Dimension(5,5)));
        
        box.add(Box.createHorizontalGlue());
        root.add(box);
        root.add(Box.createRigidArea(new Dimension(5,5)));
        
        // Finish
        
        root.add(Box.createVerticalGlue());
        frame.pack();
        frame.setVisible(true);
    }
    
// -- Commands : ---------------------------------------------------------------
    
    private void createChannel() {
        try {
            channel = new JChannel(System.getProperty("user.dir") + "/src/main/java/org/lsst/ccs/messaging/jgroups/tester/udp_0.xml");
            channel.setName(name);
            String text = "Channel configuration:\n"+ channel.getProperties();
            log(text, Level.FINE);
            channel.setReceiver(new MessageReceiver());
        } catch (Exception x) {
            throw new RuntimeException("Error creating channel", x);
        }
    }
    
    private void connectChannel() {
        try {
            channel.connect(CLUSTER_NAME);
        } catch (Exception x) {
            throw new RuntimeException("Error connecting channel", x);
        }
    }
    
    private void disconnectChannel() {
        try {
            channel.disconnect();
        } catch (Exception x) {
            throw new RuntimeException("Error disconnecting channel", x);
        }
    }
    
    private void sendMessage() {
        try {
            channel.send(null, "default payload");
        } catch (Exception x) {
            throw new RuntimeException("Error sending message", x);
        }
    }
    
    private void logView() {
        log(channel.getViewAsString(), Level.OFF);
    }
    
    private void probe(String text) {
        String[] ss = text.split("\\s");
        log(Arrays.toString(ss), Level.FINE);
        try {
            Probe.main(ss);
        } catch (Exception x) {
            log(x, Level.WARNING);
        }
    }
    
    
// -- Local methods : ----------------------------------------------------------
    
    private void log(Object what, Level level) {
        if (level.intValue() >= this.level) {
            System.out.println("");
            System.out.println("From " + name + ": ");
            System.out.println(what);
        }
    }
    
    
// -- Receiver class : ---------------------------------------------------------
    
    private class MessageReceiver extends ReceiverAdapter {

        @Override
        public void unblock() {
            log("unblock()", Level.FINEST);
        }

        @Override
        public void block() {
            log("block()", Level.FINEST);
        }

        @Override
        public void suspect(Address mbr) {
            log("suspect() "+ mbr, Level.FINEST);
        }

        @Override
        public void viewAccepted(View view) {
            log("viewAccepted() "+ view, Level.FINEST);
        }

        @Override
        public void setState(InputStream input) throws Exception {
            super.setState(input);
        }

        @Override
        public void getState(OutputStream output) throws Exception {
            super.getState(output);
        }

        @Override
        public void receive(Message msg) {
            log("receive() "+ msg.getSrc(), Level.FINEST);
        }
        
    }
    
    
// -- Running : ----------------------------------------------------------------
    
    static public void main(String... args)  throws Exception {

//      JOptionPane.showMessageDialog(null, "New node "+ Arrays.deepToString(args)); // uncomment to enable debugger attachement

        Node node = new Node();
        node.start();
        
        node = new Node();
        node.start();
        
        node = new Node();
        node.start();

    }
    
}
