package org.lsst.ccs.messaging.jgroups;

import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.messaging.BusMessagingLayer;
import org.lsst.ccs.messaging.NetworkUtilities;
import org.lsst.ccs.messaging.TransportStateException;
import org.lsst.ccs.messaging.BusMessageForwarder;
import org.jgroups.*;
import org.jgroups.conf.ProtocolConfiguration;
import org.jgroups.conf.XmlConfigurator;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.logging.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;

import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.messaging.ClusterDisconnectionsListener;
import org.lsst.ccs.messaging.DuplicateAgentNameException;
import org.lsst.ccs.messaging.HasClusterDisconnectionNotifications;
import org.lsst.ccs.messaging.MessagingAccessLayer;
import org.lsst.ccs.messaging.MessagingAccessLayer.BusAccess;
import org.lsst.ccs.messaging.MessagingLayer;
import org.lsst.ccs.messaging.util.Dispatcher;
import org.lsst.ccs.utilities.taitime.CCSTimeStamp;

/**
 * JGroups based implementation of {@link MessagingLayer}.
 */
public class JGroupsBusMessagingLayer implements BusMessagingLayer, HasClusterDisconnectionNotifications {

    public static final String DEFAULT_UDP_PROTOCOL = "jgroups:udp_ccs";
    public static final String DEFAULT_DISPATCHER = "org.lsst.ccs.messaging.util.LegacyDispatcher";
    private final Properties jgroupsProperties;
    private String xmlConfigurationFile = "udp_ccs.xml";
    private static final String propertyKeyForJGroups = "org.lsst.ccs.jgroups.";
    private static final String propertyKeyForAllBuses = propertyKeyForJGroups + "ALL.";
    private static final Logger logger = Logger.getLogger("org.lsst.ccs.messaging.jgroups");
        
    private volatile boolean closed;
    private volatile String agentName;
    private final EnumMap<Bus,BusHandler> busHandlers = new EnumMap<>(Bus.class);
    private final List<ClusterDisconnectionsListener> disconnectionListeners = new CopyOnWriteArrayList<>();
    private final Dispatcher dispatcher;
    
// -- Life cycle : -------------------------------------------------------------
    
    /**
     * Constructor.
     * 
     * @param protocolString jgroups*:[URL]:[message dispatcher configuration]
           URL: (optional, default "udp_ccs") - XML configuration file for JGroups.
           dispatcher configuration: (optional, default "org.lsst.ccs.messaging.util.LegacyDispatcher")
           [Dispatcher implementation class name]?[par1]&...&[parN]
     */
    JGroupsBusMessagingLayer(String protocolString) {
        String[] protocolInfos = protocolString.split(":");
        String rawResource = protocolInfos[1];
        if (rawResource.length() > 0) {
            xmlConfigurationFile = String.format("/%s.xml", rawResource);
        }
        
        // The properties file corresponding to the XML file
        
        String propertiesConfigurationFile = xmlConfigurationFile.replace(".xml", ".properties");
        jgroupsProperties = BootstrapResourceUtils.getBootstrapProperties(propertiesConfigurationFile, this.getClass());

        //Add the default bind address ONLY if we are using the udp protocol or else we will have protocol contamination.
        
        if ( xmlConfigurationFile.toLowerCase().contains("udp") ) {
            if ((jgroupsProperties.getProperty("org.lsst.ccs.jgroups.ALL.UDP.bind_interface_str") == null)
                    && (jgroupsProperties.getProperty("org.lsst.ccs.jgroups.ALL.UDP.bind_addr") == null)) {
                String en = NetworkUtilities.getMainInterfaceName();
                jgroupsProperties.setProperty("org.lsst.ccs.jgroups.ALL.UDP.bind_interface_str", en);
            }
        }
        
        // Create Dispatcher
        
        String mps = (protocolInfos.length > 2 && !protocolInfos[2].trim().isEmpty()) ?
                     protocolInfos[2] : DEFAULT_DISPATCHER;
        String[] mpsPar = mps.split("\\?");
        String[] constructorParameters = mpsPar.length > 1 ? mpsPar[1].split("\\&") : new String[0];
        try {
            Class<?> clazz = getClass().getClassLoader().loadClass(mpsPar[0]);
            dispatcher = (Dispatcher) clazz.getConstructor(String[].class).newInstance((Object)constructorParameters);
            StringBuilder sb = new StringBuilder();
            sb.append("Message Dispatcher: ").append(dispatcher.getClass().getSimpleName()).append("\n");
            if (constructorParameters.length > 0) {
                sb.append("Parameters:\n");
                for (String s : constructorParameters) {
                    sb.append(s);
                }
            }
            logger.info(sb.toString());
        } catch (ClassNotFoundException | NoSuchMethodException | InstantiationException | IllegalAccessException | InvocationTargetException x) {
            logger.error("Failed to initialize message Dispatcher", x);
            throw new RuntimeException(x);
        }
    }

    /**
     * Creates an XmlConfiguration object for a given bus.
     * This object is used to initialize a JChannel. The XmlConfiguration object is created from an xml file that is loaded from
     * the bootstrap environment. The properties object is then scanned to find relevant values to modify XmlConfiguration object.
     *
     * The property keys are of the form
     * org.lsst.ccs.jgroups.[BusIdentifier].[ProtocolName].[ProtocolPropertyName]=[ProtocolPropertyValue]
     *
     * Where: - BusIdentifier can be either COMMAND, LOG, STATUS if the property
     * applied to an individual bus or ALL if it applies to all the buses. -
     * ProtocolName is a valid JGroups protocol name as specified at
     * http://www.jgroups.org/manual/html/protlist.html - ProtocolPropertyName
     * is a property for the given protocol - ProtocolPropertyValue is the value
     * to be given to it
     *
     * If the ProtocolName does not exist in the original xml file, a new
     * ProtocolConfiguration object will be added to the XmlConfiguration.
     *
     * @param xmlConfigurationFile Name of the xml file to be loaded.
     * @param properties Properties object to be used to overwrite the xml file.
     * @param bus Bus for which the properties are scanned.
     * @return XmlConfigurator object to be used to create the JChannel.
     */
    static XmlConfigurator createXmlConfiguratorForBus(String xmlConfigurationFile, Properties properties, String bus) {

        InputStream xmlConfigurationFileInputStream = BootstrapResourceUtils.getBootstrapResource(xmlConfigurationFile, JGroupsBusMessagingLayer.class);
        if (xmlConfigurationFileInputStream == null) {
            throw new IllegalArgumentException(xmlConfigurationFile +" not found as a resource");
        }

        XmlConfigurator configurator = null;
        try {
            configurator = XmlConfigurator.getInstance(xmlConfigurationFileInputStream);
        } catch (IOException ioe) {
            throw new RuntimeException("Could not create JGroups xml configuration object for "+ xmlConfigurationFile, ioe);
        }

        List<ProtocolConfiguration> protocolConfigurationList = configurator.getProtocolStack();

        // First look for properties that applies to all BUSES, then for bus
        // specific properties.
        String[] propertyKeys = new String[]{propertyKeyForAllBuses, propertyKeyForJGroups + bus.toUpperCase() + "."};

        for (String propertyKey : propertyKeys) {
            // When looping over all the keys of a property object we have to
            // keep in mind that the
            // properties might be chained. The following is a utility method to
            // get all keys in a
            // Properties object, including its parents. See
            // https://jira.slac.stanford.edu/browse/LSSTCCS-227
            Set<Object> keys = BootstrapResourceUtils.getAllKeysInProperties(properties);
            for (Object key : keys) {
                String property = (String) key;
                if (property.startsWith(propertyKey)) {

                    // Remove the property key so that only
                    // PROTOCOL.propertyName is left
                    String shortProperty = property.replace(propertyKey, "");
                    int divider = shortProperty.lastIndexOf(".");
                    String protocolPropertyName = shortProperty.substring(divider + 1);
                    String protocolName = shortProperty.substring(0, divider);
                    String protocolPropertyValue = properties.getProperty(property);

                    boolean found = false;
                    for (ProtocolConfiguration protocolConfiguration : protocolConfigurationList) {
                        if (protocolConfiguration.getProtocolName().equals(protocolName)) {
                            protocolConfiguration.getProperties().put(protocolPropertyName, protocolPropertyValue);
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        HashMap<String, String> hash = new HashMap<>();
                        hash.put(protocolPropertyName, protocolPropertyValue);
                        ProtocolConfiguration newProtocol = new ProtocolConfiguration(protocolName, hash);
                        protocolConfigurationList.add(newProtocol);
                    }
                }
            }
        }
        return configurator;
    }

    @Override
    public void close() throws IOException {
        dispatcher.shutdown();
        closed = true;
        for (BusHandler bh : busHandlers.values()) {
            bh.close();
        }
    }


// -- Implementing MessagingLayer and HasClusterDisconnectionNotifications : ---

    private class BusHandler extends ReceiverAdapter {

        private final JChannel channel;
        private final BusAccess accessLayer;
        private View lastView;

        BusHandler(BusAccess accessLayer, JChannel chan, View view) {
            this.accessLayer = accessLayer;
            channel = chan;
            lastView = view;
        }

        @Override
        public void receive(Message message) {
            Dispatcher.Task task = new Dispatcher.Task() {
                private volatile CCSTimeStamp toQueue, fromQueue;
                @Override
                public void stageEnded(Dispatcher.Stage... stages) {
                    for (Dispatcher.Stage stage : stages) {
                        switch (stage) {
                            case START:
                                toQueue = CCSTimeStamp.currentTime();
                                break;
                            case WAIT:
                                fromQueue = CCSTimeStamp.currentTime();
                                break;
                        }
                    }
                }
                @Override
                public void run() {
                    BusMessage busMessInit;
                    try {
                        busMessInit = (BusMessage) message.getObject();
                        busMessInit.setIncomingQueueInTimeStamp(toQueue);
                        busMessInit.setIncomingQueueOutTimeStamp(fromQueue);
                    } catch (final RuntimeException exc) {
                        accessLayer.processClusterDeserializationError(message.getSrc().toString(), exc);
                        return;
                    }
                    final BusMessage busMess = busMessInit;
                    accessLayer.processBusMessage(busMess);
                }                
            };
            dispatcher.in(task, getBus(), message.getSrc().toString());
        }

        @Override
        public void viewAccepted(View view) {
            logger.fine("view accepted : " + view.getMembers());

            // diffArray[0] joining members
            // diffArray[1] leaving members
            Address[][] diffArray = View.diff(lastView, view);

            lastView = view;
            if (getBus() == Bus.STATUS) {
                ArrayList<String> disconnectedAgents = new ArrayList();
                for (Address address : diffArray[1]) {
                    disconnectedAgents.add(address.toString());
                }
                logger.fine("Firing notification of cluster disconnection: ====== " + disconnectedAgents);
                dispatcher.in(
                        () -> {
                            for (ClusterDisconnectionsListener l : disconnectionListeners) {
                                try {
                                    l.membersLeft(disconnectedAgents);
                                } catch (Exception x) {
                                    logger.warn("Exception while notifying listeners of agent disconnection", x);
                                }
                            }
                        },
                        getBus(), disconnectedAgents.toArray(new String[0]));

            }
            logger.fine("Updated current view : "+ lastView.getMembers());
        }
        
        void close() {
            channel.close();
        }
        
        void send(BusMessage message) {
            Dispatcher.Task task = new Dispatcher.Task() {
                @Override
                public void stageEnded(Dispatcher.Stage... stages) {
                    for (Dispatcher.Stage stage : stages) {
                        switch (stage) {
                            case START:
                                message.setOutgoingQueueInTimeStamp(CCSTimeStamp.currentTime());
                                break;
                            case WAIT:
                                message.setOutgoingQueueOutTimeStamp(CCSTimeStamp.currentTime());
                                break;
                        }
                    }
                }
                @Override
                public void run() {
                    try {
                        channel.send(null, message);
                    } catch (Exception exc) {
                        throw new RuntimeException(exc);
                    }
                }                
            };
            dispatcher.out(task, getBus(), Dispatcher.Order.NORM);
        }

        Bus getBus() {
            return accessLayer.getBus();
        }
    }

    @Override
    public void connect(MessagingAccessLayer accessLayer) throws DuplicateAgentNameException, IOException {
        logger.fine("### Registering " + accessLayer.getName() +" on the buses");

        agentName = accessLayer.getName();
        if (agentName == null || "".equals(agentName)) {
            throw new IllegalArgumentException("Invalid agent name "+ agentName);
        }
        
        dispatcher.initialize();

        for (BusAccess ba : accessLayer.getBusAccesses()) {
            Bus bus = ba.getBus();
            try {
                XmlConfigurator configurator = createXmlConfiguratorForBus(xmlConfigurationFile, jgroupsProperties, bus.toString());
                JChannel channel = new JChannel(configurator);
                logger.debug("######################################################\n"
                        + "JGroup Configuration for bus: "+ bus.toString() + "\n"
                        + channel.getProtocolStack().printProtocolSpecAsXML() + "\n"
                        + "######################################################");
                channel.setName(agentName);
                channel.connect(bus.toString());
                View view = channel.getView();
                if (isDuplicate(view, channel.getAddress())) {
                    channel.close();
                    throw new DuplicateAgentNameException(agentName, "Channel with same name already exists in view");
                }

                String[] propertiesArray = channel.getProperties().split(";");
                String bind_addr = null;
                for (int i = 0; i < propertiesArray.length; i++) {
                    if (propertiesArray[i].startsWith("bind_addr=")) {
                        bind_addr = propertiesArray[i];
                        break;
                    }
                }
                logger.debug("*** new view " + channel.getClusterName() + " " + view + " " + bind_addr);
                
                BusHandler bh = new BusHandler(ba, channel, view);
                busHandlers.put(bus, bh);
                channel.setReceiver(bh);
            } catch (Exception e) {
                throw new IOException(e);
            }
        }

        closed = false;
    }

    @Override
    public void disconnect(MessagingAccessLayer accessLayers) {
        try {
            close();
        } catch (IOException x) {
        }
    }

    @Override
    public void addClusterMembershipListener(ClusterDisconnectionsListener listener) {
        disconnectionListeners.add(listener);
    }

    @Override
    public void removeClusterMembershipListener(ClusterDisconnectionsListener listener) {
        disconnectionListeners.remove(listener);
    }

    @Override
    public <T extends BusMessage> void sendMessage(String senderAgent, Bus bus, T message) {
        if (closed) {
            throw new TransportStateException("JGroups layer not connected");
        }
        if (senderAgent == null) {
            throw new IllegalArgumentException("no sender agent");
        }
        if (bus == null) {
            throw new IllegalArgumentException("no bus");
        }
        if (message == null) {
            throw new IllegalArgumentException("no message");
        }
        busHandlers.get(bus).send(message);
    }


// -- Getters : ----------------------------------------------------------------

    @Override
    public Set<String> getRegisteredLocalAgents(Bus... buses) {
        return agentName == null ? Collections.emptySet() : Collections.singleton(agentName);
    }

    private boolean isDuplicate(View view, Address address) {
        for (Address a : view.getMembers()) {
            if (address.equals(a)) {
                continue;
            }
            if (a.toString().equals(address.toString())) {
                return true;
            }
        }
        return false;
    }


// -- Dummy implementations for useless BusMessagingLayer : --------------------
    
    @Override
    public void register(String agentName, Bus... buses) throws IOException, DuplicateAgentNameException {
    }

    @Override
    public void addMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        throw new UnsupportedOperationException("BusMessagingLayer called");
    }

    @Override
    public void removeMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        throw new UnsupportedOperationException("BusMessagingLayer called");
    }

    @Override
    public void closeFor(String agentName, Bus... buses) {
        try {
            close();
        } catch (IOException x) {
        }
    }
    
}
