package org.lsst.ccs.messaging.jgroups;

import org.lsst.ccs.bus.definition.Bus;
import org.lsst.ccs.messaging.BusMessagingLayer;
import org.lsst.ccs.messaging.NetworkUtilities;
import org.lsst.ccs.messaging.TransportStateException;
import org.lsst.ccs.messaging.ProvidesDisconnectionInformation;
import org.lsst.ccs.messaging.BusMessageForwarder;
import org.jgroups.*;
import org.jgroups.conf.ProtocolConfiguration;
import org.jgroups.conf.XmlConfigurator;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.logging.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.util.*;

import org.lsst.ccs.bus.messages.BusMessage;
import org.lsst.ccs.messaging.DuplicateAgentNameException;
import org.lsst.ccs.messaging.MessagingAccessLayer;
import org.lsst.ccs.messaging.MessagingAccessLayer.BusAccess;
import org.lsst.ccs.messaging.MessagingAccessLayer.StatusBusAccess;

/**
 * Each Bus is represented by a JGroups cluster.
 * <p/>
 * Each agent creates a JChannel, setName to name of agent and connect to
 * corresponding Bus/cluster (so each agent may have up to 3 JChannel
 * connected). <BR>
 * Each of these 3 channels have a JGroups Address.
 * <p/>
 * for sending messages to an agent we have to find its Address in the group AND
 * we have to find the addresses of the multiple anonymous agents that will
 * receive a copy
 * <p/>
 * There is a dictionary to match agent name to jGroups Address. for each agent
 * there is an Address except for anonymous agent that can be matched to
 * multiple addresses.
 * <p/>
 * < * p/> <B>ImPortant</B> every Jchannel in the JVM MUST be closed!
 */
public class JGroupsBusMessagingLayer implements BusMessagingLayer,
        ProvidesDisconnectionInformation {

    public static final String DEFAULT_UDP_PROTOCOL = "jgroups:udp_ccs";
    private final Properties jgroupsProperties;
    private String xmlConfigurationFile = "udp_ccs.xml";
    private static final String propertyKeyForJGroups = "org.lsst.ccs.jgroups.";
    private static final String propertyKeyForAllBuses = propertyKeyForJGroups
            + "ALL.";
    private static final Logger logger = Logger
            .getLogger("org.lsst.ccs.messaging.jgroups");

    private final LocalAgentChannels localAgentChannels = new LocalAgentChannels();
    
    // accessed from synchronized code :so no volatile
    // This is needed to figure out which nodes have left the cluster when
    // viewAccepted is invoked. By that time the Channel's view has already been
    // updated.
    private final View[] lastViews = new View[Bus.values().length];
    private volatile boolean closed;

    /**
     * for a local agent the list of its channels and of its adapters
     */
    class LocalAgentChannels {
        String agentName;
        JChannel[] channels = new JChannel[Bus.values().length];
        BusReceiver[] receivers = new BusReceiver[Bus.values().length];
    }

    /**
     * Receiver class for the tunnel. Any message sent on the tunnel, of type
     * TunnelMessage, is sent on the corresponding CCS bus (unless it's coming
     * from this end of the tunnel).
     *
     */
    class TunnelReceiver extends ReceiverAdapter {

        private final String agentName;
        private final JChannel channel;

        TunnelReceiver(JChannel channel, String agentName) {
            this.channel = channel;
            this.agentName = agentName;
        }

        @Override
        public void receive(Message message) {
            // If published by the Tunnel itself it has to be ignored.
            if (message.getSrc().toString().equals(agentName)) {
                return;
            }

            // Otherwise broadcast the message on the corresponding channel
            sendObjectInTunnelMessageOnChannel(message.getObject(), channel);
        }

    }

    /**
     * Utility method to send a TunnelMessage on a Channel. A TunnelMessage is a
     * message that either goes into a Tunnel or comes out of it.
     *
     */
    private void sendObjectInTunnelMessageOnChannel(Object obj, JChannel channel) {
        try {
            TunnelMessage msg = new TunnelMessage(obj);
            channel.send(msg);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * instance of this class act as a receiver of messages then forward to the
     * BusMessageForwarders registered
     */
    class BusReceiver extends ReceiverAdapter {

        JChannel curChan;
        // todo: again a duplicate!
        BusMessagingLayer layer;
        BusAccess accessLayer;
        private final JChannel tunnel;

        BusReceiver(BusAccess accessLayer, JChannel curChan,
                BusMessagingLayer layer, JChannel tunnel) {
            this.accessLayer = accessLayer;
            this.curChan = curChan;
            curChan.setReceiver(this);
            this.layer = layer;
            this.tunnel = tunnel;
        }

        @Override
        public void receive(Message message) {

            BusMessage busMessInit;
            try {
                // if (accessLayer.getForwarderList().isEmpty()) {
                // return;
                // }
                
                Object obj = message.getObject();
                if (obj instanceof DisconnectedAgent) {
                    String agent = ((DisconnectedAgent) obj).getAgentName();
                    ((StatusBusAccess) accessLayer)
                            .processDisconnectionSuspicion(agent);
                    return;
                }

                busMessInit = (BusMessage) message.getObject();
            } catch (final RuntimeException exc) {
                accessLayer.processClusterDeserializationError(message.getSrc().toString());
                logger.warn("cluster deserialization exception", exc);
                return;
            }
            final BusMessage busMess = busMessInit;
            accessLayer.processBusMessage(busMess);
            // If the message is coming from the tunnel there is no need to re-send it 
            // It would cause an infinite loop.
            if (tunnel != null) {
                if (!message.getSrc().toString().equals(tunnel.getAddressAsString())) {
                    sendObjectInTunnelMessageOnChannel(message.getObject(),
                            tunnel);
                }
            }
        }

        private void processDisconnection(Address a) {
            ((StatusBusAccess) accessLayer).processDisconnectionSuspicion(a
                    .toString());
            if (tunnel != null) {
                sendObjectInTunnelMessageOnChannel(
                        new DisconnectedAgent(a.toString()), tunnel);
            }
            logger.fine("removing ====== " + a);

        }

        @Override
        public void viewAccepted(View view) {
            logger.fine("view accepted : " + view.getMembers());
            int index = accessLayer.getBus().ordinal();
            View lastView = lastViews[index];

            // Algorithm using the new API : diffArray[0] are the joined members
            // diffArray[1] are the left members
            Address[][] diffArray = org.jgroups.View.diff(lastView, view);

            lastViews[index] = view;
            if (index == Bus.STATUS.ordinal()) {
                for (Address address : diffArray[1]) {
                    processDisconnection(address);
                }
            }
            logger.fine("Updated current view : "
                    + lastViews[accessLayer.getBus().ordinal()].getMembers());
        }

        @Override
        public void suspect(Address address) {
            logger.fine("disconnection suspicion on " + accessLayer.getBus()
                    + " bus : " + address.toString());
            // get the address in other
            int index = accessLayer.getBus().ordinal();
            if (index == Bus.STATUS.ordinal()) {
                processDisconnection(address);
            }
        }
    }

    /**
     * Constructor
     *
     * @param protocolString
     */
    JGroupsBusMessagingLayer(String protocolString) {
        // index 1 : is for URL
        // index 2 :is for String to be passed to JChannel
        String[] protocolInfos = protocolString.split(":");
        String rawResource = protocolInfos[1];
        if (rawResource.length() > 0) {
            xmlConfigurationFile = String.format("/%s.xml", rawResource);
        }
        // The properties file corresponding to the XML file
        String propertiesConfigurationFile = xmlConfigurationFile.replace(
                ".xml", ".properties");
        jgroupsProperties = BootstrapResourceUtils.getBootstrapProperties(
                propertiesConfigurationFile, this.getClass());

        if ((jgroupsProperties
                .getProperty("org.lsst.ccs.jgroups.ALL.UDP.bind_interface_str") == null)
                && (jgroupsProperties
                .getProperty("org.lsst.ccs.jgroups.ALL.UDP.bind_addr") == null)) {
            String en = NetworkUtilities.getMainInterfaceName();
            jgroupsProperties.setProperty(
                    "org.lsst.ccs.jgroups.ALL.UDP.bind_interface_str", en);
        }

    }

    // /////////////////////////////////////////// METHODS
    @Override
    public void register(String agentName, Bus... buses) throws IOException,
            DuplicateAgentNameException {
    }

    @Override
    public Set<String> getRegisteredLocalAgents(Bus... buses) {

        // If there are no provided buses, it means for all buses.
        if (buses == null || buses.length == 0) {
            buses = Bus.values();
        }
        Set<String> result = new HashSet<>();
        for (Bus bus : buses) {
            BusReceiver receiver = localAgentChannels.receivers[bus.ordinal()];
            if (receiver != null) {
                if (!result.contains(localAgentChannels.agentName)) {
                    result.add(localAgentChannels.agentName);
                }
            }
        }
        return result;
    }

    /**
     * Creates an XmlConfiguration object for a given bus. This object is used
     * to initialize a JChannel. The XmlConfiguration object is created from an
     * xml file that is loaded from the bootstrap environment. The properties
     * object is then scanned to find relevant values to modify XmlConfiguration
     * object.
     *
     * The property keys are of the form
     * org.lsst.ccs.jgroups.[BusIdentifier].[ProtocolName
     * ].[ProtocolPropertyName]=[ProtocolPropertyValue]
     *
     * Where: - BusIdentifier can be either COMMAND, LOG, STATUS if the property
     * applied to an individual bus or ALL if it applies to all the buses. -
     * ProtocolName is a valid JGroups protocol name as specified at
     * http://www.jgroups.org/manual/html/protlist.html - ProtocolPropertyName
     * is a property for the given protocol - ProtocolPropertyValue is the value
     * to be given to it
     *
     * If the ProtocolName does not exist in the original xml file, a new
     * ProtocolConfiguration object will be added to the XmlConfiguration.
     *
     *
     * @param xmlConfigurationFile
     * The name of the xml file to be loaded
     * @param properties
     * The properties object to be used to overwrite the xml file
     * @param bus
     * The bus for which the properties are scanned
     * @return The XmlConfiguration object to be used to create the JChannel
     *
     */
    static XmlConfigurator createXmlConfiguratorForBus(
            String xmlConfigurationFile, Properties properties, String bus) {

        InputStream xmlConfigurationFileInputStream = BootstrapResourceUtils
                .getBootstrapResource(xmlConfigurationFile,
                        JGroupsBusMessagingLayer.class);
        if (xmlConfigurationFileInputStream == null) {
            throw new IllegalArgumentException(xmlConfigurationFile
                    + " not found as a resource");
        }

        XmlConfigurator configurator = null;
        try {
            configurator = XmlConfigurator
                    .getInstance(xmlConfigurationFileInputStream);
        } catch (IOException ioe) {
            throw new RuntimeException(
                    "Could not create JGroups xml configuration object for "
                    + xmlConfigurationFile, ioe);
        }

        List<ProtocolConfiguration> protocolConfigurationList = configurator
                .getProtocolStack();

        // First look for properties that applies to all BUSES, then for bus
        // specific properties.
        String[] propertyKeys = new String[]{propertyKeyForAllBuses,
            propertyKeyForJGroups + bus.toUpperCase() + "."};

        for (String propertyKey : propertyKeys) {
            // When looping over all the keys of a property object we have to
            // keep in mind that the
            // properties might be chained. The following is a utility method to
            // get all keys in a
            // Properties object, including its parents. See
            // https://jira.slac.stanford.edu/browse/LSSTCCS-227
            Set<Object> keys = BootstrapResourceUtils
                    .getAllKeysInProperties(properties);
            for (Object key : keys) {
                String property = (String) key;
                if (property.startsWith(propertyKey)) {

                    // Remove the property key so that only
                    // PROTOCOL.propertyName is left
                    String shortProperty = property.replace(propertyKey, "");
                    int divider = shortProperty.lastIndexOf(".");
                    String protocolPropertyName = shortProperty
                            .substring(divider + 1);
                    String protocolName = shortProperty.substring(0, divider);
                    String protocolPropertyValue = properties
                            .getProperty(property);

                    boolean found = false;
                    for (ProtocolConfiguration protocolConfiguration : protocolConfigurationList) {
                        if (protocolConfiguration.getProtocolName().equals(
                                protocolName)) {
                            protocolConfiguration.getProperties()
                                    .put(protocolPropertyName,
                                            protocolPropertyValue);
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        HashMap<String, String> hash = new HashMap<>();
                        hash.put(protocolPropertyName, protocolPropertyValue);
                        ProtocolConfiguration newProtocol = new ProtocolConfiguration(
                                protocolName, hash);
                        protocolConfigurationList.add(newProtocol);
                    }

                }
            }
        }
        return configurator;
    }

    private boolean isDuplicate(View view, Address address) {
        for (Address a : view.getMembers()) {
            if (address.equals(a)) {
                continue;
            }
            if (a.toString().equals(address.toString())) {
                return true;
            }
        }
        return false;
    }

    @Override
    public void closeFor(String agentName, Bus... buses) {
        if (agentName == null || "".equals(agentName)) {
            throw new IllegalArgumentException(
                    "Agent name cannot be null or empty");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        if (localAgentChannels != null) {
            for (Bus bus : buses) {
                int index = bus.ordinal();
                JChannel channel = localAgentChannels.channels[index];
                if (channel != null) {
                    channel.close();
                }
            }
        }
    }

    @Override
    public void close() throws IOException {
        closed = true;
        for (JChannel channel : localAgentChannels.channels) {
            if (channel != null) {
                channel.close();
            }
        }
    }

    @Override
    public <T extends BusMessage> void sendMessage(String senderAgent, Bus bus,
            T message) {
        if (closed) {
            throw new TransportStateException("closing");
        }
        if (senderAgent == null) {
            throw new IllegalArgumentException("no sender agent");
        }
        if (bus == null) {
            throw new IllegalArgumentException("no bus");
        }
        if (message == null) {
            throw new IllegalArgumentException("no message");
        }
        int index = bus.ordinal();

        // now trying to find the proper Channel
        JChannel channel = localAgentChannels.channels[index];
        if (channel == null) {
            throw new IllegalArgumentException("agent " + senderAgent
                    + " not registered on " + bus);
        }
        try {
            // broadcast
            synchronized (channel) {
                channel.send(null, message);
            }
        } catch (Exception exc) {
            throw new RuntimeException(exc);
        }
    }

    @Override
    public void addMessageListener(String agentName,
            BusMessageForwarder forwarder, Bus... buses) {
        if (agentName == null || "".equals(agentName)) {
            throw new IllegalArgumentException(
                    "Invalid name for an agent. It cannot be empty or null.");
        }
        if (forwarder == null) {
            throw new IllegalArgumentException("no forwarder");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        if (localAgentChannels == null) {
            throw new IllegalArgumentException(" agent " + agentName
                    + " not registered");
        }
        for (Bus bus : buses) {
            BusReceiver receiver = localAgentChannels.receivers[bus.ordinal()];
            if (receiver == null) {
                throw new IllegalArgumentException(" agent " + agentName
                        + " not registered on bus " + bus);
            }
            // receiver.addForwarder(forwarder);
        }
    }

    @Override
    public void removeMessageListener(String agentName,
            BusMessageForwarder forwarder, Bus... buses) {
        if (agentName == null || "".equals(agentName)) {
            throw new IllegalArgumentException(
                    "Invalid name for an agent. It cannot be empty or null.");
        }
        if (forwarder == null) {
            throw new IllegalArgumentException("no forwarder");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        if (localAgentChannels == null) {
            return;
        }
        for (Bus bus : buses) {
            BusReceiver receiver = localAgentChannels.receivers[bus.ordinal()];
            if (receiver == null) {
                continue;
            }
            // receiver.removeForwarder(forwarder);
        }
    }

    @Override
    public void connect(MessagingAccessLayer accessLayer)
            throws DuplicateAgentNameException, IOException {
        StringBuilder allBuses = new StringBuilder();
        for (BusAccess ba : accessLayer.getBusAccesses()) {
            allBuses.append(ba.getBus() + " ");
        }
        logger.fine("### Registering " + accessLayer.getName() + " for buses "
                + allBuses);

        if (closed) {
            // TODO: problem with multiple tests
            throw new TransportStateException("closing");
        }

        String agentName = accessLayer.getName();

        if (agentName == null || "".equals(agentName)) {
            throw new IllegalArgumentException("Invalid agent name "
                    + agentName);
        }

        for (BusAccess ba : accessLayer.getBusAccesses()) {
            Bus bus = ba.getBus();
            int index = ba.getBus().ordinal();
            try {

                // TUNNEL CONFIGURATION. Refer to documentation:
                // https://confluence.slac.stanford.edu/x/ahtQCw
                // Tunnel configuration if available
                String tunnelConfigurationFileStup = "tunnel_ccs";
                Properties tunnelProperties = BootstrapResourceUtils
                        .getBootstrapProperties(tunnelConfigurationFileStup
                                + ".properties", this.getClass());
                String tunnelProperty = tunnelProperties.getProperty(
                        "org.lsst.ccs.tunnel", "");

                JChannel tunnelChannel = null;
                JChannel busChannel = null;

                /**
                 * There are three tunnel configurations: - no Tunnel - Tunnel
                 * Server - Tunnel Client
                 *
                 * In the first two options we create the CCS buses with
                 * whatever configuration is chosen. In the last case the
                 * client's buses will be TCP tunnel buses. In the case of a
                 * Tunnel Server for each CCS bus we create a TCP tunnel bus
                 * that will echo all the messages from the CCS buses into the
                 * Tunnel and vice-versa from the tunnel to the CCS bus.
                 * */
                if (tunnelProperty.isEmpty() || tunnelProperty.equals("server")) {

                    XmlConfigurator configurator = createXmlConfiguratorForBus(
                            xmlConfigurationFile, jgroupsProperties,
                            bus.toString());
                    JChannel channel = new JChannel(configurator);
                    logger.debug("######################################################\n"
                            + "JGroup Configuration for bus: "
                            + bus.toString()
                            + "\n"
                            + channel.getProtocolStack()
                            .printProtocolSpecAsXML()
                            + "\n"
                            + "######################################################");
                    channel.setName(agentName);
                    channel.connect(bus.toString());
                    if (isDuplicate(channel.getView(), channel.getAddress())) {
                        channel.close();
                        throw new DuplicateAgentNameException(agentName,
                                "Channel with same name already exists in view");
                    }
                    lastViews[index] = channel.getView();

                    String[] propertiesArray = channel.getProperties().split(
                            ";");
                    String bind_addr = null;
                    for (int i = 0; i < propertiesArray.length; i++) {
                        if (propertiesArray[i].startsWith("bind_addr=")) {
                            bind_addr = propertiesArray[i];
                            break;
                        }
                    }
                    logger.debug("*** new view " + channel.getClusterName()
                            + " " + channel.getView() + " " + bind_addr);

                    busChannel = channel;
                }

                if (!tunnelProperty.isEmpty()) {

                    boolean isServer = tunnelProperty.equals("server");
                    try {

                        // String tunnelHost =
                        // tunnelProperties.getProperty("org.lsst.ccs.tunnel.host",
                        // "");
                        String tunnelPort = tunnelProperties.getProperty(
                                "org.lsst.ccs.tunnel.port", "7800");
                        int port = Integer.parseInt(tunnelPort) + 10
                                * bus.ordinal();

                        BusGossipRouter router = new BusGossipRouter(bus);
                        if (isServer) {
                            String startRouter = tunnelProperties
                                    .getProperty(
                                            "org.lsst.ccs.tunnel.start.router",
                                            "false");
                            if (startRouter.equals("true")) {
                                Thread t = new Thread(router);
                                t.start();
                            }
                        }
                        if (tunnelProperties
                                .getProperty(
                                        "org.lsst.ccs.jgroups."
                                        + bus
                                        + "_TUNNEL.TCPGOSSIP.initial_hosts",
                                        "").isEmpty()) {
                            tunnelProperties
                                    .setProperty(
                                            "org.lsst.ccs.jgroups."
                                            + bus
                                            + "_TUNNEL.TCPGOSSIP.initial_hosts",
                                            router.getHost() + "["
                                            + router.getPort() + "]");
                        }

                        if (tunnelProperties.getProperty(
                                "org.lsst.ccs.jgroups." + bus
                                + "_TUNNEL.TCP.bind_port", "")
                                .isEmpty()) {
                            tunnelProperties.setProperty(
                                    "org.lsst.ccs.jgroups." + bus
                                    + "_TUNNEL.TCP.bind_port",
                                    String.valueOf(port));
                        }
                        XmlConfigurator tunnelConfigurator = createXmlConfiguratorForBus(
                                tunnelConfigurationFileStup + ".xml",
                                tunnelProperties, bus + "_TUNNEL");
                        tunnelChannel = new JChannel(tunnelConfigurator);
                        logger.debug("######################################################\n"
                                + "JGroup Configuration for bus: "
                                + bus
                                + "_TUNNEL \n"
                                + tunnelChannel.getProtocolStack()
                                .printProtocolSpecAsXML()
                                + "\n"
                                + "######################################################");
                        tunnelChannel.setName(agentName);
                        tunnelChannel.connect(bus + "_TUNNEL");
                        lastViews[index] = tunnelChannel.getView();
                        if (isServer) {
                            tunnelChannel.setReceiver(new TunnelReceiver(
                                    busChannel, agentName));
                        }
                    } catch (Exception e) {
                        throw new IOException(e);
                    }

                }

                // If the busChannel was not created, then we are in the Tunnel
                // Client configuration.
                // This means that the busChannel is the tunnel itself.
                if (busChannel == null) {
                    busChannel = tunnelChannel;
                    tunnelChannel = null;
                }

                localAgentChannels.agentName = agentName;
                localAgentChannels.channels[bus.ordinal()] = busChannel;

                // registers a BusReceiver
                localAgentChannels.receivers[index] = new BusReceiver(ba,
                        busChannel, this, tunnelChannel);

            } catch (Exception e) {
                throw new IOException(e);
            }
        }

    }

    @Override
    public void disconnect(MessagingAccessLayer accessLayers) {
        Bus[] buses = new Bus[accessLayers.getBusAccesses().size()];
        for (BusAccess layer : accessLayers.getBusAccesses()) {
            buses[layer.getBus().ordinal()] = layer.getBus();
        }
        this.closeFor(accessLayers.getName(), buses);
    }

}
