package org.lsst.ccs.bus.jgroups;

import org.jgroups.*;
import org.jgroups.conf.ProtocolConfiguration;
import org.jgroups.conf.XmlConfigurator;
import org.jgroups.util.UUID;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.*;
import org.lsst.ccs.utilities.logging.Logger;

import java.io.IOException;
import java.io.InputStream;
import java.io.Serializable;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Each Bus is represented by a JGroups cluster.
 * <p/>
 * Each agent creates a JChannel, setName to name of agent and connect to corresponding Bus/cluster 
 * (so each agent may have up to 3 JChannel connected).
 * <BR>
 * Each of these 3 channels have a JGroups Address.
 * <p/>
 * for sending messages to an agent we have to find its Address in the group 
 * AND we have to find the addresses of the multiple anonymous agents that will receive a copy
 * <p/>
 * There is a dictionary to match agent name to jGroups Address. 
 * for each agent there is an Address except for anonymous agent that can be matched to multiple addresses.
 * <p/>
 * <
 * p/>
 * <B>ImPortant</B> every Jchannel in the JVM MUST be closed!
 */
public class JGroupsBusMessagingLayer implements BusMessagingLayer, ProvidesDisconnectionInformation {

    public static final String DEFAULT_UDP_PROTOCOL = "jgroups:udp_ccs";
    private Properties jgroupsProperties;
    private String xmlConfigurationFile = "udp_ccs.xml";
    private final static String propertyKeyForJGroups = "org.lsst.ccs.jgroups.";
    private final static String propertyKeyForAllBuses = propertyKeyForJGroups + "ALL.";

    //added to circumvent problems with wrong view of addresses
    private boolean tryBroadcast;
    private static Logger logger = Logger.getLogger("org.lsst.ccs.bus.jgroups");
    /**
     * the pair name of agent / array of addresses on the buses. 
     *  the array index is the bus index.
     */
    CopyOnWriteArrayList<AgentAddresses> namedAgentAddresses = new CopyOnWriteArrayList<>();
    // should be a set!
    /**
     * the anonymous agents adresses
     */
    private CopyOnWriteArrayList<AgentAddresses> anonymousAddresses = new CopyOnWriteArrayList<>();

    /**
     * for a known agent the infos for each bus linked to channels and to codes to handle the message (BusReceiver)
     */
    private Map<String, LocalAgentChannels> mapLocalChannels = new ConcurrentHashMap<>();
    /**
     *  the membership Listener for each bus. 
     *  Todo: allow multiple membershipListeners of provide a delegate or decorator
     */
    private BusMembershipListener[] membershipListeners = new BusMembershipListener[Bus.values().length];
    /**
     * the last known View on a Bus (kept to know if there is a creation or an update)
     */
    // accessed from synchronized code :so no volatile
    private View[] lastViews = new View[Bus.values().length];
    private volatile boolean closed;
    
    /**
     * for a given agent Name lists the addresses on the corresponding buses,
     * local agents are also registered here. (TODO: modify that?)
     */
    class AgentAddresses {

        String agentName;
        Address[] busAddresses = new Address[Bus.values().length];
    }

    /**
     * for a local agent the list of its channels nd of its adapter
     */
    class LocalAgentChannels {

        String agentName;
        JChannel[] channels = new JChannel[Bus.values().length];
        BusReceiver[] receivers = new BusReceiver[Bus.values().length];
    }

    /**
     * instance of this class act as a receiver of messages then forward to the BusMessageForwarders registered
     */
    class BusReceiver extends ReceiverAdapter {

        Bus bus;
        //todo: duplicate information
        JChannel curChan;
        //ParallelCommandDispatcher dispatcher;
        CopyOnWriteArrayList<BusMessageForwarder> listForwarders;
        //todo: again a duplicate!
        String agentName;

        BusReceiver(String agentName, Bus bus, JChannel curChan) {
            this.agentName = agentName;
            this.bus = bus;
            this.curChan = curChan;
            curChan.setReceiver(this);
        }

        public void addForwarder(BusMessageForwarder forwarder) {
            //TODO: reinstall Parallel ?
            /*
             if (dispatcher == null) {
             dispatcher = new ParallelCommandDispatcher();
             }
             dispatcher.addExecutant(forwarder);
             */
            if (listForwarders == null) {
                listForwarders = new CopyOnWriteArrayList<BusMessageForwarder>();
            }
            listForwarders.add(forwarder);
        }

        public void removeForwarder(BusMessageForwarder forwarder) {
            /*
             if (dispatcher == null) return;
             dispatcher.removeExecutant(forwarder);
             */
            if (listForwarders == null) {
                return;
            }
            listForwarders.remove(forwarder);
        }

        @Override
        public void receive(Message message) {

            BusMessage busMessInit;
            try {
                if (listForwarders == null) {
                    return;
                }
                if (listForwarders.size() == 0) {
                    return;
                }
                busMessInit = (BusMessage) message.getObject();
            } catch (final RuntimeException exc) {
                throw exc;
                //TODO: create a generic Message for reporting errors
                /* busMessInit = new BusMessage() {
                 @Override
                 public String getMessageType() {
                 return "error" + exc;
                 }
                 } ;
                 */
            }
            final BusMessage busMess = busMessInit;
            if (busMess.getOrigin().equals(agentName)) {
                logger.trace("message received by same agent than sender");
                return;
            }
            if (tryBroadcast) { // we can receive a message which is not for us!
                if (!ANONYMOUS_AGENT.equals(agentName)) {
                    if (busMess instanceof Command) {
                        String destination = ((Command) busMess).getDestination();
                        if ((destination == null) || "".equals(destination)
                                || "*".equals(destination)) {
                            //do nothing it's ok
                        } else {
                            String[] dests = destination.split(",");
                            boolean found = false;
                            for (String dest : dests) {
                                if (destination.equalsIgnoreCase(dest)) {
                                    found = true;
                                    break;
                                }
                            }
                            if (!found) {
                                return;
                            }
                        }
                    }
                }
            }
            /*
             CommandFor<BusMessageForwarder> cmdMessage = new CommandFor<BusMessageForwarder>() {
             @Override
             public void invokeOn(BusMessageForwarder instance) {
             instance.update(busMess);
             }
             };
             dispatcher.dispatchCommand(cmdMessage);
             */
            // TODO : reinstall parallel
            for (BusMessageForwarder forwarder : listForwarders) {
                forwarder.update(busMess);
            }

        }

        @Override
        public void viewAccepted(View view) {
            updateMapAddressesFromView(bus, curChan, view);
        }

        @Override
        public void suspect(Address address) {
            //get the address in other
            int index = bus.ordinal();
            for (AgentAddresses agentAddresses : namedAgentAddresses) {
                Address localAddress = agentAddresses.busAddresses[index];
                if (address.equals(localAddress)) {
                    BusMembershipListener busMembershipListener = membershipListeners[index];
                    if (busMembershipListener != null) {
                        String info = "";
                        if (address instanceof UUID) {
                            info = ((UUID) address).toStringLong();
                        }
                        busMembershipListener.disconnecting(address.toString(), info);
                       // busMembershipListener.disconnecting(" guess on " + address.toString(), info);
                    } else {
                      //  logger.warn("disconnection suspicion on " + address);
                    }
                    agentAddresses.busAddresses[index] = null;
                    break;
                }
            }
            // and warn
        }
    }
    /////////////////////////////////////////////// CTOR

    /**
     * there should be only one instance perJVM: it's up to the calling code
     * to check that.
     */
    @Deprecated
    public JGroupsBusMessagingLayer() {
        this(DEFAULT_UDP_PROTOCOL);
        //register a close
        /* Runtime runtime = Runtime.getRuntime() ;
         runtime.addShutdownHook( new Thread() {
         public void run() {
         try {
         //close() ;
         } catch (IOException e) {
         //do nothing
         }
         }
         });
         */

    }

    JGroupsBusMessagingLayer(String protocolString) {
        //index 1 : is for URL
        //index 2 :is for String to be passed to JChannel
        String[] protocolInfos = protocolString.split(":");
        String rawResource = protocolInfos[1];
        if (rawResource.length() > 0) {
            xmlConfigurationFile = String.format("/%s.xml", rawResource);
        }

        //The properties file corresponding to the XML file
        String propertiesConfigurationFile = xmlConfigurationFile.replace(".xml", ".properties");
        jgroupsProperties = BootstrapResourceUtils.getBootstrapProperties(propertiesConfigurationFile, this.getClass());

        tryBroadcast = Boolean.parseBoolean(jgroupsProperties.getProperty("org.lsst.ccs.jgroups.trybroadcast"));
    }

    //////////////////////////////////////////////
    ///////////////////////////////////////////// METHODS
    @Override
    public void register(String agentName, Bus... buses) throws IOException {

        StringBuilder allBuses = new StringBuilder();
        for (Bus bus : buses) {
            allBuses.append(bus.toString() + " ");
        }
        logger.info("### Registering " + agentName + " for buses " + allBuses);

        if (closed) {
            // TODO: problem with multiple tests
            throw new TransportStateException("closing");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        if (agentName == null || "".equals(agentName)) {
            agentName = ANONYMOUS_AGENT;
        } else {

            /*
             is this bus already registered?
             TODO: stop or not if already connected to the same bus?
             problem : the call must be idempotent so should stop ONLY IF
             a different process is trying to create another agent with same name
             */
            for (AgentAddresses addresses : namedAgentAddresses) {
                if (agentName.equals(addresses.agentName)) {
                    for (Bus bus : buses) {
                        if (addresses.busAddresses[bus.ordinal()] != null) {
                            DuplicateBusNameException exception = new DuplicateBusNameException(agentName, " already registered on " + bus);
                            logger.error(exception);
                            BusMembershipListener listener = membershipListeners[bus.ordinal()];
                            if (listener != null) {
                                listener.anormalEvent(exception);
                            }
                        }
                    }
                }
            }
        }

        // is this agent already known to the map
        LocalAgentChannels localChannels = mapLocalChannels.get(agentName);
        // if not: we create an entry in the Map
        if (localChannels == null) {
            localChannels = new LocalAgentChannels();
            localChannels.agentName = agentName;
            mapLocalChannels.put(agentName, localChannels);
        }
        // if anonymous registers local channel and add it to the list of anonymous
        for (Bus bus : buses) {
            int index = bus.ordinal();
            JChannel channel = localChannels.channels[index];
            // todo replace isConnected
            if (channel == null || !channel.isConnected()) {
                try {

                    XmlConfigurator configurator = createXmlConfiguratorForBus(xmlConfigurationFile, jgroupsProperties, bus.toString());
                    channel = new JChannel(configurator);

                    // bamade: 4 log messages grouped into one
                    logger.debug("######################################################\n"
                            + "JGroup Configuration for bus: " + bus.toString() + "\n"
                            + channel.getProtocolStack().printProtocolSpecAsXML() + "\n"
                            + "######################################################"
                    );

                    localChannels.channels[index] = channel;
                    channel.setName(agentName);
                    channel.connect(bus.toString());
                    // who is connected before ?
                    View view = channel.getView();
                    //todo: check for synchronization problems between receiver and update
                    updateMapAddressesFromView(bus, channel, view);
                    // registers a BusReceiver
                    localChannels.receivers[index] = new BusReceiver(agentName, bus, channel);

                } catch (Exception e) {
                    throw new IOException(e);
                }
            } else {
                //we do nothing when we re-register for same bus
            }

        }
    }

    /**
     * Creates an XmlConfiguration object for a given bus. This object is used to initialize a JChannel. 
     * The XmlConfiguration object is created from an xml file that is loaded from the bootstrap environment.
     * The properties object is then scanned to find relevant values to modify XmlConfiguration object.
     *
     * The property keys are of the form org.lsst.ccs.jgroups.[BusIdentifier].[ProtocolName].[ProtocolPropertyName]=[ProtocolPropertyValue]
     *
     * Where: 
     * - BusIdentifier can be either COMMAND, LOG, STATUS if the property applied to an individual bus or ALL if it applies to all the buses. 
     * - ProtocolName is a valid JGroups protocol name as specified at http://www.jgroups.org/manual/html/protlist.html 
     * - ProtocolPropertyName is a property for the given protocol 
     * - ProtocolPropertyValue is the value to be given to it
     *
     * If the ProtocolName does not exist in the original xml file, a new ProtocolConfiguration object will be added to the 
     * XmlConfiguration.
     *
     *
     * @param xmlConfigurationFile The name of the xml file to be loaded
     * @param properties The properties object to be used to overwrite the xml file
     * @param bus The bus for which the properties are scanned
     * @return The XmlConfiguration object to be used to create the JChannel
     *
     */
    public static XmlConfigurator createXmlConfiguratorForBus(String xmlConfigurationFile, Properties properties, String bus) {

        InputStream xmlConfigurationFileInputStream = BootstrapResourceUtils.getBootstrapResource(xmlConfigurationFile, JGroupsBusMessagingLayer.class);
        if (xmlConfigurationFileInputStream == null) {
            throw new IllegalArgumentException(xmlConfigurationFile + " not found as a resource");
        }

        XmlConfigurator configurator = null;
        try {
            configurator = XmlConfigurator.getInstance(xmlConfigurationFileInputStream);
        } catch (IOException ioe) {
            throw new RuntimeException("Could not create JGroups xml configuration object for " + xmlConfigurationFile, ioe);
        }

        List<ProtocolConfiguration> protocolConfigurationList = configurator.getProtocolStack();

        //First look for properties that applies to all BUSES, then for bus specific properties.
        String[] propertyKeys = new String[]{propertyKeyForAllBuses, propertyKeyForJGroups + bus.toUpperCase() + "."};

        for (String propertyKey : propertyKeys) {
            // When looping over all the keys of a property object we have to keep in mind that the
            // properties might be chained. The following is a utility method to get all keys in a
            // Properties object, including its parents. See https://jira.slac.stanford.edu/browse/LSSTCCS-227
            Set<Object> keys = BootstrapResourceUtils.getAllKeysInProperties(properties);
            for (Object key : keys) {
                String property = (String) key;
                if (property.startsWith(propertyKey)) {

                    //Remove the property key so that only PROTOCOL.propertyName is left
                    String shortProperty = property.replace(propertyKey, "");
                    int divider = shortProperty.lastIndexOf(".");
                    String protocolPropertyName = shortProperty.substring(divider + 1);
                    String protocolName = shortProperty.substring(0, divider);
                    String protocolPropertyValue = properties.getProperty(property);

                    boolean found = false;
                    for (ProtocolConfiguration protocolConfiguration : protocolConfigurationList) {
                        if (protocolConfiguration.getProtocolName().equals(protocolName)) {
                            protocolConfiguration.getProperties().put(protocolPropertyName, protocolPropertyValue);
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        HashMap<String, String> hash = new HashMap<>();
                        hash.put(protocolPropertyName, protocolPropertyValue);
                        ProtocolConfiguration newProtocol = new ProtocolConfiguration(protocolName, hash);
                        protocolConfigurationList.add(newProtocol);
                    }

                }
            }
        }
        return configurator;
    }

    /**
     *  update local data about which are the agents connected to the Clusters.
     *  this is called in two circumstances :
     *
     *      <UL>
     *          <LI/> when creating an access to a cluster we want to know who is connected and what are their addresses
     *          <LI/> when we get a <TT>suspect</TT> call due to the fact that a remote agent got 
     *          disconnect (or is lost!)
     *      </UL>
     *  if a remote agent springs to life either ther is not entry for it and we must create one 
     *  or if there is an empty entry (left by a previous connection) we should fill it.
     *  <P/>
     *  if a remote agent is disconnected we must empty its entry.
     *  <BR/>
     *  we now must compare the last known view with the current one to know which agent appeared and which diasppeared!
     *
     * @param bus
     * @param channel
     * @param view
     */
    //todo : in a separate thread?
    private synchronized void updateMapAddressesFromView(Bus bus, JChannel channel, View view) {
        int index = bus.ordinal();
        View lastView = lastViews[index];
        // due to short size complexity is not an issue
        // so there should be two lists: those added and those removed
        
        // Algorithm using the new API : diffArray[0] are the joined members
        // diffArray[1] are the left members
        Address[][] diffArray = org.jgroups.View.diff(lastView, view);
        
        lastViews[index] = view;
        for (Address address : diffArray[0]) {
            // looks if address is already in table
            String name = address.toString(); // Returns the name set by setName() JChannel method
            if (ANONYMOUS_AGENT.equals(name)) {
                // the problem here is that we don not know which
                // anonymous agent is concerned
                // so we just add on blindly
                // TODO: re-evaluate strategy
                AgentAddresses agentAddresses = new AgentAddresses();
                agentAddresses.agentName = ANONYMOUS_AGENT;
                agentAddresses.busAddresses[index] = address;
                anonymousAddresses.add(agentAddresses);
                logger.info("adding ====== " + address + " to " + bus);
                // TODO: add anonymous connect message ?
            } else {
                // Looking for an agent with the name "name" in namedAgentAddresses
                boolean found = false;
                for (AgentAddresses agentAddresses : namedAgentAddresses) {
                    if (agentAddresses.agentName.equals(name)) {
                        //TODO: if already set?
                        agentAddresses.busAddresses[index] = address;
                        logger.info("adding ====== " + address + " to " + bus);
                        found = true;
                        break;
                    }
                }
                if (!found) { // A new agent is created
                    AgentAddresses agentAddresses = new AgentAddresses();
                    agentAddresses.agentName = name;
                    agentAddresses.busAddresses[index] = address;
                    namedAgentAddresses.add(agentAddresses);
                    logger.info("adding ====== " + address + " to " + bus);
                }
                BusMembershipListener listener = membershipListeners[index];
                if (listener != null) {
                    String info = "";
                    if (address instanceof UUID) {
                        info = ((UUID) address).toStringLong();
                    }
                    listener.connecting(name, info);
                }
            }
        }

        for (Address address : diffArray[1]) { // Removal
            // looks if address is already in table
            String name = address.toString();
            if (ANONYMOUS_AGENT.equals(name)) {
                Iterator<AgentAddresses> it = anonymousAddresses.iterator();
                AgentAddresses agentAddressesToRemove = null;
                while (it.hasNext()) {
                    AgentAddresses agentAddresses = it.next();
                    Address storedAddress = agentAddresses.busAddresses[index];
                    if (address.equals(storedAddress)) {
                        //it.remove(); iterator on copyOnwrite does not work!
                        // TODO: add anonymous remove message?
                        agentAddressesToRemove = agentAddresses;
                    }
                }
                //iterator
                if (agentAddressesToRemove != null) {
                    anonymousAddresses.remove(agentAddressesToRemove);
                }

            } else {
                for (AgentAddresses agentAddresses : namedAgentAddresses) {
                    if (agentAddresses.agentName.equals(name)) {
                        agentAddresses.busAddresses[index] = null;
                        BusMembershipListener listener = membershipListeners[index];
                        if (listener != null) {
                            String info = "";
                            if (address instanceof UUID) {
                                info = ((UUID) address).toStringLong();
                            }
                            listener.disconnecting(name, info);
                        } else {
                            logger.info("removing ====== " + address + " from " + bus);
                        }
                        break;
                    }
                }

            }
        }

    }

    @Override
    public void closeFor(String agentName, Bus... buses) {
        if (agentName == null || "".equals(agentName)) {
            agentName = ANONYMOUS_AGENT;
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        LocalAgentChannels localAgentChannels = mapLocalChannels.get(agentName);
        if (localAgentChannels != null) {
            //get the view and if address is local then remove!
            //Todo! make namedAgentAddresses a Map?
            AgentAddresses agentAddresses = null;
            //TODO: this code a shortcut .... you need to be more clever!
            if (!ANONYMOUS_AGENT.equals(agentName)) {
                for (AgentAddresses agAddress : namedAgentAddresses) {
                    if (agentName.equals(agAddress.agentName)) {
                        //TODO: check the real address!
                        agentAddresses = agAddress;
                        break;
                    }
                }
            }
            for (Bus bus : buses) {
                int index = bus.ordinal();
                JChannel channel = localAgentChannels.channels[index];
                if (channel != null) {
                    channel.close();
                }
                if (agentAddresses != null) {
                    agentAddresses.busAddresses[index] = null;
                }
            }
        }
    }

    @Override
    public void close() throws IOException {
        closed = true;
        //for all local agents
        Iterator<String> itKeys = mapLocalChannels.keySet().iterator();
        while (itKeys.hasNext()) {
            String key = itKeys.next();
            logger.info("******** removing " + key);
            LocalAgentChannels localAgentChannels = mapLocalChannels.get(key);
            for (JChannel channel : localAgentChannels.channels) {
                if (channel != null) {
                    channel.close();
                }
            }
        }
        //TODO: do not reuse this object! create a volatil boolean closed!
        //close explicitly all channels
    }

    @Override
    public <T extends BusPayload> void sendMessage(String senderAgent, Bus<T> bus, T message, String... destinations) throws IOException {
        if (closed) {
            throw new TransportStateException("closing");
        }
        if (senderAgent == null) {
            throw new IllegalArgumentException("no sender agent");
        }
        if (bus == null) {
            throw new IllegalArgumentException("no bus");
        }
        if (message == null) {
            throw new IllegalArgumentException("no message");
        }
        // if explicit list it is initialized, if null -> broadcast
        ArrayList<Address> listDestinationAddress = null;
        int index = bus.ordinal();
        // list of unknown destinations to be kept
        List<String> failed = new ArrayList<String>();
        if (destinations.length == 0 || "".equals(destinations[0]) || "*".equals(destinations[0])) {
            //send to all
        } else { // explicit list created
            listDestinationAddress = new ArrayList<Address>();
            // send to destinations
            for (String destination : destinations) {
                boolean found = false;
                for (AgentAddresses agentAddresses : namedAgentAddresses) {
                    if (agentAddresses.agentName.equals(destination)) {
                        Address address = agentAddresses.busAddresses[index];
                        if (address != null) {
                            listDestinationAddress.add(address);
                        } else {
                            if (tryBroadcast) {
                                listDestinationAddress.add(null);
                                BusMembershipListener listener = membershipListeners[index];
                                if (listener != null) {
                                    listener.anormalEvent(new DuplicateBusNameException(destination, " default broadcast"));
                                }
                                logger.info("trying broadcast instead of " + agentAddresses.agentName);
                            } else { /**/

                                failed.add(destination);
                                logger.info("sending fail (closed)" + agentAddresses.agentName);
                            } /**/

                        }
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    if (tryBroadcast) {
                        listDestinationAddress.add(null);
                        logger.info("trying broadcast instead of " + destination);
                    } else { /**/

                        failed.add(destination);
                        logger.info("sending fail to : " + destination);
                    }
                }
            }
            // send to anonymous
            for (AgentAddresses agentAddresses : anonymousAddresses) {
                Address address = agentAddresses.busAddresses[index];
                if (address != null) {
                    listDestinationAddress.add(address);
                }
            }
        }
        // now trying to find the proper Channel
        LocalAgentChannels localAgentChannels = mapLocalChannels.get(senderAgent);
        if (localAgentChannels == null) {
            throw new IllegalArgumentException("agent " + senderAgent + " not registered on " + bus);
        }
        JChannel channel = localAgentChannels.channels[index];
        if (channel == null) {
            throw new IllegalArgumentException("agent " + senderAgent + " not registered on " + bus);
        }
        try {
            //broadcast
            if (listDestinationAddress == null) {
                // versiion 2 channel.send(null, null, (Serializable) message);
                channel.send(null, (Serializable) message);
            } else { //explicit
                boolean nullSent = false;
                for (Address dest : listDestinationAddress) {
                    if (dest == null) {
                        if (nullSent) {
                            continue;
                        }
                        nullSent = true;
                    }
                    // version 2 channel.send(dest, null, (Serializable) message);
                    channel.send(dest, (Serializable) message);
                }
            }
        } catch (Exception exc) {
            throw new IOException(exc);
        }
        //TODO : what to do if send fails on an existing adresse but there are also fails (unlikely)= getSuppressed?
        // throwing exception for failed addresses
        if (failed.size() != 0) {
            DestinationsException exc = new DestinationsException(senderAgent, failed.toArray());
            BusMembershipListener listener = membershipListeners[index];
            if (listener != null) {
                listener.anormalEvent(exc);
            }
            throw exc;
        }
    }

    @Override
    public void addMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (agentName == null || "".equals(agentName)) {
            agentName = ANONYMOUS_AGENT;
        }
        if (forwarder == null) {
            throw new IllegalArgumentException("no forwarder");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        LocalAgentChannels localAgentChannels = mapLocalChannels.get(agentName);
        if (localAgentChannels == null) {
            throw new IllegalArgumentException(" agent " + agentName + " not registered");
        }
        for (Bus bus : buses) {
            BusReceiver receiver = localAgentChannels.receivers[bus.ordinal()];
            if (receiver == null) {
                throw new IllegalArgumentException(" agent " + agentName + " not registered on bus " + bus);
            }
            receiver.addForwarder(forwarder);
        }
    }

    @Override
    public void removeMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (agentName == null || "".equals(agentName)) {
            agentName = ANONYMOUS_AGENT;
        }
        if (forwarder == null) {
            throw new IllegalArgumentException("no forwarder");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        LocalAgentChannels localAgentChannels = mapLocalChannels.get(agentName);
        if (localAgentChannels == null) {
            return;
        }
        for (Bus bus : buses) {
            BusReceiver receiver = localAgentChannels.receivers[bus.ordinal()];
            if (receiver == null) {
                continue;
            }
            receiver.removeForwarder(forwarder);
        }
    }

    @Override
    public void setMembershipListener(BusMembershipListener listener, Bus... buses) {
        if (buses.length == 0) {
            buses = Bus.values();
        }
        for (Bus bus : buses) {
            membershipListeners[bus.ordinal()] = listener;
        }
    }

    /**
     * gets a list of agent names connected to a bus. 
     * There is no distinction between agents that are passive or not.
     * @param bus
     * @return a List of names (can be empty), anonymous agents are listed with a special Name
     */
    @Override
    public List<String> getConnectedNames(Bus bus) {
        int index = bus.ordinal();
        List<String> res = new ArrayList<>() ;
        for(LocalAgentChannels localAgentChannels : mapLocalChannels.values()) {
            JChannel channel  = localAgentChannels.channels[index] ;
            if(channel == null) continue ;
            // once we get a connected Channel we return the View information on this Channel
            View view = channel.getView() ;
            for( Address address : view.getMembers()) {
                res.add(channel.getName(address)) ;
            }
            return res ;
        }
        return res ;
    }
    
    // TODO : this method should not be public
    @Override
    public BusMembershipListener getBusMembershipListener(Bus bus){
        return membershipListeners[bus.ordinal()];
    }
    
}