package org.lsst.ccs.bus.jgroups;

import org.apache.log4j.Logger;
import org.jgroups.*;
import org.jgroups.util.UUID;
import org.lsst.ccs.bus.*;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringReader;
import java.net.URL;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Each Bus is represented by a JGroups cluster.
 * <p/>
 * Each agent creates a JChannel, setName to name of agent and connect to corresponding Bus/cluster
 * (so each agent may have up to 3 JChannel connected).
 * <BR>
 * Each of these 3 channels have a JGroups Address.
 * <p/>
 * for sending messages to an agent we have to find its Address in the group
 * AND we have to find the addresses of the multiple anonymous agents that will receive a copy
 * <p/>
 * There is a dictionary to match agent name to jGroups Address.
 * for each agent there is an Address except for anonymous agent that can be matched to multiple adresses.
 * <p/>
 * <p/>
 * <B>ImPortant</B> every Jchannel in the JVM MUST be closed!
 */
public class JGroupsBusMessagingLayer implements BusMessagingLayer {
    public static final String DEFAULT_UDP_PROTOCOL = "jgroups:udp_ccs:" ;
    public static final String DEFAULT_UDP_PROPERTIES =
            "LOG:udp.mcast_port=26969;STATUS:udp.mcast_port=36969;COMMAND:udp.mcast_port=46969;" ;
    public static final String DEFAULT_TCP_PROTOCOL = "jgroups:tcp_ccs:" ;
    public static final String DEFAULT_TCP_PROPERTIES =
            "jgroups.bind_addr=localhost;lsst.groups.trybroadcast=true; " ;
    //todo: change
    URL protocolURL = JGroupsBusMessagingLayer.class.getResource("/udp_ccs.xml");
    String protocolInfo ;
    /*
    TODO: check number are in right range
    + make it variable (accept method)to be determined
     */
    String[] portDesc = {"26969", "36969", "46969"};
    Properties properties = new Properties() ;
    //added to circumvent problems with wrong view of adresses
    boolean tryBroadcast ;
    static Logger logger = Logger.getLogger("lsst.ccs.groups");

    CopyOnWriteArrayList<AgentAdresses> otherAddresses = new CopyOnWriteArrayList<AgentAdresses>();
    // should be a set!
    CopyOnWriteArrayList<AgentAdresses> anonymousAddresses = new CopyOnWriteArrayList<AgentAdresses>();
    Map<String, LocalAgentChannels> mapLocalChannels = new ConcurrentHashMap<String, LocalAgentChannels>();
    BusMembershipListener[] membershipListeners = new BusMembershipListener[Bus.values().length];
    // accessed from synchronized code :so no volatile
    View[] lastViews = new View[Bus.values().length];
    volatile boolean closed;

    /**
     * for  a given agent Name lists the adresses on  the corresponding buses,
     * local agents are also registerd here. (TODO: modiffy that?)
     */
    class AgentAdresses {
        String agentName;
        Address[] busAdresses = new Address[Bus.values().length];
    }

    /**
     * for a local agent the list of its channels
     */
    class LocalAgentChannels {
        String agentName;
        JChannel[] channels = new JChannel[Bus.values().length];
        BusReceiver[] receivers = new BusReceiver[Bus.values().length];
    }

    class BusReceiver extends ReceiverAdapter {
        Bus bus;
        JChannel curChan;
        //ParallelCommandDispatcher dispatcher;
        CopyOnWriteArrayList<BusMessageForwarder> listForwarders;
        String agentName;

        BusReceiver(String agentName, Bus bus, JChannel curChan) {
            this.agentName = agentName;
            this.bus = bus;
            this.curChan = curChan;
            curChan.setReceiver(this);
        }

        public void addForwarder(BusMessageForwarder forwarder) {
            //TODO: reinstall Parallel
            /*
               if (dispatcher == null) {
                   dispatcher = new ParallelCommandDispatcher();
               }
               dispatcher.addExecutant(forwarder);
           */
            if (listForwarders == null) {
                listForwarders = new CopyOnWriteArrayList<BusMessageForwarder>();
            }
            listForwarders.add(forwarder);
        }

        public void removeForwarder(BusMessageForwarder forwarder) {
            /*
               if (dispatcher == null) return;
               dispatcher.removeExecutant(forwarder);
           */
            if (listForwarders == null) return;
            listForwarders.remove(forwarder);
        }

        @Override
        public void receive(Message message) {
            final BusMessage busMess = (BusMessage) message.getObject();
            //if (dispatcher != null) {
            if (listForwarders != null) {
                if (tryBroadcast) { // we can receive a message which is not for us!
                    if (!ANONYMOUS_AGENT.equals(agentName)) {
                        if (busMess instanceof Command) {
                            String destination = ((Command) busMess).getDestination();
                            if ((destination == null) || "".equals(destination)
                                    || "*".equals(destination)) {
                                //do nothing it's ok
                            } else {
                                String[] dests = destination.split(",");
                                boolean found = false;
                                for (String dest : dests) {
                                    if (destination.equalsIgnoreCase(dest)) {
                                        found = true;
                                        break;
                                    }
                                }
                                if (!found) return;
                            }
                        }
                    }
                } /**/
                /*
                    CommandFor<BusMessageForwarder> cmdMessage = new CommandFor<BusMessageForwarder>() {
                        @Override
                        public void invokeOn(BusMessageForwarder instance) {
                            instance.update(busMess);
                        }
                    };
                    dispatcher.dispatchCommand(cmdMessage);
            */
                // TODO : reinstall parallel
                for (BusMessageForwarder forwarder : listForwarders) {
                    forwarder.update(busMess);
                }
            }

        }

        @Override
        public void viewAccepted(View view) {
            updateMapAdressesFromView(bus, curChan, view);
        }

        @Override
        public void suspect(Address address) {
            //get the address in other
            int index = bus.ordinal();
            for (AgentAdresses agentAdresses : otherAddresses) {
                Address localAddress = agentAdresses.busAdresses[index];
                if (address.equals(localAddress)) {
                    BusMembershipListener busMembershipListener = membershipListeners[index];
                    if (busMembershipListener != null) {
                        String info = "";
                        if (address instanceof UUID) {
                            info = ((UUID) address).toStringLong();
                        }
                        busMembershipListener.disconnecting(address.toString(), info);
                    }
                    agentAdresses.busAdresses[index] = null;
                    break;
                }
            }
            // and warn
        }


    }
    /////////////////////////////////////////////// CTOR

    /**
     * there should be only one instance perJVM: it's up to the calling code
     * to check that.
     */
    @Deprecated
    public JGroupsBusMessagingLayer() {
        this(DEFAULT_UDP_PROTOCOL, DEFAULT_UDP_PROPERTIES) ;
        //register a close
        /* Runtime runtime = Runtime.getRuntime() ;
        runtime.addShutdownHook( new Thread() {
            public void run() {
                try {
                    //close() ;
                } catch (IOException e) {
                    //do nothing
                }
            }
        });
        */
        // getting rid of obnoxious traces : se the corresponding xml file
        //System.setProperty("jgroups.pbCast.GMS.print_local_addr", "false");

    }
    
    JGroupsBusMessagingLayer(String protocolString, String propertiesString) {
        //index 1 : is for URL
        //index 2 :is for String to be passed to JChannel
        String[] protocolInfos = protocolString.split(":") ;
        String rawResource = protocolInfos[1] ;
        if(rawResource.length() > 0) {
            String resourceName = String.format("/%s.xml", rawResource) ;
            protocolURL = JGroupsBusMessagingLayer.class.getResource(resourceName);
            if(protocolURL == null) {
                throw new IllegalArgumentException(resourceName + " not found as a resource") ;
            }
        } else {
            protocolInfo = protocolInfos[2] ;
        }

        StringReader reader = new StringReader(propertiesString.replace(';', '\n')) ;
        try {
            //Properties properties = System.getProperties() ;
            properties.load(reader);
            // copy to system Properties all what is not BUs
            Properties systemProperties = System.getProperties() ;
            //TODO: do not copy some properties such as BUS properties
            systemProperties.putAll(properties);
            tryBroadcast = Boolean.getBoolean("lsst.groups.trybroadcast");
        } catch (IOException e) {
            throw new IllegalArgumentException("properties syntax? " + propertiesString) ;
        }

        //map.put(getKey(protocolString, propertiesString), this) ;
    }
    

    //////////////////////////////////////////////
    ///////////////////////////////////////////// METHODS

    @Override
    public void register(String agentName, Bus... buses) throws IOException {
        if (closed) throw new TransportStateException("closing");
        if (buses.length == 0) {
            buses = Bus.values();
        }
        if (agentName == null || "".equals(agentName)) {
            agentName = ANONYMOUS_AGENT;
        } else {

            for (AgentAdresses adresses : otherAddresses) {
                if (agentName.equals(adresses.agentName)) {
                    for (Bus bus : buses) {
                        if (adresses.busAdresses[bus.ordinal()] != null) {
                            DuplicateBusNameException exception = new DuplicateBusNameException(agentName, " already registered on " + bus);
                            BusMembershipListener listener = membershipListeners[bus.ordinal()];
                            if (listener != null) {
                                listener.anormalEvent(exception);
                            }
                        }
                    }
                }
            }
        }

        LocalAgentChannels localChannels = mapLocalChannels.get(agentName);
        if (localChannels == null) {
            localChannels = new LocalAgentChannels();
            localChannels.agentName = agentName;
            mapLocalChannels.put(agentName, localChannels);
        }
        // if anonymous registers local channel and add it  to thelist of anonymous
        for (Bus bus : buses) {
            int index = bus.ordinal();
            JChannel channel = localChannels.channels[index];
            if (channel == null || !channel.isConnected()) {
                try {
                    String propValueString = properties.getProperty(bus.toString()) ;
                    if(propValueString != null) {
                        String[] keyVal = propValueString.split("=") ;
                        System.setProperty("jgroups."+ keyVal[0], keyVal[1]);
                    }
                    // NO !System.setProperty("jgroups.udp.mcast_port", "56969");
                    if(protocolInfo == null) {
                        channel = new JChannel(protocolURL);
                    } else {
                        channel = new JChannel(protocolInfo) ;
                    }
                    localChannels.channels[index] = channel;
                    channel.setName(agentName);
                    channel.connect(bus.toString());
                    // who is connected before ?
                    View view = channel.getView();
                    updateMapAdressesFromView(bus, channel, view);
                    // registers a BusReceiver
                    localChannels.receivers[index] = new BusReceiver(agentName, bus, channel);

                } catch (Exception e) {
                    throw new IOException(e);
                }
            }

        }
    }

    //todo : in a separate thread?
    private synchronized void updateMapAdressesFromView(Bus bus, JChannel channel, View view) {
        boolean adding;
        int index = bus.ordinal();
        Collection<Address> newMembers = view.getMembers();
        Collection<Address> modifiedMembers;
        View lastView = lastViews[index];
        if (lastView == null) {
            adding = true;
            modifiedMembers = newMembers;
        } else {
            //. TODO: rewrite this algorithm : does not work if add and remove in the same call
            // due to short size complexity is not an issue
            // so there should be two lists: those added and those removed
            Collection<Address> oldMembers = lastView.getMembers();
            adding = newMembers.containsAll(oldMembers);
            if (adding) {
                modifiedMembers = new ArrayList<Address>(newMembers);
                modifiedMembers.removeAll(oldMembers);
            } else {
                modifiedMembers = new ArrayList(oldMembers);
                modifiedMembers.removeAll(newMembers);
            }
        }
        lastViews[index] = view;
        if (adding) {
            for (Address address : modifiedMembers) {
                // looks if address is already in table
                String name = address.toString();
                if (ANONYMOUS_AGENT.equals(name)) {
                    // teh problem here is that we don not know which
                    // anonymous agent is concerned
                    // so we just add on blindly
                    // TODO: re-evaluate strategy
                    AgentAdresses agentAdresses = new AgentAdresses();
                    agentAdresses.agentName = ANONYMOUS_AGENT;
                    agentAdresses.busAdresses[index] = address;
                    anonymousAddresses.add(agentAdresses);
                    logger.info("adding ====== " + address + " to " + bus);
                    // TODO: add anonymous connect message ?
                } else {
                    boolean found = false;
                    for (AgentAdresses agentAdresses : otherAddresses) {
                        if (agentAdresses.agentName.equals(name)) {
                            //TODO: if already set?
                            agentAdresses.busAdresses[index] = address;
                            logger.info("adding ====== " + address + " to " + bus);
                            found = true;
                            break;
                        }
                    }
                    if (!found) {
                        AgentAdresses agentAdresses = new AgentAdresses();
                        agentAdresses.agentName = name;
                        agentAdresses.busAdresses[index] = address;
                        otherAddresses.add(agentAdresses);
                        logger.info("adding ====== " + address + " to " + bus);
                    }
                    BusMembershipListener listener = membershipListeners[index];
                    if (listener != null) {
                        String info = "";
                        if (address instanceof UUID) {
                            info = ((UUID) address).toStringLong();
                        }
                        listener.connecting(name, info);
                    }
                }

            }

        } else { // removing adresses
            for (Address address : modifiedMembers) {
                // looks if address is already in table
                String name = address.toString();
                if (ANONYMOUS_AGENT.equals(name)) {
                    Iterator<AgentAdresses> it = anonymousAddresses.iterator();
                    AgentAdresses agentAdressesToRemove = null;
                    while (it.hasNext()) {
                        AgentAdresses agentAdresses = it.next();
                        Address storedAddress = agentAdresses.busAdresses[index];
                        if (address.equals(storedAddress)) {
                            //it.remove(); iterator on copyOnwrite does not work!
                            // TODO: add anonymous remove message?
                            agentAdressesToRemove = agentAdresses ;
                        }
                    }
                    //iterator
                    if(agentAdressesToRemove != null) {
                        anonymousAddresses.remove(agentAdressesToRemove) ;
                    }

                } else {
                    for (AgentAdresses agentAdresses : otherAddresses) {
                        if (agentAdresses.agentName.equals(name)) {
                            agentAdresses.busAdresses[index] = null;
                            logger.info("removing ====== " + address + " from " + bus);
                            BusMembershipListener listener = membershipListeners[index];
                            if (listener != null) {
                                String info = "";
                                if (address instanceof UUID) {
                                    info = ((UUID) address).toStringLong();
                                }
                                listener.disconnecting(name, info);
                            }
                            break;
                        }
                    }

                }
            }

        }

    }

    @Override
    public void closeFor(String agentName, Bus... buses) {
        if (agentName == null || "".equals(agentName)) {
            agentName = ANONYMOUS_AGENT;
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        LocalAgentChannels localAgentChannels = mapLocalChannels.get(agentName);
        if (localAgentChannels != null) {
            //get the view and if adress is local then remove!
            //Todo! make otherAdresses a Map?
            AgentAdresses agentAdresses = null;
            //TODO: this code a shortcut .... you need to be more clever!
            if (!ANONYMOUS_AGENT.equals(agentName)) {
                for (AgentAdresses agAddress : otherAddresses) {
                    if (agentName.equals(agAddress.agentName)) {
                        //TODO: check the real address!
                        agentAdresses = agAddress;
                        break;
                    }
                }
            }
            for (Bus bus : buses) {
                int index = bus.ordinal();
                JChannel channel = localAgentChannels.channels[index];
                if (channel != null) {
                    channel.close();
                }
                if (agentAdresses != null) {
                    agentAdresses.busAdresses[index] = null;
                }
            }
        }
    }

    @Override
    public void close() throws IOException {
        closed = true;
        //for all local agents
        Iterator<String> itKeys = mapLocalChannels.keySet().iterator();
        while (itKeys.hasNext()) {
            String key = itKeys.next();
            logger.info("******** removing " + key);
            LocalAgentChannels localAgentChannels = mapLocalChannels.get(key);
            for (JChannel channel : localAgentChannels.channels) {
                if (channel != null) {
                    channel.close();
                }
            }
        }
        //TODO: do not reuse this object! create a volatil boolean closed!
        //close explicitly all channels
    }

    @Override
    public <T extends BusPayload> void sendMessage(String senderAgent, Bus<T> bus, T message, String... destinations) throws IOException {
        if (closed) throw new TransportStateException("closing");
        if (senderAgent == null) {
            throw new IllegalArgumentException("no sender agent");
        }
        if (bus == null) {
            throw new IllegalArgumentException("no bus");
        }
        if (message == null) {
            throw new IllegalArgumentException("no message");
        }
        ArrayList<Address> listDestinationAddress = null;
        int index = bus.ordinal();
        List<String> failed = new ArrayList<String>();
        if (destinations.length == 0 || "".equals(destinations[0]) || "*".equals(destinations[0])) {
            //send to all

        } else {
            listDestinationAddress = new ArrayList<Address>();
            // send to destinations
            for (String destination : destinations) {
                boolean found = false;
                for (AgentAdresses agentAdresses : otherAddresses) {
                    if (agentAdresses.agentName.equals(destination)) {
                        Address address = agentAdresses.busAdresses[index];
                        if (address != null) {
                            listDestinationAddress.add(address);
                        } else {
                            if (tryBroadcast) {
                                listDestinationAddress.add(null);
                                BusMembershipListener listener = membershipListeners[index];
                                if (listener != null) {
                                    listener.anormalEvent(new DuplicateBusNameException(destination, " default broadcast"));
                                }
                                logger.info("trying broadcast instead of " + agentAdresses.agentName);
                            } else { /**/
                                failed.add(destination);
                                logger.info("sending fail (closed)" + agentAdresses.agentName);
                            } /**/
                        }
                        found = true;
                        break;
                    }
                }
                if (!found) {
                    if (tryBroadcast) {
                        listDestinationAddress.add(null);
                        logger.info("trying broadcast instead of " + destination);
                    } else { /**/
                        failed.add(destination);
                        logger.info("sending fail )" + destination);
                    }
                }
            }
            // send to anonymous
            for (AgentAdresses agentAdresses : anonymousAddresses) {
                Address address = agentAdresses.busAdresses[index];
                if (address != null) {
                    listDestinationAddress.add(address);
                }
            }
        }
        // now trying to find the proper Channel
        LocalAgentChannels localAgentChannels = mapLocalChannels.get(senderAgent);
        if (localAgentChannels == null) {
            throw new IllegalArgumentException("agent " + senderAgent + " not registered on " + bus);
        }
        JChannel channel = localAgentChannels.channels[index];
        if (channel == null) {
            throw new IllegalArgumentException("agent " + senderAgent + " not registered on " + bus);
        }
        try {
            if (listDestinationAddress == null) {
                // versiion 2 channel.send(null, null, (Serializable) message);
                channel.send(null,  (Serializable) message);
            } else {
                boolean nullSent = false;
                for (Address dest : listDestinationAddress) {
                    if (dest == null) {
                        if (nullSent) continue;
                        nullSent = true;
                    }
                    // version 2 channel.send(dest, null, (Serializable) message);
                    channel.send(dest,  (Serializable) message);
                }
            }
        } catch (Exception exc) {
            throw new IOException(exc);
        }
        // throwing exception for failed adresses
        if (failed.size() != 0) {
            //TODO : do it for BusMembershipListener
            throw new DestinationsException(senderAgent, failed.toArray());
        }
    }


    @Override
    public void addMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (agentName == null || "".equals(agentName)) {
            agentName = ANONYMOUS_AGENT;
        }
        if (forwarder == null) {
            throw new IllegalArgumentException("no forwarder");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        LocalAgentChannels localAgentChannels = mapLocalChannels.get(agentName);
        if (localAgentChannels == null) {
            throw new IllegalArgumentException(" agent " + agentName + "not registered");
        }
        for (Bus bus : buses) {
            BusReceiver receiver = localAgentChannels.receivers[bus.ordinal()];
            if (receiver == null) {
                throw new IllegalArgumentException(" agent " + agentName + "not registered on bus " + bus);
            }
            receiver.addForwarder(forwarder);
        }
    }

    @Override
    public void removeMessageListener(String agentName, BusMessageForwarder forwarder, Bus... buses) {
        if (agentName == null || "".equals(agentName)) {
            agentName = ANONYMOUS_AGENT;
        }
        if (forwarder == null) {
            throw new IllegalArgumentException("no forwarder");
        }
        if (buses.length == 0) {
            buses = Bus.values();
        }
        LocalAgentChannels localAgentChannels = mapLocalChannels.get(agentName);
        if (localAgentChannels == null) {
            return;
        }
        for (Bus bus : buses) {
            BusReceiver receiver = localAgentChannels.receivers[bus.ordinal()];
            if (receiver == null) {
                continue;
            }
            receiver.removeForwarder(forwarder);
        }
    }

    @Override
    public void setMembershipListener(BusMembershipListener listener, Bus... buses) {
        if (buses.length == 0) {
            buses = Bus.values();
        }
        for (Bus bus : buses) {
            membershipListeners[bus.ordinal()] = listener;
        }
    }
}
