package org.lsst.ccs.maven.pugin.properties;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import org.apache.maven.model.Profile;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.project.MavenProject;

/**
 * Generate jms jndi properties
 * @goal jms
 */
public class JndiPropertiesFileGeneratorMojo extends AbstractMojo {

    /**
     * @parameter expression="${project}"
     */
    private MavenProject mavenProject;
    /**
     *
     * @parameter expression="${org.lsst.ccs.jms.server.name}" default-value="localhost"
     */
    private String serverName;
    /**
     *
     * @parameter expression="${org.lsst.ccs.jms.server.port}" default-value="3700"
     */
    private String serverPort;

    /**
     *
     * @parameter expression="${org.lsst.ccs.resources.directory}" default-value="${project.build.directory}"
     */
    private String resourcesDirectory;

    private OutputStreamWriter out;

    final static Set<String> supportedServers = new HashSet<String>();
    static {
        supportedServers.add("GlassFish3");
        supportedServers.add("JBoss");
    }

    static final String lineSeparator = System.getProperty ( "line.separator" );

    public void execute() throws MojoExecutionException {



        List<Profile> activeProfiles = getAllActiveProfiles(mavenProject);
        Set activeServerProfile = new HashSet();
        for ( Profile p : activeProfiles ) {
            String id = p.getId();
            if (supportedServers.contains(id)) {
                activeServerProfile.add(id);
            }
        }


        if (activeServerProfile.isEmpty())
            throw new MojoExecutionException("No Profiles were active for the supported servers (JBoss,GlassFish3). No Properties file has been generated ");
        if (activeServerProfile.size() != 1)
            throw new MojoExecutionException("More than one Server profile has been activated: " + activeServerProfile + ". Please make sure to choose only one. No Properties file has been generated ");

        try {
            if ( ! resourcesDirectory.endsWith("/") )
                resourcesDirectory += "/";
            File targetDirectory = new File(resourcesDirectory);
            targetDirectory.mkdirs();
            File outputFile = new File(targetDirectory, "jndi.properties");
            out = new OutputStreamWriter(new FileOutputStream(outputFile));

            getLog().info("Generating output file "+outputFile.toString()+" for "+activeServerProfile.iterator().next()+" server (" + serverName + ":" + serverPort+")");

            if (activeServerProfile.contains("JBoss")) {
                createJBossJndiProperties();
            } else if (activeServerProfile.contains("GlassFish3")) {
                createGlassFishJndiProperties();
            }

            out.flush();
        } catch (Exception ffe) {
            throw new MojoExecutionException("Problem creating jndi.properties file ",ffe);
        }

    }

    void createJBossJndiProperties() throws IOException {
        out.write("java.naming.factory.initial=org.jnp.interfaces.NamingContextFactory"+lineSeparator);
        out.write("java.naming.factory.url.pkgs=org.jboss.naming"+lineSeparator);
        out.write("jboss.host="+serverName+lineSeparator);
        out.write("java.naming.provider.url=jnp://"+serverName+":"+serverPort);
    }

    void createGlassFishJndiProperties() throws IOException {
        out.write("java.naming.factory.initial=com.sun.enterprise.naming.SerialInitContextFactory"+lineSeparator);
        out.write("java.naming.factory.url.pkgs=com.sun.enterprise.naming"+lineSeparator);
        out.write("java.naming.factory.state=com.sun.corba.ee.impl.presentation.rmi.JNDIStateFactoryImpl"+lineSeparator);
        out.write("org.omg.CORBA.ORBInitialHost="+serverName+lineSeparator);
        out.write("org.omg.CORBA.ORBInitialPort="+serverPort);
    }

    public static List<Profile> getAllActiveProfiles(MavenProject project) {
        List<Profile> list = new ArrayList<Profile>();
        getAllActiveProfiles(project,list);
        return list;
    }

    public static void getAllActiveProfiles(MavenProject project, List<Profile> list) {
        List activeProfiles = project.getActiveProfiles();
        for (int i = 0; i < activeProfiles.size(); i++)
            list.add((Profile) activeProfiles.get(i));
        MavenProject parent = project.getParent();
        if ( parent != null )
            getAllActiveProfiles(parent, list);
    }

    /*
     <configuration>
    <target>
    <echo message="Building log4j.properties file in ${project.build.outputDirectory}"/>
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.rootLogger=info, stdout${line.separator}" />
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.appender.stdout=org.apache.log4j.ConsoleAppender${line.separator}" append="true" />
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.appender.stdout.layout=org.apache.log4j.PatternLayout${line.separator}" append="true" />
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.appender.stdout.layout.ConversionPattern=%d{ISO8601} - %m%n${line.separator}" append="true" />
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.logger.net.aubourg=debug${line.separator}" append="true" />
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.logger.lsst=debug${line.separator}" append="true" />
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.logger.lsst.ccs.Subsystem=debug${line.separator}" append="true" />
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.logger.org.springframework.aop=DEBUG${line.separator}" append="true" />
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.logger.org.springframework=DEBUG${line.separator}" append="true" />
    <echo file="${project.build.outputDirectory}/log4j.properties" message="log4j.logger.lsst.ccs.framework=debug${line.separator}" append="true" />
    </target>
    </configuration>
    </execution>
    </executions>
    </plugin>
     */
}
