package org.lsst.ccs.localdb.configdb;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.config.CategoryDescription;
import org.lsst.ccs.config.ConfigurationDescription;
import org.lsst.ccs.config.ConfigurationView;
import org.lsst.ccs.config.SingleCategoryTag;
import org.lsst.ccs.localdb.configdb.model.Description;
import org.lsst.ccs.localdb.configdb.model.Configuration;
import org.lsst.ccs.localdb.configdb.model.ConfigurationParameterValue;
import org.lsst.ccs.localdb.configdb.model.ConfigurationRun;
import org.lsst.ccs.localdb.configdb.model.BaseDescription;
import org.lsst.ccs.localdb.configdb.model.GlobalConfiguration;
import org.lsst.ccs.localdb.configdb.model.HibernateDAO;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.localdb.configdb.model.ConfigurationInfoData;
import org.lsst.ccs.localdb.configdb.model.ConfigurationParameter;
import org.lsst.ccs.localdb.configdb.model.ParameterPath;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Configuration database high level facade. Each of its public methods is
 * responsible for opening a session, starting a transaction and committing it,
 * and must perform a rollback in case of a failure.
 *
 * @author LSST CCS Team
 */
public class ConfigurationFacade {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.configdb");
    
    private final HibernateDAO dbInterface;
    
    public ConfigurationFacade(HibernateDAO dbInterface) {
        this.dbInterface = dbInterface;
    }
    
    /**
     * Finds or registers in database the agent description corresponding to the
     * given {@link org.lsst.ccs.bus.data.ConfigurationInfo} object.
     *
     * @param ad the agent description
     * @param cpis the list of configuration parameter info.
     * @return the id of the corresponding {@code BaseDescription} entity
     * @throws java.lang.Exception
     */
    public Description registerDescription(AgentDesc ad, List<ConfigurationParameterInfo> cpis) throws Exception {
        try {
            
            Description res = null;
            
            dbInterface.begin();
            Map<ParameterPath, List<ConfigurationParameter>> parameters = dbInterface.getAllConfigurationParameters(ad.getAgentName());
            List<ConfigurationParameter> actualParameters = new ArrayList<>();
            for(ConfigurationParameterInfo cpi : cpis) {
                ConfigurationParameter cp = new ConfigurationParameter(ad.getAgentName(),
                        new ParameterPath(cpi.getComponentName(), cpi.getParameterName()), 
                        cpi.getActualType().getTypeName(), 
                        cpi.getDescription(), 
                        cpi.getCategoryName(),
                        cpi.isFinal() ? cpi.getConfiguredValue() : null);
                ParameterPath pp = new ParameterPath(cpi.getComponentName(), cpi.getParameterName());
                boolean found = false;
                if(parameters.containsKey(pp)) {
                    for(ConfigurationParameter parm : parameters.get(pp)) {
                        if(parm.equals(cp)) {
                            // it does not need to be created
                            actualParameters.add(parm);
                            found = true;
                            break;
                        }
                    }
                }
                if (!found) {
                    actualParameters.add(dbInterface.createConfigurationParameter(
                        ad.getAgentName(), 
                        cpi.getComponentName(), 
                        cpi.getParameterName(), 
                        cpi.getDescription(), 
                        cpi.getActualType().getTypeName(), 
                        cpi.getCategoryName(), 
                        cpi.isFinal() ? cpi.getConfiguredValue() : null));
                }
            }
            
            res =  dbInterface.getDescription(ad, actualParameters);
            
            dbInterface.end();
            log.debug("registered description for " + ad.getAgentName() + " : " +res.getId());
            return res;
        } catch (Exception ex) {
            dbInterface.rollback();
            throw ex;
        } finally {
            dbInterface.closeSession();
        }
    }
    
    public void stopConfigurationRuns(Description desc, long time) throws Exception {
        try {
            dbInterface.begin();
            Map<String, ConfigurationRun> activeRuns = dbInterface.getActiveConfigurationRuns(desc);
            for (ConfigurationRun cr : activeRuns.values()) {
                cr.setTstop(time);
            }
            dbInterface.end();
        } catch (Exception ex) {
            dbInterface.rollback();
            throw ex;
        } finally {
            dbInterface.closeSession();
        }
    }
    
    /**
     * Registration of configuration information coming from the Command bus.
     * There is no associated StatusConfigurationInfo yet.
     *
     * @param desc
     * @param ci
     * @return
     * @throws java.lang.Exception
     */
    public ConfigurationDescription registerStandaloneConfiguration(Description desc, ConfigurationInfo ci) throws Exception {
        try {
            dbInterface.begin();
            Map<String, ConfigurationRun> newRuns = registerRuns(desc, ci, ci.getCCSTimeStamp().getUTCInstant().toEpochMilli());
            
            ConfigurationDescription cd = new ConfigurationDescription(ci.getCategorySet());
            
            for (ConfigurationRun  cr : newRuns.values()) {
                cd.putTagForCategory(cr.getConfiguration().getCategory(), cr.getConfiguration().getConfigName(), CategoryDescription.DEFAULT_V_STRING);
            }
            dbInterface.end();
            return cd;
        } catch (Exception ex) {
            dbInterface.rollback();
            throw ex;
        } finally {
            dbInterface.closeSession();
        }
    }
    
    /**
     * This methods resolves the association between the ConfigurationInfo
     * registered previously via the command bus and the subsequent
     *
     * @param desc the subsystem description
     * @param cid the ConfigurationInfoData to persist
     */
    public void registerStandaloneStatusConfigurationInfo(Description desc, ConfigurationInfoData cid) throws Exception {
        try {
            dbInterface.begin();
            Map<String, ConfigurationRun> currentRuns = dbInterface.getActiveConfigurationRuns(desc);
            cid.setConfigurationRuns(currentRuns);
            dbInterface.persist(cid);
            dbInterface.end();
        } catch (Exception ex) {
            dbInterface.rollback();
            throw ex;
        } finally {
            dbInterface.closeSession();
        }
    }
    
    /**
     * Registration of an incoming {@code ConfigurationInfo} object via the Status Bus. The
     * registration happens by comparing the received {@code ConfigurationInfo}
     * with the previous configuration state persisted in the current
     * {@code ConfigurationRun}
     *
     * @param desc
     * @param ci
     * @param cid the associated ConfigurationInfoData entity
     * 
     */
    public void registerConfiguration(Description desc, ConfigurationInfo ci, ConfigurationInfoData cid) throws Exception {
        try {
            long time = cid.getTime();
            cid.setDescription(desc);
            dbInterface.begin();
            Map<String, ConfigurationRun> newRuns = registerRuns(desc, ci, time);
            // Taking care of the global configuration
            
            cid.setConfigurationRuns(newRuns);
            dbInterface.persist(cid);
            log.fine("persisted configuration state at : " + cid.getTime() + " : " + cid.getConfigurationDescriptionString());
            dbInterface.end();
        } catch (Exception ex) {
            dbInterface.rollback();
            throw ex;
        } finally {
            dbInterface.closeSession();
        }
    }
    
    private Map<String, ConfigurationRun> registerRuns(Description desc, ConfigurationInfo ci, long time) throws Exception {
            Map<String, ConfigurationRun> currentRuns = dbInterface.getActiveConfigurationRuns(desc);
            log.debug("fetched " + currentRuns.size() + " active runs for " + desc.getAgentDesc().getAgentName());
            Map<String, ConfigurationRun> newRuns = new HashMap<>();
            Map<String, List<ConfigurationParameterInfo>> cpiMap = ConfigurationInfo.getParameterInfoGroupByCategory(ci.getAllParameterInfo());
            
            Map<String, Configuration> newConfigs = new HashMap<>();
            
            // First pass : finding the base configurations
            for (String cat : ci.getCategorySet()) {
                
                //If we ever work on this code again this will have to be fixed!
                throw new RuntimeException("This code must be fixed");
//                String nextTag = ci.getConfigNameForCategory(cat);
//                Integer version = ci.getConfigVersion(cat);
//                newConfigs.put(cat, dbInterface.getConfigurationOrCreate(desc.getBaseDescription(), nextTag, cat, version, cpiMap.get(cat), time, currentRuns.get(cat)));
            }
            
            // Finding the globalConfiguration
            GlobalConfiguration gc = null;
            // The global configuration information is only set when the state is CONFIGURED (no run time changes)
//            if (ci.getGlobalName() != null && !ci.hasChanges()) {
//                gc = dbInterface.findGlobalConfigurationOrCreate(desc.getBaseDescription(), ci.getGlobalName(), newConfigs);
//            }
            
            // Second pass : create and populate the configuration runs
            for (String cat : ci.getCategorySet()) {
                Configuration baseConfig = newConfigs.get(cat);
                ConfigurationRun newRun = new ConfigurationRun(desc, baseConfig, time);
                newRun.setGlobalConfiguration(gc);
                ConfigurationRun currentRun = currentRuns.get(cat);
                List<ConfigurationParameterInfo> changesForCat = cpiMap.get(cat)
                        .stream().filter(
                                (cpi) -> {
                                    // Getting rid of final and non dirty parameters
                                    return !cpi.isFinal() && cpi.isDirty();
                                }
                        ).collect(Collectors.toList());
                
                if (currentRun == null) {
                    /** there is no previous run to compare with. */
                    newRun = dbInterface.populateRunFromScratch(newRun, changesForCat);
                } else {
                    if (time <= currentRun.getTstart()) {
                        if(time < currentRun.getTstart()) {
                            log.info("for " + desc.getAgentDesc().getAgentName() + ": received configuration info time ("+new Date(time)+") is previous to an already persisted configuration state at " + new Date(currentRun.getTstart()));
                        }
                        newRun = currentRun;
                    } else {
                        newRun.setDescription(currentRun.getDescription());
                        newRun = dbInterface.populateRunFromPreviousRun(currentRun, newRun, changesForCat);
                    }
                }
                
                newRuns.put(cat, newRun);
            }
            return newRuns;
    }
    
    public ConfigurationView loadConfiguration(BaseDescription desc, ConfigurationDescription configDesc) throws Exception {
        try {
            ConfigurationView view = new ConfigurationView(configDesc);
            dbInterface.begin();
            try {
                for (String cat : configDesc.getCategoriesSet() ) {
                    CategoryDescription catTag = configDesc.getCategoryTag(cat);
                    List<SingleCategoryTag> sourcedTags = catTag.getSingleCategoryTags();
                    for ( SingleCategoryTag sourcedTag: sourcedTags ) {
                        Configuration request = dbInterface.getConfigurationOrNull(desc, cat, sourcedTag.getTag(), Integer.valueOf(sourcedTag.getVersion()));
                        if (request == null) {
                            throw new RuntimeException("cannot find configuration \"" + cat + "\":\"" + sourcedTag.getTag() + "\"(" + sourcedTag.getVersion() + ")");
                        }
                        for (ConfigurationParameterValue cpv : request.getConfigurationParameterValues().values()) {
                            view.putParameterValue(cpv.getPath().getComponentName(), cpv.getPath().getParameterName(), cpv.getValue());
                        }
                    }
                }
            } finally {
                dbInterface.end();
            }
            return view;
        } catch (Exception ex) {
            dbInterface.rollback();
            throw ex;
        } finally {
            dbInterface.closeSession();
        }
    }
    
    public ConfigurationView loadGlobalConfiguration(BaseDescription desc, String name, int version) throws Exception {
        try {
            dbInterface.begin();
            GlobalConfiguration gc = dbInterface.findGlobalConfiguration(desc, name, version);
            ConfigurationView res = null;
            if (gc != null) {
                ConfigurationDescription cd = new ConfigurationDescription(gc.getConfigurations().keySet());
                res = new ConfigurationView(cd);
                for(Configuration c : gc.getConfigurations().values()) {
                    cd.putTagForCategory(c.getCategory(), c.getCategory(), CategoryDescription.DEFAULT_V_STRING);
                    for (ConfigurationParameterValue cpv : c.getConfigurationParameterValues().values()) {
                        res.putParameterValue(cpv.getPath().getComponentName(), cpv.getPath().getParameterName(), cpv.getValue());
                    }
                }
            }
            dbInterface.end();
            return res;
        } catch (Exception ex) {
            dbInterface.rollback();
            throw ex;
        } finally {
            dbInterface.closeSession();
        }
    }
    
    // Read only operation
    public Set<String> findAvailableConfigurationsForCategory(BaseDescription desc, String category) throws Exception {
        try {
            Set<String> res = new HashSet<>();
            dbInterface.begin();
            res.addAll(dbInterface.findAvailableConfigurationsForCategory(desc, category));
            dbInterface.end();
            return res;
        } catch (Exception ex) {
            dbInterface.rollback();
            throw ex;
        } finally {
            dbInterface.closeSession();
        }
    }
    
    void cleanupConfigurationRuns(Description desc, long time) throws Exception {
        try {
            dbInterface.begin();
            dbInterface.cleanupConfigurationRuns(desc, time);
            dbInterface.end();
        } catch (Exception ex) {
            dbInterface.rollback();
            throw ex;
        } finally {
            dbInterface.closeSession();
        }
    }
}
