package org.lsst.ccs.localdb.configdb;

import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import org.influxdb.dto.Point;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.config.ConfigurationDescription;
import org.lsst.ccs.config.ConfigurationServiceException;
import org.lsst.ccs.config.ConfigurationView;
import org.lsst.ccs.config.PackCst;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.configdb.model.Description;
import org.lsst.ccs.localdb.configdb.model.HibernateDAO;
import org.lsst.ccs.localdb.configdb.model.ConfigurationInfoData;
import org.lsst.ccs.localdb.statusdb.AgentPresenceEvent;
import org.lsst.ccs.localdb.statusdb.LocalDBAlert;
import org.lsst.ccs.localdb.statusdb.StatusDataPersister;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.services.InfluxDbClientService;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * A CCS component that stores Configuration information in two different ways,
 * depending on an agent's chosen protocol.
 * <UL>
 * <LI/> Listening on the status bus
 * <LI/> Responding to configuration commands on the command bus
 * </UL>
 *
 * @author LSST CCS Team
 */
public class ConfigurationPersister implements /*ConfigurationDAO,*/ HasLifecycle  {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.configdb");
    
    private ConfigurationFacade facade;
    
    /** A map of alive agents to their running description. */
    private final Map<String, Description> aliveDescriptions = new HashMap<>();
    
    /** Synchronization between Status and Command bus when the persister is in active mode for an agent. */
    private final Map<String, CountDownLatch> descriptionReady = new HashMap<>();
    
    @LookupField(strategy = Strategy.TOP)
    private Agent s;
    
    @LookupName
    private String name;

    @LookupField(strategy = Strategy.TREE)
    private AlertService alertService;
    
    @LookupField(strategy=Strategy.TREE)
    AgentPeriodicTaskService periodicTaskService;

    @LookupField(strategy=Strategy.TREE)
    StatusDataPersister statusDataPersister;
    
    @LookupField(strategy=Strategy.TREE)
    InfluxDbClientService influxDbClientService;
    
    private final BlockingQueue<Object[]> ciQueue = new ArrayBlockingQueue<>(1000);
    
    
    private class ConfigurationInfoPersister implements Runnable {

        @Override
        public void run() {
            long tstart = System.currentTimeMillis();
            int nProcessed = 0;
            while (!ciQueue.isEmpty()) {
                try {
                    Object[] o = ciQueue.take();
                    nProcessed ++;
                    Object key = o[0];
                    if (key instanceof ConfigurationInfoData) {
                        ConfigurationInfoData cid = (ConfigurationInfoData) key;
                        String name = cid.getAgentDesc().getAgentName();
                        StatusConfigurationInfo sci = (StatusConfigurationInfo) o[1];
                        Description desc = aliveDescriptions.get(name);
                        if (desc == null) {
                            // Initialization phase
                            processFirstStatusConfigurationInfo(sci, cid);
                        } else {
                            if (descriptionReady.get(name) != null) {
                                // active mode
                                descriptionReady.get(name).countDown();
                                facade.registerStandaloneStatusConfigurationInfo(aliveDescriptions.get(name), cid);
                            } else {
                                // passive mode
                                log.fine("received configuration state at : " + sci.getCCSTimeStamp().getUTCInstant().toEpochMilli()+ " : " + sci.getConfigurationInfo().getConfigurationDescription());
                                facade.registerConfiguration(aliveDescriptions.get(name), sci.getConfigurationInfo(), cid);
                            }
                        }
                    } else if (key instanceof AgentPresenceEvent) {
                        AgentPresenceEvent ape = (AgentPresenceEvent) key;
                        AgentInfo agent = ape.getAgent();
                        if (ape.isConnecting()) {
                            String remote = agent.getAgentProperty(PackCst.IS_CONFIG_REMOTE);
                            if ("true".equals(remote)) {
                                log.finest("got connection event from " + agent.getName());
                                descriptionReady.put(agent.getName(), new CountDownLatch(1));
                            }
                        } else {
                            log.finest("got disconnection event from " + agent.getName());
                            Description desc = aliveDescriptions.remove(agent.getName());
                            if (desc != null) {
                                // setting a stop date to the current runs
                                facade.stopConfigurationRuns(desc, ape.getTime());
                            }
                        }
                    } else {
                        log.info("configuration persister received unexpected object " + key.getClass());
                    }
                } catch (InterruptedException ex) {
                    log.debug("configuration info serializer thread interrupted", ex);
                    return;
                } catch (Exception ex) {
                    log.severe("caught exception when persisting", ex);
                    alertService.raiseAlert(LocalDBAlert.ConfigurationPersistenceException.getAlert(null, ex), AlertState.WARNING, LocalDBAlert.getFirstException(ex));
                } 
            }
            if(nProcessed > 0) {                
                long time = System.currentTimeMillis() - tstart;
                if (influxDbClientService.getInfluxDbClient() != null) {
                    Point.Builder batchPersisterPointBuilder = Point.measurement("db_persist")
                            .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);

                    batchPersisterPointBuilder = batchPersisterPointBuilder.
                            addField("time_tot", time).
                            addField("count", nProcessed).
                            addField("time_avg", (double) time / (double) nProcessed);
                    Point point = batchPersisterPointBuilder.tag("persister", name).tag(influxDbClientService.getGlobalTags()).build();
                    influxDbClientService.getInfluxDbClient().write(point);
                } else {
                    KeyValueDataList kvdl = new KeyValueDataList(name);
                    kvdl.addData(name + "/transactionTime", time);
                    kvdl.addData(name + "/entityAverageTime", (double) time / nProcessed);
                    statusDataPersister.processEncodedData(s.getName(), kvdl);
                }
            }
        }
    }
    
    @Override
    public void build() {
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("configuration-accumulator", new ConfigurationInfoPersister()
        ).withIsFixedRate(false).withPeriod(Duration.ofSeconds(1)));
    }

    @Override
    public void init() {
        s.getAgentService(AgentPropertiesService.class).setAgentProperty(PackCst.CONFIG_DB, "true");
    }
    
    @Override
    public void start() {
        facade = new ConfigurationFacade(new HibernateDAO(StatusdbUtils.getSessionFactory()));
    }
    

    /** Read only operations. */
    /**
     * 
     * @param agentName
     * @param configDescription
     * @return
     * @throws ConfigurationServiceException 
     */
    @Command
//    @Override
    public ConfigurationView loadConfiguration(String agentName, ConfigurationDescription configDescription) throws ConfigurationServiceException {
        try {
            return facade.loadConfiguration(descriptionWaitAndGet(agentName).getBaseDescription(),configDescription);
        } catch (Exception ex) {
            throw new ConfigurationServiceException("error at persistence layer", ex);
        }
    }
    
    @Command
//    @Override
    public ConfigurationView loadGlobalConfiguration(String agentName, String name, int version) {
        try {
            return facade.loadGlobalConfiguration(descriptionWaitAndGet(agentName).getBaseDescription(), name, version);
        } catch (Exception ex) {
            throw new ConfigurationServiceException("error at persistence layer", ex);
        }
    }

    @Command
//    @Override
    public Set<String> findAvailableConfigurationsForCategory(String agentName, String category) {
        try {
            return facade.findAvailableConfigurationsForCategory(descriptionWaitAndGet(agentName).getBaseDescription(), category);
        } catch (Exception ex) {
            throw new ConfigurationServiceException("error at persistence layer", ex);
        }
    }

    @Command
//    @Override
    public boolean isAvailable() {
        return true;
    }

    /** Read-write operations. */
    /**
     * @param agentName
     * @param configInfo
     * @return the resulted configuration versions.
     * @throws ConfigurationServiceException 
     */
    @Command
//    @Override
    public ConfigurationDescription registerConfiguration(String agentName, ConfigurationInfo configInfo) throws ConfigurationServiceException {
        try {
            return facade.registerStandaloneConfiguration(descriptionWaitAndGet(agentName), configInfo);
        } catch (Exception ex) {
            throw new ConfigurationServiceException("error at persistence layer", ex);
        }
    }
    
//    @Override
    public ConfigurationDescription saveChangesForCategoriesAs(String agentName, ConfigurationDescription configDesc, ConfigurationInfo configInfo) throws ConfigurationServiceException {
        return null;
        // empty implementation
    }
    
    private Description descriptionWaitAndGet(String agentName) {
        try {
            descriptionReady.get(agentName).await();
            return aliveDescriptions.get(agentName);
        } catch (Exception ex) {
            throw new RuntimeException("cannot get description for " + agentName, ex);
        }
    }
    
    private void processFirstStatusConfigurationInfo(StatusConfigurationInfo sci, ConfigurationInfoData cid) throws Exception {
        // Registering the description
        ConfigurationInfo ci = sci.getConfigurationInfo();
        String agentName = cid.getAgentDesc().getAgentName();
        Description desc = facade.registerDescription(cid.getAgentDesc(), ci.getAllParameterInfo());
        log.finest("registered description for " + agentName + " with id " + desc.getId());
        // Initiates the configuration runs
        facade.cleanupConfigurationRuns(desc, cid.getTime());
        facade.registerConfiguration(desc, ci, cid);
        log.finest("registered initial configuration info : " + ci.getConfigurationDescription() + " at " + sci.getCCSTimeStamp().getUTCInstant().toEpochMilli());
        // Keeping the description in memory as long as the agent in on the buses.
        aliveDescriptions.put(agentName, desc);
    }
    
    public void submit(Object[] objs) {
        ciQueue.offer(objs);
    }
}
