package org.lsst.ccs.localdb.statusdb;

import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.hibernate.FlushMode;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.resource.transaction.spi.TransactionStatus;
import org.influxdb.dto.Point;
import org.influxdb.dto.Point.Builder;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.services.InfluxDbClientService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Persists entities as a batch.
 *
 * @author LSST CCS Team
 */
public abstract class BatchPersister<T> implements Runnable, ClearAlertHandler {

    protected static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.statusdb");

    @LookupName
    private String persisterName;
    private static final int CONSECUTIVE_MAX_CAPACITY_LIMIT = 15;

    /**
     * The maximum number of entities to process in a single transaction.
     */
    private final int nMax;

    private final boolean flushAtCommit;

    private final Queue<T> rq = new ConcurrentLinkedQueue<>();

    @LookupField(strategy = Strategy.TREE)
    protected AlertService alertService;
    
    @LookupField(strategy=Strategy.TOP)
    private Agent a;

    @LookupField(strategy=Strategy.TREE)
    StatusDataPersister statusDataPersister;

    @LookupField(strategy=Strategy.TREE)
    InfluxDbClientService influxDbClientService;

    private int nThreads = 1;

    public static ExecutorService newFixedThreadPoolBoundedQueue(int nThreads, int nQueue) {
        LinkedBlockingQueue queue = nQueue > 0 ? new LinkedBlockingQueue<>(nQueue) : new LinkedBlockingQueue<>();
        return new ThreadPoolExecutor(nThreads, nThreads, 0, TimeUnit.MILLISECONDS, queue);
    }

    /**
     * Constructor for a batch persister.
     *
     * @param nMax
     *            The maximum number of entities to persist in a single
     *            transaction.
     * @param flushAtCommit
     *            true if the session that process the batch should be set to
     *            FlushMode.COMMIT
     * @param threads
     *            database feeding threads
     * @param threadQueueSize
     *            The size of the queue; if negative the queue is unbound.
     */
    public BatchPersister(int nMax, boolean flushAtCommit, int threads, int threadQueueSize) {        
        this.nMax = nMax;
        this.flushAtCommit = flushAtCommit;
        this.nThreads = threads;
        exec = newFixedThreadPoolBoundedQueue(nThreads, threadQueueSize);
    }


    /**
     * Constructor for a batch persister.
     *
     * @param nMax
     *            The maximum number of entities to persist in a single
     *            transaction.
     * @param flushAtCommit
     *            true if the session that process the batch should be set to
     *            FlushMode.COMMIT
     * @param threads
     *            database feeding threads
     */
    public BatchPersister(int nMax, boolean flushAtCommit, int threads) {
        this(nMax, flushAtCommit, threads, 10*threads);
    }

    /**
     * Constructor for a batch persister.
     *
     * @param nMax
     *            The maximum number of entities to persist in a single
     *            transaction.
     * @param flushAtCommit
     *            true if the session that process the batch should be set to
     *            FlushMode.COMMIT
     */
    public BatchPersister(int nMax, boolean flushAtCommit) {
        this(nMax, flushAtCommit, 1);
    }

    @Override
    public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
        return ClearAlertCode.CLEAR_ALERT;
    }

    private void persistBatch(List<T> batch) {
        log.debug("start batch " + this.getClass().getName() + " " + this + " " + batch.size());
        long tstart = System.currentTimeMillis();
        Session sess = StatusdbUtils.getSessionFactory(null).openSession();
        if (flushAtCommit) {
            sess.setFlushMode(FlushMode.COMMIT);
        }
        Transaction tx = null;
        try {
            tx = sess.beginTransaction();
            for (T t : batch) {
                persist(t, sess);
            }
            tx.commit();
        } catch (Exception ex) {
            log.error("caught exception when persisting", ex);
            alertService.raiseAlert(LocalDBAlert.BatchException.getAlert(persisterName, ex), AlertState.WARNING,
                    LocalDBAlert.getFirstException(ex));
            try {
                if (tx != null && (tx.getStatus() == TransactionStatus.ACTIVE
                        || tx.getStatus() == TransactionStatus.MARKED_ROLLBACK)) {
                    tx.rollback();
                }
            } catch (Exception rbEx) {
                log.error("Rollback of transaction failed : " + rbEx, rbEx);
                alertService.raiseAlert(LocalDBAlert.BatchRollbackException.getAlert(persisterName, ex),
                        AlertState.WARNING, LocalDBAlert.getFirstException(ex));
            }
        } finally {
            if (sess.isOpen()) {
                sess.close();
            }
        }

        long time = System.currentTimeMillis() - tstart;
        if (influxDbClientService != null) {
            Builder batchPersisterPointBuilder = Point.measurement("db_persist").time(System.currentTimeMillis(),
                    TimeUnit.MILLISECONDS);

            batchPersisterPointBuilder = batchPersisterPointBuilder.addField("btime_tot", time)
                    .addField("bcount", batch.size()).addField("btime_avg", (double) time / (double) batch.size());
            Point point = batchPersisterPointBuilder.tag("persister", persisterName).tag(influxDbClientService.getGlobalTags()).build();
            influxDbClientService.getInfluxDbClient().write(point);
        } else {
            KeyValueDataList kvdl = new KeyValueDataList(persisterName);
            kvdl.addData(persisterName + "/transactionTime", time);
            kvdl.addData(persisterName + "/batchSize", batch.size());
            kvdl.addData(persisterName + "/entityAverageTime", (double) time / batch.size());
            statusDataPersister.processEncodedData(a.getName(), kvdl);
        }

        log.debug("end batch ");
    }

    ExecutorService exec;

    @Override
    public void run() {
        log.fine("start run " + rq.size());
        int n = 0;
        int nAcc = 0;
        boolean done = false;
        long tstart = System.currentTimeMillis();
        int queueAtStart = rq.size();
        int tasksAtStart = ((ThreadPoolExecutor) exec).getQueue().size();
        int nbatch = 0;
        while (!done) {
            List<T> batch = new ArrayList<T>(nMax);
            // get at most nMax objets from the queue
            for (int i = 0; i < nMax; i++) {
                T t = rq.poll();
                n = i;
                if (t == null) {
                    done = (i == 0);
                    break;
                } else {
                    batch.add(t);
                }
            }
            if (batch.size() > 0) {
                try {
                    exec.submit(() -> persistBatch(batch));
                } catch (RejectedExecutionException e) {
                    log.warn("max capacity, queue size " + rq.size() + " thread queue size "
                            + ((ThreadPoolExecutor) exec).getQueue().size(), e);
                    alertService.raiseAlert(LocalDBAlert.BatchMaxCapacity.getAlert(persisterName, null),
                            AlertState.WARNING, "current queue size : " + rq.size() + " thread queue size "
                                    + ((ThreadPoolExecutor) exec).getQueue().size());
                    //Put the batch into the queue for reprocessing 
                    rq.addAll(batch);
                }
                nbatch++;
                done = (rq.size() < nMax / 5); // too few objects left for a
                                               // batch, we will wait for the
                                               // next round
                if (done)
                    log.fine("run done, left in queue " + rq.size());
            }

            if (n > 0) {
                nAcc += n;
                log.debug(this.getClass().getSimpleName() + " processed " + n + " entities.");
            }
        }
        if (nAcc > 0)

        {
            long time = System.currentTimeMillis() - tstart;
            if ( influxDbClientService != null  ) {
                Builder batchPersisterPointBuilder = Point.measurement("db_persist")
                        .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);

                batchPersisterPointBuilder = batchPersisterPointBuilder.
                        addField("time_tot", time).
                        addField("count", nAcc).
                        addField("time_avg", (double)time/(double)nAcc);
                Point point = batchPersisterPointBuilder.tag("persister", persisterName).tag(influxDbClientService.getGlobalTags()).build();
                influxDbClientService.getInfluxDbClient().write(point);                
            } else {
                KeyValueDataList kvdl = new KeyValueDataList(persisterName);
                kvdl.addData(persisterName + "/queueProcessingTime", time);
                kvdl.addData(persisterName + "/queueSize", queueAtStart);
                kvdl.addData(persisterName + "/nBatchInRun", nbatch);
                kvdl.addData(persisterName + "/activeThreads", ((ThreadPoolExecutor) exec).getActiveCount());
                kvdl.addData(persisterName + "/queuedTasks", tasksAtStart);
                log.fine("queued tasks " + tasksAtStart + " " + ((ThreadPoolExecutor) exec).getQueue().size());
                statusDataPersister.processEncodedData(a.getName(), kvdl);
            }
        }
        log.fine("end run ");
    }

    public void addToQueue(T obj) {
        rq.add(obj);
    }

    /**
     * Called from inside an open transaction.
     *
     * @param obj
     * @param sess
     */
    public abstract void persist(T obj, Session sess);

}
