package org.lsst.ccs.localdb.statusdb;

import java.util.concurrent.TimeUnit;
import java.util.logging.Logger;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.stat.EntityStatistics;
import org.hibernate.stat.NaturalIdCacheStatistics;
import org.hibernate.stat.QueryStatistics;
import org.hibernate.stat.SecondLevelCacheStatistics;
import org.hibernate.stat.Statistics;
import org.influxdb.dto.BatchPoints;
import org.influxdb.dto.Point;
import org.influxdb.dto.Point.Builder;
import org.lsst.ccs.services.InfluxDbClientService;

/**
 * Hibernate cache statistics to be stored.
 *
 * @author LSST CCS team
 */
public class CacheStatistics {

    private static final Logger LOG = Logger.getLogger(CacheStatistics.class.getName());

    private static final String[] NAMED_QUERIES = new String[]{
        "findAlertDesc", "findInnerStateDesc", "findDataDesc", "findStatData", "findStatDesc", "findDataGroup", "findStateBundleDesc"};

    public CacheStatistics(SessionFactory fac) {
        this(fac, null);
    }

    public CacheStatistics(SessionFactory fac, InfluxDbClientService influxDb) {
        if (influxDb.isEnabled()) {

            Statistics stats = fac.getStatistics();

            if (stats.isStatisticsEnabled()) {
                // named queries
                try ( Session sess = fac.openSession()) {

                    BatchPoints bp = influxDb.createBatchPoints();
                    // named queries
                    for (String query : NAMED_QUERIES) {
                        QueryStatistics queryStats = stats.getQueryStatistics(sess.getNamedQuery(query).getQueryString());

                        Builder namedQueriesPointBuilder = Point.measurement("db_named_queries")
                                .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);
                        namedQueriesPointBuilder = namedQueriesPointBuilder.
                                addField("cache_hit", queryStats.getCacheHitCount()).
                                addField("cache_miss", queryStats.getCacheMissCount()).
                                addField("cache_put", queryStats.getCachePutCount()).
                                addField("exec_time_avg", queryStats.getExecutionAvgTime()).
                                addField("exec_time_max", queryStats.getExecutionMaxTime()).
                                addField("exec_time_min", queryStats.getExecutionMinTime()).
                                addField("exec_count", queryStats.getExecutionCount()).
                                addField("exec_row_count", queryStats.getExecutionRowCount());

                        Point point = namedQueriesPointBuilder.tag("query_name", query).tag(influxDb.getGlobalTags()).build();
                        bp.point(point);
                    }

                    // Second Level Cache
                    for (String slcName : stats.getSecondLevelCacheRegionNames()) {
                        if (!slcName.startsWith("org.lsst.ccs")) {
                            continue;
                        }
                        SecondLevelCacheStatistics slcStats = stats.getSecondLevelCacheStatistics(slcName);
                        String shortName = slcName.substring(slcName.lastIndexOf(".") + 1);
                        Builder secondLevelCachePointBuilder = Point.measurement("db_second_level_cache")
                                .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);

                        secondLevelCachePointBuilder = secondLevelCachePointBuilder.
                                addField("cache_hit", slcStats.getHitCount()).
                                addField("cache_miss", slcStats.getMissCount()).
                                addField("cache_put", slcStats.getPutCount());

                        Point point = secondLevelCachePointBuilder.tag("class_name", shortName).tag(influxDb.getGlobalTags()).build();
                        bp.point(point);
                    }

                    // Cached Entities
                    for (String entityName : stats.getEntityNames()) {
                        if (!entityName.startsWith("org.lsst.ccs")) {
                            continue;
                        }
                        String shortName = entityName.substring(entityName.lastIndexOf(".") + 1);
                        EntityStatistics entityStats = stats.getEntityStatistics(entityName);

                        Builder cachedEntitiesPointBuilder = Point.measurement("db_cached_entities")
                                .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);
                        cachedEntitiesPointBuilder = cachedEntitiesPointBuilder.
                                addField("delete", entityStats.getDeleteCount()).
                                addField("fetch", entityStats.getFetchCount()).
                                addField("insert", entityStats.getInsertCount()).
                                addField("load", entityStats.getLoadCount()).
                                addField("failure", entityStats.getOptimisticFailureCount()).
                                addField("update", entityStats.getUpdateCount());

                        Point point = cachedEntitiesPointBuilder.tag("class_name", shortName).tag(influxDb.getGlobalTags()).build();
                        bp.point(point);
                    }

                    // Cached Entities
                    for (String entityName : stats.getEntityNames()) {
                        if (!entityName.startsWith("org.lsst.ccs") || !entityName.endsWith("##NaturalId")) {
                            continue;
                        }
                        String shortName = entityName.substring(entityName.lastIndexOf(".") + 1).replace("##NaturalId", "");
                        NaturalIdCacheStatistics natIdStats = stats.getNaturalIdCacheStatistics(entityName);

                        Builder naturalIdPointBuilder = Point.measurement("db_natural_id")
                                .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);
                        naturalIdPointBuilder = naturalIdPointBuilder.
                                addField("hit", natIdStats.getHitCount()).
                                addField("miss", natIdStats.getMissCount()).
                                addField("put", natIdStats.getPutCount()).
                                addField("count_mem", natIdStats.getElementCountInMemory());

                        Point point = naturalIdPointBuilder.tag("class_name", shortName).tag(influxDb.getGlobalTags()).build();
                        bp.point(point);
                    }

                    Builder sessionPointBuilder = Point.measurement("db_session")
                            .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);

                    sessionPointBuilder = sessionPointBuilder.
                            addField("connect", stats.getConnectCount()).
                            addField("flush", stats.getFlushCount()).
                            addField("sess_open", stats.getSessionOpenCount()).
                            addField("sess_close", stats.getSessionCloseCount()).
                            addField("transact", stats.getTransactionCount()).
                            addField("transact_success", stats.getSuccessfulTransactionCount());
                    Point point = sessionPointBuilder.tag(influxDb.getGlobalTags()).build();
                    bp.point(point);
                    stats.clear();
                    
                    influxDb.write(bp);
                }
            }
        }
    }
}
