package org.lsst.ccs.localdb.statusdb;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.hibernate.FlushMode;

import org.hibernate.LockMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.DataProviderInfo.Attribute;
import org.lsst.ccs.bus.data.DataProviderInfo.Type;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusData;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.DataDescLastUpdate;
import org.lsst.ccs.localdb.statusdb.model.DataGroup;
import org.lsst.ccs.localdb.statusdb.model.DataPath;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.services.DataProviderDictionaryService;

/**
 * StatusDataPersister object. Implements StatusListener and updates the db.
 * <p/>
 * Can be run inside a StatusPersisterSubsystem wrapper.
 * <p/>
 * Deprecated use: inside a Message Driver Bean (tied to JMS implementation)
 * 
 * TODO this class could be split, to separate the StatusMessageListener functions
 * and put at the same level the three accumulators (status, stat, metadata).
 *
 * @author aubourg
 */
public class StatusDataPersister extends FastBatchPersister<Object[]> implements StatusMessageListener, AgentPresenceListener, DataProviderDictionaryService.DataProviderDictionaryListener {

    private final Logger ignore_log = Logger.getLogger(getClass().getCanonicalName()+".ignore");
    private static final DataDesc NULL_DESCRIPTION = new DataDesc();
    private static final long[] DEFAULT_TIME_BIN_WIDTH = new long[] { 300000, 1800000 }; // 5
                                                                                         // and
                                                                                         // 30
                                                                                         // minutes
    @LookupField(strategy=Strategy.TOP)
    private Agent subsys;
        
    private final Map<DataPath, DataDesc> map = new ConcurrentHashMap<>();
  
    @LookupField(strategy=Strategy.TREE)
    private StatDataAccumulator statDataAccumulator;
    
    @LookupField(strategy=Strategy.TREE)
    private MetaDataPersister mdPersister;

    @LookupField(strategy=Strategy.TREE)
    private DataProviderDictionaryService dictionaryService;

    private final Set<String> dictionaryProcessed = new CopyOnWriteArraySet<>();
    private final Map<String,Set<String>> noDictionaryProcessedWarning = new ConcurrentHashMap<>();

    private final Map<String,Set<String>> doNotTrendData = new ConcurrentHashMap<>();

    private final Map<Long,Long> lastUpdateMap = new ConcurrentHashMap<>();
    
// ---- HasLeifecycle methods
    
    @Override
    public void build() {
        super.build();
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("data-last-update", () -> updateDataLastUpdate()).withIsFixedRate(true).withPeriod(Duration.ofMinutes(1)));        
    }
    
    public void updateDataLastUpdate() {
        synchronized(lastUpdateMap) {
            addToQueue(new Object[]{new RegisterDataDescLastUpdate(lastUpdateMap)});
            lastUpdateMap.clear();
        }
    }
    
    /**
     * The constructor initializes entries in StatDesc with regards of existing
     * entries in DataDesc. It also creates an internal map to analyze data
     * description integrity with future entries.
     **/ 
    @Override
    public void init() {
        super.init();
        SessionFactory fac = StatusdbUtils.getSessionFactory();
        
        log.fine("init StatusPersister");
        Session sess = fac.openSession();
        Transaction tx = sess.beginTransaction();
        List<DataDesc> l = sess.createQuery("from DataDesc").list();
        for (DataDesc dd : l) {
            map.put(dd.getDataPath(), dd);
            log.fine("storing " + dd.getDataPath().getFullKey());
            // Create default statistical binning for existing entries that
            // don't have any.
            getStatDescs(dd, sess, log);
        }
        tx.commit();
        sess.close();

        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        
        dictionaryService.addDataProviderDictionaryListener(this);
    }
    
    
    @Override
    public void postInit() {        
        subsys.getMessagingAccess().addStatusMessageListener(this, 
                (BusMessageFilterFactory.messageClass(StatusData.class).or(
                        BusMessageFilterFactory.messageClass(StatusConfigurationInfo.class)))
                        .and(BusMessageFilterFactory.messageOrigin(null)));
    }

    @Override
    public void shutdown() {
        subsys.getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
        subsys.getMessagingAccess().removeStatusMessageListener(this); 
        super.shutdown();
    }
    
    @Override
    public void disconnected(AgentInfo... agents) {
        for ( AgentInfo ai : agents ) {
            if (ai.getType().compareTo(AgentInfo.AgentType.WORKER) < 0 ) {
                continue;
            }
            String agentName = ai.getName();
//        mdAccumulatorsMap.get(agentName).addToQueue(new CloseMetaData(agentName));
            dictionaryProcessed.remove(agentName);
            noDictionaryProcessedWarning.remove(agentName);
            doNotTrendData.remove(agentName);
        }
    }

    @Override
    public void dataProviderDictionaryUpdate(DataProviderDictionaryService.DataProviderDictionaryEvent evt) {
        if (evt.getEventType() == DataProviderDictionaryService.DataProviderDictionaryEvent.EventType.ADDED) {
            String name = evt.getAgentInfo().getName();
            log.log(Level.FINER, "received a data dictionary for {0}", name);
            if (dictionaryProcessed.contains(name)) {
                return;
            }
            dictionaryProcessed.add(name);
            DataProviderDictionary dpd = evt.getDictionary();
            // A dictionary of data is received : resolving the datadescs.
            RegisterDataGroup rdg = new RegisterDataGroup(evt.getAgentInfo());
            addToQueue(new Object[]{rdg, dpd});
            for (DataProviderInfo dpi : dpd.getDataProviderInfos()) {
                // get the attributes of each DataProviderInfo and persist 
                // the ones that correspond to metadata but only for TRENDING
                // or MONITORING data
                String typeStr = dpi.getAttributeValue(Attribute.DATA_TYPE);
                if (typeStr != null) {
                    Type dataType = Type.valueOf(typeStr);
                    if (dataType == Type.MONITORING || dataType == Type.TRENDING) {
                        if ("true".equals(dpi.getAttributeValue(Attribute.DO_NOT_TREND))) {
                            Set<String> ignoreData = doNotTrendData.getOrDefault(name, new CopyOnWriteArraySet<>());
                            ignoreData.add(name + "/" + dpi.getFullPath());
                            ignore_log.fine("Adding " + name + " to ignore list");
                            doNotTrendData.put(name, ignoreData);
                        } else {
                            for (Attribute attr : dpi.getAttributes()) {
                                if (attr.isMetadata()) {
                                    queueImmediateMetaData(evt.getAgentInfo().getAgentStartTime().getUTCInstant().toEpochMilli(), new DataPath(name, dpi.getPath()), attr.getName(), dpi.getAttributeValue(attr));
                                }
                            }
                        }
                    }
                }
            }
        }
    }
    
    @Override
    public void onStatusMessage(StatusMessage s) {
        if (s.getOriginAgentInfo().getType().compareTo(AgentInfo.AgentType.WORKER) < 0 ) {
            return;
        }
        if (s instanceof StatusData) {
            KeyValueDataList encodedData = (KeyValueDataList) s.getEncodedData();
            if (encodedData != null) {
                processEncodedData(s.getOriginAgentInfo().getName(), encodedData);
            }
        } else if ( s instanceof StatusConfigurationInfo ) {
            ConfigurationInfo ci = ((StatusConfigurationInfo) s).getConfigurationInfo();
            log.log(Level.FINE, "Processing configurationInfo for {0}", ((StatusConfigurationInfo) s).getOriginAgentInfo().getName());
            for ( ConfigurationParameterInfo par : ci.getAllParameterInfo() ) {
                String parName = par.getParameterName();
                if ( parName.equals("limitLo") || parName.equals("limitHi") || parName.equals("warnLo") || parName.equals("warnHi") )  {
                    String channelPath = par.getComponentName();
                    String metadataName = parName.replace("limit", "alarm").replace("warn", "warning").replace("Hi", "High").replace("Lo", "Low");
                    queueImmediateMetaData(s.getCCSTimeStamp().getUTCInstant().toEpochMilli(),
                            new DataPath(s.getOriginAgentInfo().getName(), channelPath),
                            metadataName, par.getCurrentValue());
                }
            }
        }
    }

    private void queueImmediateMetaData(long tStamp, DataPath name, String metadata, String value) {
        mdPersister.queueImmediateMetaData(tStamp, name, metadata, value);        
    }
        
    //This method is no longer private so that it can be invoked externally
    //by other components of the localdb subsystem.
    //It has to be synchronized to regulate access to it.
    public synchronized void processEncodedData(String source, KeyValueDataList encodedData) {
        // Status data are persisted as key value pairs
        for (KeyValueData d : encodedData) {
            KeyValueData.KeyValueDataType type = d.getType();
            if (null != type) {
                switch (type) {
                    case KeyValueTrendingData:
                        queueImmediateScalar(source, d);
                        break;
                    case KeyValueMetaData:
                        String metaPath = d.getKey();
                        int lastIndex = metaPath.lastIndexOf('/');
                        String metaname = metaPath.substring(lastIndex + 1);
                        //Since we are now getting alarm/warning High/Low from
                        //the ConfigurationInfo message we no longer use the metadata publication
                        //for these quantities.
                        //This test can be removed once we are sure that these values are no longer
                        //published as metadata.
                        if ( metaname.equals("alarmHigh") || metaname.equals("alarmLow") ||
                                metaname.equals("warningHigh") || metaname.equals("warningLow")
                                ) {
                            break;
                        }
                        String key = metaPath.replace("/" + metaname, "");
                        queueImmediateMetaData(d.getCCSTimeStamp().getUTCInstant().toEpochMilli(), new DataPath(source, key), metaname, (String) d.getValue());
                        break;
                    default:
                        break;
                }
            }
        }
        
    }

    
    private boolean ignoreData(DataPath path) {
        String source = path.getAgentName();
        Set<String> ignoreData = doNotTrendData.get(source);
        if ( ignoreData != null && ignoreData.contains(path.getFullKey()) ) {
            ignore_log.log(Level.FINE, "Skipping data {0} for source {1}", new Object[]{path.getFullKey(), source});
            return true;
        }
        return false;
    }
    
    private void queueImmediateScalar(String source, KeyValueData kvd) {

        if ( !dictionaryProcessed.contains(source) ) {
            Set<String> warningIssued = noDictionaryProcessedWarning.computeIfAbsent(source, (s) -> new CopyOnWriteArraySet<>());
            if ( ! warningIssued.contains(kvd.getKey()) ) {
                log.log(Level.WARNING, "Skipping data {0} {1} (no dictionary)", new Object[]{source, kvd.getKey()});    
                warningIssued.add(kvd.getKey());
            }
            return;
        }
        long tStamp = kvd.getCCSTimeStamp().getUTCInstant().toEpochMilli();
        DataPath name = new DataPath(source, kvd.getKey());
        if ( ignoreData(name) ) {
            return;
        }
        Object d = kvd.getValue();
        
        log.log(Level.FINER, "got update {0}", name);
        RawData data = new RawData();
        data.setTime(tStamp);
        if (d instanceof Double) {
            data.setDoubleData((Double) d);
        } else if (d instanceof Float) {
            data.setDoubleData(((Float) d).doubleValue());
        } else if (d instanceof Integer) {
            data.setDoubleData(((Integer) d).doubleValue());
        } else if (d instanceof Short) {
            data.setDoubleData(((Short) d).doubleValue());
        } else if (d instanceof Long) {
            data.setDoubleData(((Long) d).doubleValue());
        } else {
            // TODO: if array do an Array.toString
            data.setStringData(String.valueOf(d));
        }
        addToQueue(new Object[] { data, name });
    }

    private void persistData(RawData data, DataPath name, Session sess) {
        if(data.getDoubleData() != null && !Double.isFinite(data.getDoubleData())) {
            log.log(Level.WARNING, "Skipping non-finite RawData value {0} for path {1}", new Object[]{data.getDoubleData(),data.getDataDesc()});
            return;
        }
        DataDesc dd = getDataDescription(name, sess, null);
        if (dd == null || dd == NULL_DESCRIPTION) {
            return;
        }
        data.setDataDesc(dd);
        sess.lock(dd, LockMode.NONE);

        synchronized(lastUpdateMap) {
            Long storedLastUpdate = lastUpdateMap.getOrDefault(dd.getId(), -1L);
            long t = data.getTime();
            if (t > storedLastUpdate) {
                lastUpdateMap.put(dd.getId(), t);
            }
        }
        sess.persist(data);
        
        // Queuing data for stat
        statDataAccumulator.addToQueue(data);
    }

    
    public StatDataAccumulator getStatAccumulator() {
        return statDataAccumulator;
    }
    

    private void persistDataDescLastUpdate(RegisterDataDescLastUpdate lastUpdate, Session sess) {
        long start = System.currentTimeMillis();
        for ( Entry<Long,Long> e : lastUpdate.getLastUpdates().entrySet() ) {
            Long id = e.getKey();
            long t = e.getValue();
            DataDescLastUpdate last = new DataDescLastUpdate(id, t);
            sess.saveOrUpdate(last);
        }
        long delta = System.currentTimeMillis() - start;
        log.log(Level.INFO,"Took {0} ms to update last time for {1} entries.", new Object[]{delta,lastUpdate.getLastUpdates().size()});        
    }

    private void persistDataGroups(RegisterDataGroup rdg, DataProviderDictionary dpd, Session sess) {
        String agentName = rdg.getAgentName();

        //Mark as inactive DataDesc that don't belong to the dictionary
        List<DataPath> dictionaryPaths = new ArrayList<>();
        for ( DataProviderInfo dpi : dpd.getDataProviderInfos() ) {
            String typeStr = dpi.getAttributeValue(Attribute.DATA_TYPE);
            if ( typeStr != null ) {
                try {
                    Type dataType = Type.valueOf(typeStr);
                    if (dataType == Type.MONITORING || dataType == Type.TRENDING) {
                        dictionaryPaths.add(new DataPath(agentName, dpi.getPath()));
                    }
                } catch (Exception e) {
                    log.log(Level.WARNING, "Illegal type conversion from {0}", typeStr);
                }
            }
        }
        
        for (Entry<DataPath,DataDesc> entry : map.entrySet()) {
            boolean isInDictionary = dictionaryPaths.remove(entry.getKey());
            DataDesc dataDesc = entry.getValue();            
            if ( dataDesc.getDataPath() == null || !dataDesc.getDataPath().getAgentName().equals(agentName) ) {
                continue;
            }
            if ( !dataDesc.getActive()  ) {
                if ( isInDictionary ) {
                    dataDesc.setActive(true);
                    sess.saveOrUpdate(dataDesc);
                }
            } else {
                if ( ! isInDictionary ) {
                    log.log(Level.WARNING, "Disabling Data description for {0}", entry.getKey());
                    dataDesc.setActive(false);
                    sess.saveOrUpdate(dataDesc);
                }
            }
        }
        
        for ( DataProviderInfo dpi : dpd.getDataProviderInfos() ) {
            DataDesc dd = getDataDescription(new DataPath(agentName, dpi.getPath()), sess, dpi);
            if ( dd == null ) {
                continue;
            }
            rdg.addDataDescToGroup(dpi.getPath(), dd);            
        }
        
        log.log(Level.FINER,"persisting data group information");
        // A new dictionary is received
        /** creating/updating group members. */
        for(Map.Entry<String, List<DataDesc>> entry : rdg.getDataDescs().entrySet()) {
            String groupName = entry.getKey();
            DataGroup dg = getDataGroup(agentName, groupName, sess);
            log.log(Level.FINER, "updating data group {0} for {1}", new Object[]{groupName, agentName});
            for(DataDesc dd : entry.getValue()) {
                if(!dg.getMembers().containsKey(dd.getDataPath())) {
                    dg.addMember(dd);
                }
            }
        }
        sess.flush();
    }
   
    private DataDesc getDataDescription(DataPath key, Session sess, DataProviderInfo dpi) {
        Query q = sess.getNamedQuery("findDataDesc").setString("agentName", key.getAgentName()).setString("dataName", key.getDataName());
        DataDesc dd = (DataDesc) q.uniqueResult();

        if ( dpi != null ) {
            
            String typeStr = dpi.getAttributeValue(Attribute.DATA_TYPE);
            try {
                Type dataType = Type.valueOf(typeStr);
                if (dataType != Type.MONITORING && dataType != Type.TRENDING) {
                    return null;
                }
            } catch (Exception e) {
                log.log(Level.WARNING, "Illegal type conversion from {0}", typeStr);
            }

            String type = "double";
            String dpiType = dpi.getAttributeValue(Attribute.TYPE);
            if (dpiType != null) {
                type = dpiType;
            }
            
            if (dd == null) {
                dd = new DataDesc();
                dd.setDataPath(key);
                dd.setDataType(type);
                sess.persist(dd);

                log.log(Level.FINE, "Adding default Data Description for {0}: {1}", new Object[]{key, dd.getId()});
                map.put(key, dd);
            } else {
                if ( ! type.equals(dd.getDataType()) ) {
                    log.log(Level.FINE, "Updating type for Data Description for {0} from {1} to {2}", new Object[]{key, dd.getDataType(), type});
                    dd.setDataType(type);
                    sess.update(dd);
                    map.put(key, dd);
                }
            }
        }        
        return dd;
    }
    
    public synchronized static DataGroup getDataGroup(String agentName, String groupName, Session sess) {
        Query q = sess.getNamedQuery("findDataGroup").setString("agentName", agentName).setString("groupName", groupName);
        DataGroup dg = (DataGroup) q.uniqueResult();
        if (dg == null) {
            // create and persist an empty group
            dg = new DataGroup(agentName, groupName);
            sess.persist(dg);
        }
        return dg;
    }

    public static List<StatDesc> getStatDescs(DataDesc dd, Session sess, Logger log) {
        List<StatDesc> stats = sess.getNamedQuery("findStatDesc").setEntity("dd", dd)
                .setFlushMode(FlushMode.COMMIT)
                .setCacheable(true)
                .list();
        if (stats.isEmpty()) {
            try {
            for (long statInterval : DEFAULT_TIME_BIN_WIDTH) {
                StatDesc defaultSD = new StatDesc();
                defaultSD.setDataDesc(dd);
                defaultSD.setTimeBinWidth(statInterval);
                sess.persist(defaultSD);
                stats.add(defaultSD);
                log.log(Level.FINE, "Adding default statistical binning of {0} milliseconds to {1}", new Object[]{statInterval, dd.getDataPath()});
            }
            } catch (Exception e) {
                log.log(Level.WARNING, "Problem getting statistical data bin for {0} {1}", new Object[]{dd.getDataPath(), dd.getId()});
                throw e;
            }
        }
        return stats;
    }

    /**
     * Check if two data names are consistent according to
     * https://jira.slac.stanford.edu/browse/LSSTCCS-732. We check that neither
     * names starts with the other and that they have different levels of
     * directories. This last bit is so that we accept the situation where the
     * leaf name differs by only the ending: a.b.c and a.b.ca are consistent.
     * a.b and a.bc.d are consistent. a.b and a.b.c are not.
     *
     * @param name1
     *            the first data name
     * @param name2
     *            the second data name
     * @return true/false
     */
    static boolean areDataNamesConsistent(String name1, String name2) {
        if (name1.equals(name2)) {
            return true;
        }

        String[] array1 = name1.split("/");
        String[] array2 = name2.split("/");

        for (int i = 0; i < Math.min(array1.length, array2.length); i++) {
            if (!array1[i].equals(array2[i])) {
                return true;
            }
        }
        return array1.length == array2.length;
    }

    /**
     * Here is where the dataDescs must be resolved.
     * @param obj
     * @param sess 
     */
    @Override
    public void persist(Object[] obj, Session sess) {
        Object toPersist = obj[0];
        if (toPersist instanceof RegisterDataGroup) {
        
            persistDataGroups((RegisterDataGroup)toPersist, (DataProviderDictionary)obj[1], sess);
        } else if (toPersist instanceof RegisterDataDescLastUpdate) {
            persistDataDescLastUpdate((RegisterDataDescLastUpdate)toPersist, sess);
        } else {
            DataPath name = (DataPath) obj[1];
            if (toPersist instanceof RawData) {
                persistData((RawData) toPersist, name, sess);
            } else {
                if (toPersist != null) {
                    sess.persist(toPersist);
                }
            }
        }
    }
}
