package org.lsst.ccs.localdb.statusdb;

import java.util.logging.Level;
import org.hibernate.LockMode;
import org.hibernate.Session;
import org.lsst.ccs.localdb.statusdb.model.StatData;

/**
 * Batch persister for {@code StatData} related entities.
 * This persister receives clumps of statistical data bins to store in the
 * database every 5/30 mins.
 * Its purpose is to spread out the ingestion of these entities as not to affect
 * the ingestion of raw data.
 * 
 * @author LSST CCS Team
 */
public class StatDataPersister extends SlowBatchPersister<StatData> {
        
    @Override
    public void persist(StatData statData, Session sess) {
        sess.lock(statData.getStatDesc(), LockMode.NONE);
        sess.lock(statData.getStatTimeInterval(), LockMode.NONE);
        if (statData.getId() > 0) {
            // This means that the cached stat is in detached state.
            // It needs to be attached to the current session.
            log.log(Level.FINEST, "merging : {0} and timebinwidth {1} starting at {2}", new Object[]{statData.getStatDesc().getDataDesc().getDataPath().getFullKey(), statData.getStatDesc().getTimeBinWidth(), statData.getStatTimeInterval().getStartTime()});
            sess.update(statData);
        } else {
            log.log(Level.FINEST, "persisting stat data {0} and timebinwidth {1} starting at {2}", new Object[]{statData.getStatDesc().getDataDesc().getDataPath().getFullKey(), statData.getStatDesc().getTimeBinWidth(), statData.getStatTimeInterval().getStartTime()});
            sess.persist(statData);
        }
    }      
    
    /**
     * Persists all the cached statistical data.
     * @param sess
     */
    public void flush(Session sess) {
        StatData sd = rq.poll();
        while (sd != null) {
            persist(sd, sess);
            sd = rq.poll();
        }
    }
    
    
}
