package org.lsst.ccs.localdb.trendserver;

import java.net.NetworkInterface;
import java.time.Duration;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.ConfigurationParameter;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.statusdb.LocalDBAlert;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.trendServer.RestServer;
import org.lsst.ccs.localdb.utils.TrendingConnectionUtils;
import org.lsst.ccs.messaging.NetworkUtilities;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.alert.AlertService;

/**
 * An agent wrapping the rest server.
 * @author LSST CCS Team
 */
public class TrendingRestServer extends Subsystem implements HasLifecycle {

    private static final Logger log = Logger.getLogger(TrendingRestServer.class.getName());
    private RestServer rs;
    
    @ConfigurationParameter(isFinal = true)
    private volatile String hostAddr;
    @ConfigurationParameter(isFinal = true)
    private volatile int port = 8080;
    @ConfigurationParameter(category = "build")
    private volatile boolean embeddedServer = true;
    
    private TrendingConnectionUtils connectionUtils;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AlertService alertService;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    AgentPropertiesService agentPropertiesService;
    
    private final String localAddress;
    
    private final Alert restServerUnavailable = new Alert("RestServerUnavailable",
                    "Alert raised when there are problems communicating with the Rest Server.");
    
    public TrendingRestServer() throws Exception {
        super("rest-server", AgentInfo.AgentType.SERVICE);
        NetworkInterface ne = NetworkInterface.getByName(NetworkUtilities.getMainInterfaceName());

        if (ne.getInterfaceAddresses().size()>0) {
            localAddress = ne.getInterfaceAddresses().get(0).getAddress().getHostAddress();
        } else {
            localAddress = null;
        }
        getAgentInfo().getAgentProperties().setProperty("org.lsst.ccs.use.full.paths", "true");        
    }

    @Override
    public void build() {
                
        //If we are using a rest server running in a container
        //external to this Agent, then we must check that it's running.
        //If it's not running raise an alert;
        if ( !embeddedServer ) {
            AgentPeriodicTask checkRest = new AgentPeriodicTask("check-rest",() -> checkRest()).withIsFixedRate(true).withPeriod(Duration.ofMinutes(1));
            getAgentService(AgentPeriodicTaskService.class).scheduleAgentPeriodicTask(checkRest);            
        }        
        
    }
    
    @Override
    public void init() {
        if ( ! embeddedServer ) {
            if ( hostAddr == null ) {
                log.log(Level.WARNING,"Host Address for Trending connection has not been defined!");            
            }
            ClearAlertHandler alwaysClear = new ClearAlertHandler() {
                @Override
                public ClearAlertHandler.ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                    return ClearAlertHandler.ClearAlertCode.CLEAR_ALERT;
                }

            };
            connectionUtils = new TrendingConnectionUtils(hostAddr,port);

            alertService.registerAlert(restServerUnavailable, alwaysClear);
        }

        agentPropertiesService.setAgentProperty("rest-service-addr", hostAddr.equals("localhost") ? localAddress : hostAddr);
        agentPropertiesService.setAgentProperty("rest-service-port", String.valueOf(port));
        agentPropertiesService.setAgentProperty("rest-service-entrypoint", "/rest/data");
        agentPropertiesService.setAgentProperty(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.REST_SERVER.name());
    }
    

    private void checkRest() {
        long start = System.currentTimeMillis();
        boolean raiseAlert = false;
        String cause = null;
        try {
            DataChannel.DataChannelList channels = connectionUtils.getChannelList();
            if ( channels == null || channels.list.isEmpty() ) {
                raiseAlert = true;
                cause = "The Rest-Server returned " + channels == null ? "a null" : "an empty"+ " list of channels";
            }
        } catch (Exception ex) {
            cause = "An exception was raised by the Rest-Server when fetching the list of channels: "+ex.getMessage();
            raiseAlert = true;
        }
                
        if ( raiseAlert ) {
            alertService.raiseAlert(restServerUnavailable,
                    AlertState.ALARM, cause);
        }
        
        long delta = System.currentTimeMillis() - start;
        log.log(Level.FINE, "Rest Server Checked in {0} ms", delta);
    }

    @Override
    public void startAgent() {
        super.startAgent(); 
        if ( embeddedServer ) {
            rs = new RestServer(port);
            rs.start();            
        } 
        
        log.log(Level.INFO, "Rest Server "+ 
                (embeddedServer ? "started in an embedded":"running in an external") 
                        +" container ({0}:{1})", new Object[]{hostAddr, String.valueOf(port)});                
    }
    
    @Override
    public void shutdown() {
        if ( rs != null ) {
            rs.stop();
        }
    }
    
    
}
