package org.lsst.ccs.localdb.statusdb;

import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.hibernate.LockMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentPropertyPredicate;
import org.lsst.ccs.localdb.statusdb.model.DataGroup;
import org.lsst.ccs.localdb.statusdb.model.DataPath;
import org.lsst.ccs.localdb.statusdb.model.MetaDataData;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;

/**
 * Batch persistence of MetaDataData entities.
 * @author LSST CCS Team
 */
public class MetaDataPersister extends SlowBatchPersister<Object> {

    private static final Logger LIVE_MD_LOG = Logger.getLogger("org.lsst.ccs.localdb.statusdb.metadata");
    
    
    private static Map<String,MetaDataData> LIVE_METADATA;
    
    private AgentPropertyPredicate predicate;
    
    @Override
    public void init() {
        super.init();
        initializeIfNeeded();
    }
    
    @Override
    public void shutdown() {
        terminateIfNeeded();
        super.shutdown();
    }
    
    
    public boolean acceptAgentInfo(AgentInfo t) {
        return predicate == null ? true : predicate.test(t);
    }

    @Override
    public void persist(Object obj, Session sess) {
        if (obj instanceof MetaDataData) {
            persistMetaData((MetaDataData)obj, sess);
        } else if (obj instanceof CloseMetaData) {
            closeMetaData((CloseMetaData)obj, sess);
        }
    }
    
    
    public void queueImmediateMetaData(long tStamp, DataPath name, String metadata, String value) {        
        MetaDataData md = new MetaDataData(tStamp);
        md.setName(metadata);
        md.setValue(value);
        DataGroup dg = new DataGroup(name.getAgentName(),name.getDataName());
        md.setDataGroup(dg);
        if (!needsPersisting(md)){
            return;
        }        
        log.log(Level.FINER,"Queuing metadata {0} with value {1}.", new Object[]{getMetaDataDataUniqueKey(md),md.getValue()});
        addToQueue(md);
    }
    
    private synchronized boolean needsPersisting(MetaDataData md) {
        MetaDataData oldMetaData = LIVE_METADATA.get(getMetaDataDataUniqueKey(md));
        if (oldMetaData != null) {            
            //This new metadata is to be persisted only if its value has changed.
            //If the value is unchanged, then skip the rest.
            if ( oldMetaData.getValue().equals(md.getValue()) ) {
                LIVE_MD_LOG.log(Level.FINEST,"Metadata {0} does not need persisting since its value {1} is identical to the existing one in the live map {2}.", new Object[]{getMetaDataDataUniqueKey(md),md.getValue(), oldMetaData.getValue()});
                return false;
            }
        }
        return true;
    }
    
    
    
    private void persistMetaData(MetaDataData md, Session sess) {        
        String uniqueKey = getMetaDataDataUniqueKey(md);
        log.log(Level.FINE,"Persisting metadata {0} with value {1}.", new Object[]{uniqueKey,md.getValue()});
        MetaDataData oldMetaData = LIVE_METADATA.get(uniqueKey);
        if (oldMetaData != null) {
            
            //This new metadata is to be persisted only if its value has changed.
            //If the value is unchanged, then skip the rest.
            if ( oldMetaData.getValue().equals(md.getValue()) ) {
                log.log(Level.FINER,"Metadata {0} does not need persisting since its value {1} is identical to the existing one in the live map {2}.", new Object[]{uniqueKey,md.getValue(), oldMetaData.getValue()});
                return;
            }
            log.log(Level.FINEST,"Closing old metadata {0} with value {1}; new value {2}.", new Object[]{uniqueKey,oldMetaData.getValue(),md.getValue()});            
            oldMetaData.setEndTime(md.getStartTime());
            LIVE_METADATA.remove(uniqueKey);
            sess.update(oldMetaData);
        }
        
        //If the DataGroup has no id, it needs to be created
        DataGroup dg = md.getDataGroup();        
        if ( dg.getId() == 0 ) {
            dg = StatusDataPersister.getDataGroup(dg.getAgentName(), dg.getGroupName(), sess);
            md.setDataGroup(dg);
        }
        sess.lock(dg, LockMode.NONE);
        
        // Now commit the new value of the metadata
        LIVE_METADATA.put(uniqueKey, md);
        sess.persist(md);
    }
    
    private void closeMetaData(CloseMetaData cmd, Session sess) {
        // TODO : connecting does not propagate the timestamp of the status message that triggered it
        long time = System.currentTimeMillis();
        // Closing the existing open meta data for the corresponding agent
//        Query q = sess.createQuery("from MetaDataData md where dataGroup.agentName=:agName and endTime <= 0");
//        q.setString("agName", cmd.agentName);
//        List<MetaDataData> res = q.list();
        for(MetaDataData md : LIVE_METADATA.values() ) {
            if ( md.getDataGroup().getAgentName().equals(cmd.agentName)) {
                md.setEndTime(time);
                LIVE_METADATA.remove(getMetaDataDataUniqueKey(md));
                sess.update(md);
            }
        }
    }
    
    private synchronized static void initializeIfNeeded() {
        if (LIVE_METADATA == null) {
            LIVE_METADATA = new ConcurrentHashMap<>();
            try ( Session sess = StatusdbUtils.getSessionFactory().openSession()) {
                Transaction tx = sess.beginTransaction();

                Query q = sess.createQuery("from MetaDataData md where endTime = -1 order by startTime desc");
                List<MetaDataData> res = q.list();

                for (MetaDataData md : res) {
                    String uniqueKey = getMetaDataDataUniqueKey(md);
                    MetaDataData existingMetaData = LIVE_METADATA.get(uniqueKey);
                    if (existingMetaData != null) {
                        LIVE_MD_LOG.log(Level.INFO,"Metadata {0} already belongs to the live map. This should not have happened.", new Object[]{uniqueKey});
                        md.setEndTime(existingMetaData.getStartTime());
                        sess.update(md);
                    } else {
                        LIVE_METADATA.put(uniqueKey, md);
                    }
                }
                tx.commit();
            }
            LIVE_MD_LOG.log(Level.INFO, "Loaded {0} live metadata fields ", new Object[] {LIVE_METADATA.size()});
        }
    }

    private synchronized static void terminateIfNeeded() {
        if (LIVE_METADATA != null) {
            int size = LIVE_METADATA.size();
            try ( Session sess = StatusdbUtils.getSessionFactory().openSession()) {
                Transaction tx = sess.beginTransaction();

                long endTime = System.currentTimeMillis();
                for (MetaDataData md : LIVE_METADATA.values()) {
                    md.setEndTime(endTime);
                    sess.update(md);
                }                
                tx.commit();
            }
            LIVE_METADATA = null;
            LIVE_MD_LOG.log(Level.INFO, "Closed {0} live metadata fields ", new Object[] {size});
        }
    }
    
    private static String getMetaDataDataUniqueKey(MetaDataData md) {
        return md.getDataGroup().getAgentName()+"-"+md.getDataGroup().getGroupName()+"-"+md.getName();
    }
}
