package org.lsst.ccs.localdb.statusdb;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import org.hibernate.Query;
import org.hibernate.Session;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentPropertyPredicate;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.localdb.statusdb.model.MetaDataData;
import org.lsst.ccs.services.AgentPeriodicTaskService;

/**
 * Batch persistence of MetaDataData entities.
 * @author LSST CCS Team
 */
public class MetaDataPersister extends BatchPersister<Object> {
    
    
    private Map<String,MetaDataData> metadataToClose;
    
    private AgentPropertyPredicate predicate;
    
    @LookupField(strategy=LookupField.Strategy.TREE)
    private AgentPeriodicTaskService periodicTaskService;
    
    @LookupName
    private String name;

    public MetaDataPersister() {
        super(6000, true, 1, -1);
    }

    @Override
    public void build() {
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask(name, this).withIsFixedRate(false).withPeriod(Duration.ofSeconds(1)));
    }
    
    
    public boolean acceptAgentInfo(AgentInfo t) {
        return predicate == null ? true : predicate.test(t);
    }

    @Override
    public void persist(Object obj, Session sess) {
        checkInitialization(sess);
        if (obj instanceof MetaDataData) {
            persistMetaData((MetaDataData)obj, sess);
        } else if (obj instanceof CloseMetaData) {
            closeMetaData((CloseMetaData)obj, sess);
        }
    }
    
    private void persistMetaData(MetaDataData md, Session sess) {
        
        // First update the tstop of a previously existing record for this
        // metadata
//        Query q = sess.createQuery("from MetaDataData md " + "where dataGroup.id = :id " + "and name = :n and endTime <= 0");
//        q.setParameter("id", md.getDataGroup().getId());
//        q.setParameter("n", md.getName());
//        MetaDataData oldMetaData = (MetaDataData) q.uniqueResult();
        MetaDataData oldMetaData = metadataToClose.get(getMetaDataDataUniqueKey(md));
        if (oldMetaData != null) {
            
            //This new metadata is to be persisted only if its value has changed.
            //If the value is unchanged, then skip the rest.
            if ( oldMetaData.getValue().equals(md.getValue()) ) {
                return;
            }
            
            oldMetaData.setEndTime(md.getStartTime());
            metadataToClose.remove(getMetaDataDataUniqueKey(md));
            sess.update(oldMetaData);
        }
        // Now commit the new value of the metadata
        metadataToClose.put(getMetaDataDataUniqueKey(md), md);
        sess.persist(md);
    }
    
    private void closeMetaData(CloseMetaData cmd, Session sess) {
        // TODO : connecting does not propagate the timestamp of the status message that triggered it
        long time = System.currentTimeMillis();
        // Closing the existing open meta data for the corresponding agent
//        Query q = sess.createQuery("from MetaDataData md where dataGroup.agentName=:agName and endTime <= 0");
//        q.setString("agName", cmd.agentName);
//        List<MetaDataData> res = q.list();
        for(MetaDataData md : metadataToClose.values() ) {
            if ( md.getDataGroup().getAgentName().equals(cmd.agentName)) {
                md.setEndTime(time);
                metadataToClose.remove(getMetaDataDataUniqueKey(md));
                sess.update(md);
            }
        }
    }
    
    private synchronized void checkInitialization(Session sess) {
        if (metadataToClose == null) {
            metadataToClose = new ConcurrentHashMap<>();
            
            log.info("Loading metadata to be closed");
            Query q = sess.createQuery("from MetaDataData md where endTime = -1 order by startTime desc");
            List<MetaDataData> res = q.list();
            
            for(MetaDataData md : res) {
                String uniqueKey = getMetaDataDataUniqueKey(md);
                MetaDataData existingMetaData = metadataToClose.get(uniqueKey);
                if ( existingMetaData != null ) {
                    log.warn("Duplicate metadata entry for "+uniqueKey+". This should not have happened.");
                    md.setEndTime(existingMetaData.getStartTime());
                    sess.update(md);
                } else {
                    metadataToClose.put(uniqueKey, md);
                }
            }                        
            log.info("Loaded "+metadataToClose.size()+" metadata entries to close.");
        }
    }
    
    private String getMetaDataDataUniqueKey(MetaDataData md) {
        return md.getDataGroup().getId()+"-"+md.getName();
    }
}
