package org.lsst.ccs.localdb.statusdb;

import java.util.ArrayList;
import java.util.List;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.logging.Level;

import org.hibernate.FlushMode;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.resource.transaction.spi.TransactionStatus;
import org.influxdb.dto.Point;
import org.influxdb.dto.Point.Builder;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.services.InfluxDbClientService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Persists entities as a batch.
 *
 * @author LSST CCS Team
 */
public abstract class BatchPersister<T> implements Runnable, HasLifecycle {

    protected final Logger log = Logger.getLogger(getClass().getCanonicalName());

    @LookupName
    private String persisterName;

    /**
     * The maximum number of entities to process in a single transaction.
     */
    private final int nMax;

    private final boolean flushAtCommit;

    private final Queue<T> rq = new ConcurrentLinkedQueue<>();
    
    @LookupField(strategy = Strategy.TREE)
    protected AlertService alertService;
    
    @LookupField(strategy=Strategy.TOP)
    private Agent a;

    @LookupField(strategy=Strategy.TREE)
    StatusDataPersister statusDataPersister;

    @LookupField(strategy=Strategy.TREE)
    InfluxDbClientService influxDbClientService;

    private int nThreads = 1;
    private final ExecutorService exec;
    
    private final static int maximumTasksPerThread = 100;

    private volatile AlertState batchSubmissionAlertState = AlertState.NOMINAL;
    
    public static ExecutorService newFixedThreadPoolBoundedQueue(int nThreads, int nQueue) {
        LinkedBlockingQueue queue = nQueue > 0 ? new LinkedBlockingQueue<>(nQueue) : new LinkedBlockingQueue<>();
        return new ThreadPoolExecutor(nThreads, nThreads, 0, TimeUnit.MILLISECONDS, queue);
    }

    /**
     * Constructor for a batch persister.
     *
     * @param nMax
     *            The maximum number of entities to persist in a single
     *            transaction.
     * @param flushAtCommit
     *            true if the session that process the batch should be set to
     *            FlushMode.COMMIT
     * @param threads
     *            database feeding threads
     * @param threadQueueSize
     *            The size of the queue; if negative the queue is unbound.
     */
    public BatchPersister(int nMax, boolean flushAtCommit, int threads, int threadQueueSize) {        
        this.nMax = nMax;
        this.flushAtCommit = flushAtCommit;
        this.nThreads = threads;
        exec = newFixedThreadPoolBoundedQueue(nThreads, threadQueueSize);
    }


    /**
     * Constructor for a batch persister.
     *
     * @param nMax
     *            The maximum number of entities to persist in a single
     *            transaction.
     * @param flushAtCommit
     *            true if the session that process the batch should be set to
     *            FlushMode.COMMIT
     * @param threads
     *            database feeding threads
     */
    public BatchPersister(int nMax, boolean flushAtCommit, int threads) {
        this(nMax, flushAtCommit, threads, maximumTasksPerThread*threads);
    }

    /**
     * Constructor for a batch persister.
     *
     * @param nMax
     *            The maximum number of entities to persist in a single
     *            transaction.
     * @param flushAtCommit
     *            true if the session that process the batch should be set to
     *            FlushMode.COMMIT
     */
    public BatchPersister(int nMax, boolean flushAtCommit) {
        this(nMax, flushAtCommit, 1);
    }

    @Override
    public void init() {
        
        ClearAlertHandler alwaysClear = new ClearAlertHandler() {
            @Override
            public ClearAlertCode canClearAlert(Alert alert, AlertState alertState) {
                return ClearAlertCode.CLEAR_ALERT;
            }

        };
        
        alertService.registerAlert(LocalDBAlert.BatchException.getAlert(persisterName, null), alwaysClear);
        alertService.registerAlert(LocalDBAlert.BatchRollbackException.getAlert(persisterName, null), alwaysClear);
        alertService.registerAlert(LocalDBAlert.BatchIngestionQueueSize.getAlert(persisterName, null), alwaysClear);
    }
    

    private void persistBatch(List<T> batch) {
        log.debug(persisterName+" start batch " + this.getClass().getName() + " " + this + " " + batch.size());
        long tstart = System.currentTimeMillis();
        Session sess = StatusdbUtils.getSessionFactory().openSession();
        if (flushAtCommit) {
            sess.setFlushMode(FlushMode.COMMIT);
        }
        Transaction tx = null;
        try {
            tx = sess.beginTransaction();
            for (T t : batch) {
                persist(t, sess);
            }
            tx.commit();
        } catch (Exception ex) {
            log.log(Level.SEVERE, persisterName+" caught exception when persisting", ex);
            alertService.raiseAlert(LocalDBAlert.BatchException.getAlert(persisterName, ex), AlertState.WARNING,
                    LocalDBAlert.getFirstException(ex));
            try {
                if (tx != null && (tx.getStatus() == TransactionStatus.ACTIVE
                        || tx.getStatus() == TransactionStatus.MARKED_ROLLBACK)) {
                    tx.rollback();
                }
            } catch (Exception rbEx) {
                log.error(persisterName+" Rollback of transaction failed : " + rbEx, rbEx);
                alertService.raiseAlert(LocalDBAlert.BatchRollbackException.getAlert(persisterName, ex),
                        AlertState.WARNING, LocalDBAlert.getFirstException(ex));
            }
        } finally {
            if (sess.isOpen()) {
                sess.close();
            }
        }

        long time = System.currentTimeMillis() - tstart;
        if (influxDbClientService != null) {
            Builder batchPersisterPointBuilder = Point.measurement("db_persist").time(System.currentTimeMillis(),
                    TimeUnit.MILLISECONDS);

            batchPersisterPointBuilder = batchPersisterPointBuilder.addField("btime_tot", time)
                    .addField("bcount", batch.size()).addField("btime_avg", (double) time / (double) batch.size());
            Point point = batchPersisterPointBuilder.tag("persister", persisterName).tag(influxDbClientService.getGlobalTags()).build();
            influxDbClientService.getInfluxDbClient().write(point);
        } else {
            KeyValueDataList kvdl = new KeyValueDataList(persisterName);
            kvdl.addData(persisterName + "/transactionTime", time);
            kvdl.addData(persisterName + "/batchSize", batch.size());
            kvdl.addData(persisterName + "/entityAverageTime", (double) time / batch.size());
            statusDataPersister.processEncodedData(a.getName(), kvdl);
        }

        log.debug(persisterName+" end batch ");
    }

    @Override
    public void run() {
        log.fine(persisterName+" start run " + rq.size());
        int n = 0;
        int nAcc = 0;
        boolean done = false;
        long tstart = System.currentTimeMillis();
        int queueAtStart = rq.size();
        int tasksAtStart = ((ThreadPoolExecutor) exec).getQueue().size();
        int nbatch = 0;
        while (!done) {
            List<T> batch = new ArrayList<T>(nMax);
            // get at most nMax objets from the queue
            for (int i = 0; i < nMax; i++) {
                T t = rq.poll();
                n = i;
                if (t == null) {
                    done = (i == 0);
                    break;
                } else {
                    batch.add(t);
                }
            }
            if (batch.size() > 0) {
                Alert batchSubmissionAlert = LocalDBAlert.BatchIngestionQueueSize.getAlert(persisterName, null);
                try {
                    exec.submit(() -> persistBatch(batch));
                    //Check the size of the queue containing the batch tasks.
                    //If it exceeds half of the maximum number of tasks allowed per thread
                    //then raise an Alert at WARNING if we are currently in NOMINAL.
                    //Otherwise check if we are recovering and lower the state from
                    //ALARM to WARNING as we are catching up.
                    int taskSubmissionQueueSize = ((ThreadPoolExecutor) exec).getQueue().size();
                    if ( taskSubmissionQueueSize > nThreads*maximumTasksPerThread/2 ) {
                        String cause = null;
                        if ( batchSubmissionAlertState == AlertState.NOMINAL ) {
                            //We are starting to slow down the ingestion
                            cause = "The submission queue is filling up. Ingestion is falling behind.\n";                            
                        } else if ( batchSubmissionAlertState == AlertState.ALARM && taskSubmissionQueueSize < 3*nThreads*maximumTasksPerThread/4 ) {
                            cause = "The submission queue is being drained. Ingestion is catching up.\n";
                        }
                        if ( cause != null ) {
                            cause += "Executor task queue size: "+taskSubmissionQueueSize +" (nThreads="+nThreads+")";
                            batchSubmissionAlertState = AlertState.WARNING;
                            alertService.raiseAlert(batchSubmissionAlert,batchSubmissionAlertState, cause);                            
                        }
                    } else if ( taskSubmissionQueueSize <  nThreads*maximumTasksPerThread/8 && batchSubmissionAlertState != AlertState.NOMINAL) {
                        String cause = "Ingestion is caught up.\n";                            
                        cause += "Executor task queue size: "+taskSubmissionQueueSize +" (nThreads="+nThreads+")";                        
                        batchSubmissionAlertState = AlertState.NOMINAL;
                        alertService.raiseAlert(batchSubmissionAlert,batchSubmissionAlertState, cause);                            
                    }
                } catch (RejectedExecutionException e) {
                    if ( batchSubmissionAlertState != AlertState.ALARM ) {                    
                        String cause = "Failed to submit a new task. Ingestion is all filled up. \nExecutor task queue size "+
                            + ((ThreadPoolExecutor) exec).getQueue().size();
                        batchSubmissionAlertState = AlertState.ALARM;
                        alertService.raiseAlert(batchSubmissionAlert,batchSubmissionAlertState, cause);
                    }
                    //Put the batch into the queue for reprocessing 
                    rq.addAll(batch);
                }
                nbatch++;
                done = (rq.size() < nMax / 5); // too few objects left for a
                                               // batch, we will wait for the
                                               // next round
                if (done)
                    log.fine(persisterName+" run done, left in queue " + rq.size());
            }

            if (n > 0) {
                nAcc += n;
                log.debug(persisterName+" processed " + n + " entities.");
            }
        }
        if (nAcc > 0)

        {
            long time = System.currentTimeMillis() - tstart;
            if ( influxDbClientService != null  ) {
                Builder batchPersisterPointBuilder = Point.measurement("db_persist")
                        .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);

                int submissionQueueSize = ((ThreadPoolExecutor) exec).getQueue().size();
                double perThreadSize = (double)submissionQueueSize/nThreads;

                batchPersisterPointBuilder = batchPersisterPointBuilder.
                        addField("time_tot", time).
                        addField("count", nAcc).
                        addField("time_avg", (double)time/(double)nAcc)
                        .addField("exec_queue_size", (double) submissionQueueSize).addField("exec_queue_size_per_threads", perThreadSize);
                Point point = batchPersisterPointBuilder.tag("persister", persisterName).tag(influxDbClientService.getGlobalTags()).build();
                influxDbClientService.getInfluxDbClient().write(point);                
            } else {
                KeyValueDataList kvdl = new KeyValueDataList(persisterName);
                kvdl.addData(persisterName + "/queueProcessingTime", time);
                kvdl.addData(persisterName + "/queueSize", queueAtStart);
                kvdl.addData(persisterName + "/nBatchInRun", nbatch);
                kvdl.addData(persisterName + "/activeThreads", ((ThreadPoolExecutor) exec).getActiveCount());
                kvdl.addData(persisterName + "/queuedTasks", tasksAtStart);
                log.fine(persisterName+" queued tasks " + tasksAtStart + " " + ((ThreadPoolExecutor) exec).getQueue().size());
                statusDataPersister.processEncodedData(a.getName(), kvdl);
            }
        }
        log.fine(persisterName+" end run ");
    }

    public void addToQueue(T obj) {
        rq.add(obj);
    }

    /**
     * Called from inside an open transaction.
     *
     * @param obj
     * @param sess
     */
    public abstract void persist(T obj, Session sess);

}
