package org.lsst.ccs.localdb.trendserver;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.lsst.ccs.Agent;
import org.lsst.ccs.ServiceLifecycle;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.AgentCategory;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentPropertyPredicate;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.statusdb.server.TrendingData;
import org.lsst.ccs.localdb.utils.TrendingConnectionUtils;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.services.AgentPropertiesService;
import org.lsst.ccs.services.AgentService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * An AgentService to be used by Agents that need to interact with the Rest
 * Server on the buses.
 * Connections can be established by either creating a custom connection to
 * a known host:port combination or by listening to the buses for available rest services.
 * 
 * 
 * @author The LSST CCS Team
 * 
 */
public class TrendingClientService implements AgentService, ServiceLifecycle, AgentPresenceListener, HasLifecycle {

    public static final String USES_TRENDING_SERVICE = "usesTrending";
    private static final Map<String,String> props = new HashMap<>();
    static {
        props.put(AgentCategory.AGENT_CATEGORY_PROPERTY, AgentCategory.REST_SERVER.name());
    }
    private static final AgentPropertyPredicate REST_AGENT_PREDICATE = new AgentPropertyPredicate(props);

    private Logger log = Logger.getLogger(TrendingClientService.class.getName());
    
    private final Map<String, TrendingConnectionUtils> availableClients = new ConcurrentHashMap<>();
        
    @LookupField(strategy = LookupField.Strategy.TOP)
    private Agent a;
    
    @LookupField(strategy = LookupField.Strategy.TREE)
    private AgentPropertiesService agentPropertiesService;

    @Override
    public String getAgentServiceName() {
        return "trendingClientService";
    }

    @Override
    public boolean startForAgent(AgentInfo agentInfo) {
        boolean usesTrendingService = BootstrapResourceUtils.getBootstrapSystemProperties().
                getProperty("org.lsst.ccs.agent."+USES_TRENDING_SERVICE, "false").toLowerCase().equals("true");
        return usesTrendingService;
    }

    @Override
    public void build() {
        agentPropertiesService.setAgentProperty(USES_TRENDING_SERVICE, "true");
    }    

    @Override
    public void preStart() {
        a.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        try {
            if (! a.getMessagingAccess().getAgentPresenceManager().waitForAgent(REST_AGENT_PREDICATE, 3, TimeUnit.SECONDS) ) {
                log.warning("Could not establish a connection with the rest server at startup.");
            }
        } catch (InterruptedException e) {
            
        }
    }

    @Override
    public void connected(AgentInfo... agents) {
        for ( AgentInfo agent : agents ) {
            if ( REST_AGENT_PREDICATE.test(agent) ) {
                String host = agent.getAgentProperty("rest-service-addr");
                String port = agent.getAgentProperty("rest-service-port");
                String entryPoint = agent.getAgentProperty("rest-service-entrypoint");
                TrendingConnectionUtils restClient = new BusTredningConnectionUtils(host, Integer.valueOf(port), entryPoint);
                synchronized (availableClients) {
                    if ( availableClients.containsKey(agent.getName())) {
                        throw new RuntimeException("There is already a valid connection to rest server "+agent.getName());
                    }
                    availableClients.put(agent.getName(), restClient);
                }
                log.debug("Established a connection with the rest server "+restClient.getBaseUrl());
            }
        }
    }

    @Override
    public void disconnecting(AgentInfo agent) {
        if (REST_AGENT_PREDICATE.test(agent)) {
            availableClients.remove(agent.getName());
            log.debug("Lost connection with the rest-server.");
        }
    }
    
    /**
     * Create a named ClientRequestUtils object given a fixed host/port combination.
     * @param name The name for this custom connection to a rest server.
     * @param host The host on which the rest server is running.
     * @param port The port on which the reset server is running.
     * @return a ClientRequestUtils object.
     * @exception This method can throw unchecked exceptions form the underlying
     *            classes if there is a failure in connecting to the trending
     *            client. Users of this method must protect against this by
     *            sorrounding this call in a try-catch block.
     * 
     */
    public TrendingConnectionUtils createClientRequestUtils(String name, String host, int port) {
        TrendingConnectionUtils result = new TrendingConnectionUtils(host, port);
        synchronized (availableClients) {
            if (availableClients.containsKey(name)) {
                throw new RuntimeException("There is already a name connection: " + name);
            }
            availableClients.put(name, result);
        }
        log.debug("Created custom connection with the rest server "+result.getBaseUrl());
        return result;
    }
    
    /**
     * Get a list of available named client connections to the Trending rest server.
     * @return a Set<String> with the available named connections.
     * 
     */
    public Set<String> getNamedTrendingConnections() {
        return availableClients.keySet();
    }
    
    /**
     * Get named Trending Connection.
     * @return the named TrendingConnectionUtils.
     * 
     */
    public TrendingConnectionUtils getTrendingConnection(String name) {
        return availableClients.get(name);
    }
    
    /**
     * Get one of the available TrendingConnections over the buses.
     * @return a TrendingConnectionUtils established from listening to the buses.
     *         null will be returned if no connection is available.
     * 
     */
    public TrendingConnectionUtils getBusTrendingConnection() {
        for ( TrendingConnectionUtils connection : availableClients.values() ) {
            if (connection instanceof BusTredningConnectionUtils) {
                return connection;
            }
        }
        return null;
    }
    
    private class BusTredningConnectionUtils extends TrendingConnectionUtils {
        
        BusTredningConnectionUtils(String host, int port, String entryPoint) {
            super(host, port, entryPoint);
        }
        
    }

    
}
