package org.lsst.ccs.localdb.statusdb;

import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.TimeUnit;
import org.hibernate.FlushMode;
import org.hibernate.Session;
import org.hibernate.Transaction;
import org.hibernate.resource.transaction.spi.TransactionStatus;
import org.influxdb.dto.Point;
import org.influxdb.dto.Point.Builder;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.commons.annotations.LookupName;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.localdb.utils.InfluxDbClientService;
import org.lsst.ccs.services.alert.AlertService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Persists entities as a batch.
 *
 * @author LSST CCS Team
 */
public abstract class BatchPersister<T> implements Runnable, ClearAlertHandler {

    protected static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.statusdb");

    @LookupName
    private String persisterName;
    private static final int CONSECUTIVE_MAX_CAPACITY_LIMIT = 15;

    /**
     * The maximum number of entities to process in a single transaction.
     */
    private final int nMax;

    private final boolean flushAtCommit;

    private final Queue<T> rq = new ConcurrentLinkedQueue<>();

    @LookupField(strategy = Strategy.TREE)
    protected AlertService alertService;
    
    @LookupField(strategy=Strategy.TOP)
    private Agent a;

    @LookupField(strategy=Strategy.TREE)
    StatusDataPersister statusDataPersister;

    @LookupField(strategy=Strategy.TREE)
    InfluxDbClientService influxDbClientService;

    private int maxCapacityCount = 0;

    /**
     * Constructor for a batch persister.
     *
     * @param nMax The maximum number of entities to persist in a single
     * transaction.
     * @param flushAtCommit true if the session that process the batch should be
     * set to FlushMode.COMMIT
     */
    public BatchPersister(int nMax, boolean flushAtCommit) {
        this.nMax = nMax;
        this.flushAtCommit = flushAtCommit;
    }

    @Override
    public ClearAlertCode canClearAlert(Alert alert) {
        return ClearAlertCode.CLEAR_ALERT;
    }

    @Override
    public void run() {
        int n = 0;
        int nAcc = 0;
        boolean empty = false;
        long tstart = System.currentTimeMillis();
        while (!empty) {
            Session sess = StatusdbUtils.getSessionFactory(null).openSession();
            if (flushAtCommit) {
                sess.setFlushMode(FlushMode.COMMIT);
            }
            Transaction tx = null;
            try {
                tx = sess.beginTransaction();
                // get at nMax objets from the queue
                for (int i = 0; i < nMax; i++) {
                    T t = rq.poll();
                    n = i;
                    if (t == null) {
                        maxCapacityCount = 0;
                        empty = (i == 0);
                        break;
                    } else {
                        persist(t, sess);
                    }
                    if (i == nMax - 1) {
                        maxCapacityCount++;
                    }
                }
                if (!empty) {
                    tx.commit();
                }
                if (maxCapacityCount == CONSECUTIVE_MAX_CAPACITY_LIMIT) {
                    alertService.raiseAlert(LocalDBAlert.BatchMaxCapacity.getAlert(persisterName, null), AlertState.WARNING, "current queue size : " + rq.size());
                    maxCapacityCount = 0;
                }
            } catch (Exception ex) {
                log.error("caught exception when persisting", ex);
                alertService.raiseAlert(LocalDBAlert.BatchException.getAlert(persisterName, ex), AlertState.WARNING, LocalDBAlert.getFirstException(ex));
                try {
                    if (tx != null && (tx.getStatus() == TransactionStatus.ACTIVE
                            || tx.getStatus() == TransactionStatus.MARKED_ROLLBACK)) {
                        tx.rollback();
                    }
                } catch (Exception rbEx) {
                    log.error("Rollback of transaction failed : " + rbEx, rbEx);
                    alertService.raiseAlert(LocalDBAlert.BatchRollbackException.getAlert(persisterName, ex), AlertState.WARNING, LocalDBAlert.getFirstException(ex));
                }
            } finally {
                if (sess.isOpen()) {
                    sess.close();
                }
            }
            if (n>0) {
                nAcc+=n;
                log.debug(this.getClass().getSimpleName() + " processed " + n + " entities.");
            }
        }
        if (nAcc > 0) {
            long time = System.currentTimeMillis() - tstart;
            if ( influxDbClientService != null && influxDbClientService.getInfluxDbClient() != null ) {
                Builder batchPersisterPointBuilder = Point.measurement("db_persist")
                        .time(System.currentTimeMillis(), TimeUnit.MILLISECONDS);

                batchPersisterPointBuilder = batchPersisterPointBuilder.
                        addField("time_tot", time).
                        addField("count", nAcc).
                        addField("time_avg", (double)time/(double)nAcc);
                Point point = batchPersisterPointBuilder.tag("persister", persisterName).build();
                influxDbClientService.getInfluxDbClient().write(point);                
            } else {
                KeyValueDataList kvdl = new KeyValueDataList(persisterName);
                kvdl.addData(persisterName + "/transactionTime", time);
                kvdl.addData(persisterName + "/entityAverageTime", (double) time / nAcc);
                statusDataPersister.processEncodedData(a.getName(), kvdl);
            }
        }
    }

    public void addToQueue(T obj) {
        rq.add(obj);
    }

    /**
     * Called from inside an open transaction.
     *
     * @param obj
     * @param sess
     */
    public abstract void persist(T obj, Session sess);

}
