package org.lsst.ccs.localdb.statusdb;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import org.hibernate.FlushMode;

import org.hibernate.LockMode;
import org.hibernate.Query;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.bus.data.DataProviderDictionary;
import org.lsst.ccs.bus.data.DataProviderInfo;
import org.lsst.ccs.bus.data.DataProviderInfo.Attribute;
import org.lsst.ccs.bus.data.DataProviderInfo.Type;
import org.lsst.ccs.bus.data.KeyValueData;
import org.lsst.ccs.bus.data.KeyValueDataList;
import org.lsst.ccs.bus.messages.CommandRequest;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusData;
import org.lsst.ccs.bus.messages.StatusDataProviderDictionary;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.DataGroup;
import org.lsst.ccs.localdb.statusdb.model.DataPath;
import org.lsst.ccs.localdb.statusdb.model.MetaDataData;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.model.StatDesc;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.ConcurrentMessagingUtils;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * StatusDataPersister object. Implements StatusListener and updates the db.
 * <p/>
 * Can be run inside a StatusPersisterSubsystem wrapper.
 * <p/>
 * Deprecated use: inside a Message Driver Bean (tied to JMS implementation)
 * 
 * TODO this class could be split, to separate the StatusMessageListener functions
 * and put at the same level the three accumulators (status, stat, metadata).
 *
 * @author aubourg
 */
public class StatusDataPersister extends BatchPersister<Object[]> implements StatusMessageListener, HasLifecycle, AgentPresenceListener {

    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.statusdb");
    private static final DataDesc NULL_DESCRIPTION = new DataDesc();
    private static final long[] DEFAULT_TIME_BIN_WIDTH = new long[] { 300000, 1800000 }; // 5
                                                                                         // and
                                                                                         // 30
                                                                                         // minutes

    @LookupField(strategy=Strategy.TOP)
    private Agent subsys;
    
    @LookupField(strategy=Strategy.TREE)
    AgentPeriodicTaskService periodicTaskService;
    
    private final Map<DataPath, DataDesc> map = new ConcurrentHashMap<>();
  
    @LookupField(strategy=Strategy.TREE)
    private StatDataPersister statAccumulator;
    
    @LookupField(strategy=Strategy.TREE)
    private final MetaDataPersister mdAccumulator = new MetaDataPersister();

    private final Set<String> dictionaryProcessed = new CopyOnWriteArraySet<>();

    private ConcurrentMessagingUtils cmu;
    
// ---- HasLeifecycle methods
    
    public StatusDataPersister() {
        super(1000, true, 6);
    }
    
    @Override
    public void build() {
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("statdata-accumulator", statAccumulator).withIsFixedRate(false).withPeriod(Duration.ofSeconds(1)));
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("metadata-accumulator", mdAccumulator).withIsFixedRate(false).withPeriod(Duration.ofSeconds(1)));
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("data-accumulator", this).withIsFixedRate(false).withPeriod(Duration.ofSeconds(1)));        
    }
    
    /**
     * The constructor initializes entries in StatDesc with regards of existing
     * entries in DataDesc. It also creates an internal map to analyze data
     * description integrity with future entries.
     **/ 
    @Override
    public void init() {
        SessionFactory fac = StatusdbUtils.getSessionFactory(null);
        
        log.fine("init StatusPersister");
        Session sess = fac.openSession();
        Transaction tx = sess.beginTransaction();
        List<DataDesc> l = sess.createQuery("from DataDesc").list();
        for (DataDesc dd : l) {
            map.put(dd.getDataPath(), dd);
            log.fine("storing " + dd.getDataPath().getFullKey());
            // Create default statistical binning for existing entries that
            // don't have any.
            getStatDescs(dd, sess);
        }
        tx.commit();
        sess.close();

        // Check the integrity of the existing trending channel names.
        log.info("analyzing data desc integrity ...");
        analizeDataMapIntegrity();

        cmu = new ConcurrentMessagingUtils(subsys.getMessagingAccess());
    }
    
    
    @Override
    public void start() {
        
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        subsys.getMessagingAccess().addStatusMessageListener(this, 
                (BusMessageFilterFactory.messageClass(StatusData.class).or(
                        BusMessageFilterFactory.messageClass(StatusDataProviderDictionary.class)).or(
                        BusMessageFilterFactory.messageClass(StatusConfigurationInfo.class)))
                        .and(BusMessageFilterFactory.messageOrigin(null)));
    }

    @Override
    public void shutdown() {
        subsys.getMessagingAccess().getAgentPresenceManager().removeAgentPresenceListener(this);
        subsys.getMessagingAccess().removeStatusMessageListener(this); 
    }
    

    
    @Override
    public void connected(AgentInfo... agents) {
        for ( AgentInfo ai : agents ) {
            if (ai.getType().compareTo(AgentInfo.AgentType.WORKER) < 0) {
                return;
            }
            if (!dictionaryProcessed.contains(ai.getName())) {
                mdAccumulator.addToQueue(new CloseMetaData(ai.getName()));

                // It is no longer necessary to invoke the publication of the
                // Dictionary
                // on connecting agents.
                // For now we delay the request and check if it has already been
                // received.            
                // Trigger the dictionary publication
                subsys.getScheduler().schedule(() -> {
                    if (!dictionaryProcessed.contains(ai.getName())) {
                        subsys.getLogger().warn("Requesting publication of dictionary. This should not be necessary");
                        cmu.sendAsynchronousCommand(new CommandRequest(ai.getName(), "publishDataProviderDictionary"));
                    }
                },
                         3, TimeUnit.SECONDS);
            }
        }
    }
    
    @Override
    public void disconnecting(AgentInfo ai) {
        if (ai.getType().compareTo(AgentInfo.AgentType.WORKER) < 0 ) {
            return;
        }
        mdAccumulator.addToQueue(new CloseMetaData(ai.getName()));
        dictionaryProcessed.remove(ai.getName());
    }
    
    @Override
    public void onStatusMessage(StatusMessage s) {
        if (s instanceof StatusData) {
            KeyValueDataList encodedData = (KeyValueDataList) s.getEncodedData();
            if (encodedData != null) {
                processEncodedData(s.getOriginAgentInfo().getName(), encodedData);
            }
        } else if (s instanceof StatusDataProviderDictionary) {
            String name = s.getOriginAgentInfo().getName();
            log.debug("received a data dictionary for " + name);
            if(dictionaryProcessed.contains(name)) {
                return;
            }
            dictionaryProcessed.add(name);
            StatusDataProviderDictionary sdpd = (StatusDataProviderDictionary)s;
            DataProviderDictionary dpd = sdpd.getObject();
            // A dictionary of data is received : resolving the datadescs.
            RegisterDataGroup rdg = new RegisterDataGroup(s.getOriginAgentInfo());
            addToQueue(new Object[]{rdg, dpd});
            for (DataProviderInfo dpi : dpd.getDataProviderInfos()) {
                // get the attributes of each DataProviderInfo and persist 
                // the ones that correspond to metadata but only for TRENDING
                // or MONITORING data
                String typeStr = dpi.getAttributeValue(Attribute.DATA_TYPE);
                if ( typeStr != null ) {
                    Type dataType = Type.valueOf(typeStr);
                    if (dataType == Type.MONITORING || dataType == Type.TRENDING) {
                        for(Attribute attr : dpi.getAttributes()) {
                            if(attr.isMetadata()) {
                                queueImmediateMetaData(s.getOriginAgentInfo().getAgentStartTime().getUTCInstant().toEpochMilli(), new DataPath(name, dpi.getPath()), attr.getName(), dpi.getAttributeValue(attr));
                            }
                        }
                    }
                }            
            }
        } else if ( s instanceof StatusConfigurationInfo ) {
            ConfigurationInfo ci = ((StatusConfigurationInfo) s).getConfigurationInfo();
            log.fine("Processing configurationInfo for "+((StatusConfigurationInfo) s).getOriginAgentInfo().getName());
            Map<String,ChannelMetadata> channelsMetadata = new HashMap<>();
            for ( ConfigurationParameterInfo par : ci.getAllParameterInfo() ) {
                String parName = par.getParameterName();
                if ( parName.equals("limitLo") || parName.equals("limitHi") || parName.equals("dbandHi") || parName.equals("dbandLo") )  {
                    String channelPath = par.getComponentName();
                    ChannelMetadata channelMetadata = channelsMetadata.getOrDefault(channelPath, new ChannelMetadata(channelPath));
                    channelMetadata.setMetaValue(parName, par.getCurrentValue());
                    channelsMetadata.put(channelPath, channelMetadata);
                }
            }
            for ( ChannelMetadata channelMetadata : channelsMetadata.values() ) {
                queueImmediateMetaData(s.getCCSTimeStamp().getUTCInstant().toEpochMilli(),
                            new DataPath(s.getOriginAgentInfo().getName(), channelMetadata.channelPath), 
                            "alarmLow", channelMetadata.getAlarmLow());
       
                queueImmediateMetaData(s.getCCSTimeStamp().getUTCInstant().toEpochMilli(),
                            new DataPath(s.getOriginAgentInfo().getName(), channelMetadata.channelPath), 
                            "alarmHigh", channelMetadata.getAlarmHigh());
            
                queueImmediateMetaData(s.getCCSTimeStamp().getUTCInstant().toEpochMilli(),
                            new DataPath(s.getOriginAgentInfo().getName(), channelMetadata.channelPath), 
                            "warningLow", channelMetadata.getWarningLow());
            
                queueImmediateMetaData(s.getCCSTimeStamp().getUTCInstant().toEpochMilli(),
                            new DataPath(s.getOriginAgentInfo().getName(), channelMetadata.channelPath), 
                            "warningHigh", channelMetadata.getWarningHigh());
                log.fine("Updating metadata for "+channelMetadata.channelPath+": alarmLow("+channelMetadata.getAlarmLow()+") "
                        + "warningLow("+channelMetadata.getWarningLow()+") warningHigh("+channelMetadata.getWarningHigh()+") alarmHigh("+channelMetadata.getAlarmHigh()+")");
            }
        }
    }



    /**
     * Hack class used to get metadata quantities out of ConfigurationInfo.
     * This addresses limitHi/Lo and dbandHi/Lo and converts them to alarmHigh/Low 
     * and warningHigh/warningLow.
     * 
     */
    class ChannelMetadata {
        
        private final String channelPath;
        private double limitLo, limitHi, dbandLo, dbandHi;
        
        ChannelMetadata(String channelPath) {
            this.channelPath = channelPath;
        }
        
        void setMetaValue(String parName, String parValue) {
            double val = Double.parseDouble(parValue);
            if ( parName.equals("limitLo") ) {
                limitLo = val;
            } else if ( parName.equals("limitHi") ) {
                limitHi = val;
            } else if( parName.equals("dbandHi") ) {
                dbandHi = val;
            } else if ( parName.equals("dbandLo") ) {
                dbandLo = val;
            } else { 
                throw new IllegalArgumentException("Illegal Parameter name "+parName);
            }
        }
        
        String getAlarmHigh() {
            return String.valueOf(limitHi);
        }

        String getWarningHigh() {
            return String.valueOf(dbandHi <= 0 ? limitHi : limitHi - dbandHi);
        }


        String getWarningLow() {
            return String.valueOf(dbandLo <= 0 ? limitLo : limitLo + dbandLo);
        }

        
        String getAlarmLow() {
            return String.valueOf(limitLo);
        }
        
    }
    
    
    private void queueImmediateMetaData(long tStamp, DataPath name, String metadata, String value) {
        MetaDataData md = new MetaDataData(tStamp);
        md.setName(metadata);
        md.setValue(value);
        addToQueue(new Object[] { md, name });
    }
    
    //This method is no longer private so that it can be invoked externally
    //by other components of the localdb subsystem.
    //It has to be synchronized to regulate access to it.
    public synchronized void processEncodedData(String source, KeyValueDataList encodedData) {
        // Status data are persisted as key value pairs
        for (KeyValueData d : encodedData) {
            KeyValueData.KeyValueDataType type = d.getType();
            if (null != type) {
                switch (type) {
                    case KeyValueTrendingData:
                        queueImmediateScalar(source, d);
                        break;
                    case KeyValueMetaData:
                        String metaPath = d.getKey();
                        int lastIndex = metaPath.lastIndexOf('/');
                        String metaname = metaPath.substring(lastIndex + 1);
                        //Since we are now getting alarm/warning High/Low from
                        //the ConfigurationInfo message we no longer use the metadata publication
                        //for these quantities.
                        //This test can be removed once we are sure that these values are no longer
                        //published as metadata.
                        if ( metaname.equals("alarmHigh") || metaname.equals("alarmLow") ||
                                metaname.equals("warningHigh") || metaname.equals("warningLow")
                                ) {
                            break;
                        }
                        String key = metaPath.replace("/" + metaname, "");
                        queueImmediateMetaData(d.getCCSTimeStamp().getUTCInstant().toEpochMilli(), new DataPath(source, key), metaname, (String) d.getValue());
                        break;
                    default:
                        break;
                }
            }
        }
        
    }

    private void queueImmediateScalar(String source, KeyValueData kvd) {

        long tStamp = kvd.getCCSTimeStamp().getUTCInstant().toEpochMilli();
        DataPath name = new DataPath(source, kvd.getKey());
        Object d = kvd.getValue();
        
        log.debug("got update " + name);
        RawData data = new RawData();
        data.setTime(tStamp);
        if (d instanceof Double) {
            data.setDoubleData((Double) d);
        } else if (d instanceof Float) {
            data.setDoubleData(((Float) d).doubleValue());
        } else if (d instanceof Integer) {
            data.setDoubleData(((Integer) d).doubleValue());
        } else if (d instanceof Short) {
            data.setDoubleData(((Short) d).doubleValue());
        } else if (d instanceof Long) {
            data.setDoubleData(((Long) d).doubleValue());
        } else {
            // TODO: if array do an Array.toString
            data.setStringData(String.valueOf(d));
        }
        addToQueue(new Object[] { data, name });
    }

    private void persistData(RawData data, DataPath name, Session sess) {
        if(data.getDoubleData() != null && Double.isNaN(data.getDoubleData())) {
            return;
        }
        DataDesc dd = getDataDescription(name, "trending", sess);
        if (dd == null || dd == NULL_DESCRIPTION) {
            return;
        }
        data.setDataDesc(dd);
        sess.lock(dd, LockMode.NONE);
        sess.persist(data);

        // Queuing data for stat
        statAccumulator.addToQueue(data);
    }

    
    public StatDataPersister getStatAccumulator() {
        return statAccumulator;
    }
    
    private void persistMetadata(MetaDataData md, DataPath name, Session sess) {
        DataGroup dg = getDataGroup(name.getAgentName(), name.getDataName(), sess);
        md.setDataGroup(dg);
        mdAccumulator.addToQueue(md);
    }

    private void persistDataGroups(RegisterDataGroup rdg, DataProviderDictionary dpd, Session sess) {
        String agentName = rdg.getAgentName();
        boolean useFullPaths = rdg.isUseFullPath();


        //Mark as inactive DataDesc that don't belong to the dictionary
        List<DataPath> dictionaryPaths = new ArrayList<>();
        for ( DataProviderInfo dpi : dpd.getDataProviderInfos() ) {
            String typeStr = dpi.getAttributeValue(Attribute.DATA_TYPE);
            if ( typeStr != null ) {
                try {
                    Type dataType = Type.valueOf(typeStr);
                    if (dataType == Type.MONITORING || dataType == Type.TRENDING) {
                        dictionaryPaths.add(new DataPath(agentName, useFullPaths ? dpi.getPath() : dpi.getKey()));
                    }
                } catch (Exception e) {
                    log.warning("Illegal type conversion from "+typeStr);
                }
            }
        }
        
        for (Entry<DataPath,DataDesc> entry : map.entrySet()) {
            boolean isInDictionary = dictionaryPaths.remove(entry.getKey());
            DataDesc dataDesc = entry.getValue();            
            if ( dataDesc.getDataPath() == null || !dataDesc.getDataPath().getAgentName().equals(agentName) ) {
                continue;
            }
            if ( !dataDesc.getActive()  ) {
                if ( isInDictionary ) {
                    dataDesc.setActive(true);
                    sess.saveOrUpdate(dataDesc);
                }
            } else {
                if ( ! isInDictionary ) {
                    log.warning("Disabling Data description for "+entry.getKey());
                    dataDesc.setActive(false);
                    sess.saveOrUpdate(dataDesc);
                }
            }
        }
        
        for (String group : dpd.getGroups()) {
            for (DataProviderInfo dpi : dpd.getDataProviderDescriptionsForGroup(group)) {
                DataDesc dd = getDataDescription(new DataPath(agentName, useFullPaths ? dpi.getPath() : dpi.getKey()), "trending", sess);
                rdg.addDataDescToGroup(group, dd);
            }
        }
        log.debug("persisting data group information");
        // A new dictionary is received
        /** creating/updating group members. */
        for(Map.Entry<String, List<DataDesc>> entry : rdg.getDataDescs().entrySet()) {
            String groupName = entry.getKey();
            DataGroup dg = getDataGroup(agentName, groupName, sess);
            log.debug("updating data group " + groupName + " for " + agentName);
            for(DataDesc dd : entry.getValue()) {
                if(!dg.getMembers().containsKey(dd.getDataPath())) {
                    dg.addMember(dd);
                }
            }
        }
        sess.flush();
    }
   
    private DataDesc getDataDescription(DataPath key, String type, Session sess) {
        Query q = sess.getNamedQuery("findDataDesc").setString("agentName", key.getAgentName()).setString("dataName", key.getDataName());
        DataDesc dd = (DataDesc) q.uniqueResult();

        // TODO we should not do that in production.
        // this piece of code will make ALL data passing on the bus be
        // persisted by creating some default data description for it.
        // in preliminary version, datadesc should be populated by hand
        // TODO create GUI to allow user to select what should be persisted
        if (dd == null) {
            // in the map, we only use lower case, to be case independent.
            // Check if there is no conflict with existing data
            // https://jira.slac.stanford.edu/browse/LSSTCCS-732

//            for (Entry<DataPath,DataDesc> entry : map.entrySet()) {
//                DataDesc dataDesc = entry.getValue();
//                if ( !dataDesc.getActive() ) {
//                    continue;
//                }
//                DataPath existingKey = entry.getKey();
//                if (!areDataNamesConsistent(existingKey.getFullKey(), key.getFullKey())) {
//                    if (map.get(existingKey) == NULL_DESCRIPTION) {
//                        continue;
//                    }
//                    log.warning("Cannot add data description for " + key + " as it's in conflict with " + existingKey);
//                    map.put(key, NULL_DESCRIPTION);
//                    return NULL_DESCRIPTION;
//                }
//            }
            dd = new DataDesc();
            dd.setDataPath(key);
            dd.setDataType(type);
            sess.persist(dd);
                        
            log.fine("Adding default Data Description for " + key + ": " + dd.getId());
            map.put(key, dd);
//            addDefaultStatDescsFor(dd, sess);
        }
        return dd;
    }
    
    private DataGroup getDataGroup(String agentName, String groupName, Session sess) {
        Query q = sess.getNamedQuery("findDataGroup").setString("agentName", agentName).setString("groupName", groupName);
        DataGroup dg = (DataGroup) q.uniqueResult();
        if (dg == null) {
            // create and persist an empty group
            dg = new DataGroup(agentName, groupName);
            sess.persist(dg);
        }
        return dg;
    }

    public static List<StatDesc> getStatDescs(DataDesc dd, Session sess) {
        List<StatDesc> stats = sess.getNamedQuery("findStatDesc").setEntity("dd", dd)
                .setFlushMode(FlushMode.COMMIT)
                .setCacheable(true)
                .list();
        if (stats.isEmpty()) {
            try {
            for (long statInterval : DEFAULT_TIME_BIN_WIDTH) {
                StatDesc defaultSD = new StatDesc();
                defaultSD.setDataDesc(dd);
                defaultSD.setTimeBinWidth(statInterval);
                sess.persist(defaultSD);
                stats.add(defaultSD);
                log.fine("Adding default statistical binning of " + statInterval + " milliseconds to " + dd.getDataPath());
            }
            } catch (Exception e) {
                log.warn("Problem getting statistical data bin for "+dd.getDataPath()+" "+dd.getId());
                throw e;
            }
        }
        return stats;
    }

    /**
     * Check that the current data, loaded from the database is internally
     * consistent according to the specifications provided at
     * https://jira.slac.stanford.edu/browse/LSSTCCS-732. If it isn't it will
     * print out a warning message for the offending channels. The data will not
     * be touched.
     *
     */
    private void analizeDataMapIntegrity() {
//        for (DataPath existingKey1 : map.keySet()) {
//            for (DataPath existingKey2 : map.keySet()) {
//                if (existingKey1.equals(existingKey2)) {
//                    continue;
//                }
//                if (!areDataNamesConsistent(existingKey1.getFullKey(), existingKey2.getFullKey())) {
//                    log.warning("Conflict for existing keys: " + existingKey1 + " and " + existingKey2);
//                }
//            }
//        }
    }

    /**
     * Check if two data names are consistent according to
     * https://jira.slac.stanford.edu/browse/LSSTCCS-732. We check that neither
     * names starts with the other and that they have different levels of
     * directories. This last bit is so that we accept the situation where the
     * leaf name differs by only the ending: a.b.c and a.b.ca are consistent.
     * a.b and a.bc.d are consistent. a.b and a.b.c are not.
     *
     * @param name1
     *            the first data name
     * @param name2
     *            the second data name
     * @return true/false
     */
    static boolean areDataNamesConsistent(String name1, String name2) {
        if (name1.equals(name2)) {
            return true;
        }

        String[] array1 = name1.split("/");
        String[] array2 = name2.split("/");

        for (int i = 0; i < Math.min(array1.length, array2.length); i++) {
            if (!array1[i].equals(array2[i])) {
                return true;
            }
        }
        return array1.length == array2.length;
    }

    /**
     * Here is where the dataDescs must be resolved.
     * @param obj
     * @param sess 
     */
    @Override
    public void persist(Object[] obj, Session sess) {
        Object toPersist = obj[0];
        if (toPersist instanceof RegisterDataGroup) {
        
            persistDataGroups((RegisterDataGroup)toPersist, (DataProviderDictionary)obj[1], sess);
        } else {
            DataPath name = (DataPath) obj[1];
            if (toPersist instanceof RawData) {
                persistData((RawData) toPersist, name, sess);
            } else if (toPersist instanceof MetaDataData) {
                persistMetadata((MetaDataData) toPersist, name, sess);
            } else {
                if (toPersist != null) {
                    sess.persist(toPersist);
                }
            }
        }
    }
}
