package org.lsst.ccs.localdb.statusdb;

import java.math.BigInteger;
import java.time.Duration;
import java.util.Properties;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.hibernate.SQLQuery;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.lsst.ccs.Subsystem;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.AgentPeriodicTask;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.statusdb.utils.CacheStatistics;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.localdb.utils.InfluxDbClientService;
import org.lsst.ccs.services.AgentPeriodicTaskService;
import org.lsst.ccs.startup.HasCommandLineOptions;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Subsystem wrapper around a StatusDataPersister object
 */
public class StatusPersisterSubsystem extends Subsystem implements HasCommandLineOptions, HasLifecycle {
    private final Logger log = Logger.getLogger("org.lsst.ccs.localdb");
    
    private static SessionFactory fac;
    
    private Properties p;
    
    @LookupField(strategy=Strategy.TREE)
    AgentPeriodicTaskService periodicTaskService;
    
    @LookupField(strategy=Strategy.TREE)
    StatusDataPersister statusDataPersister;

    @LookupField(strategy=Strategy.TREE)
    InfluxDbClientService influxDbClientService;
    
    // define the command line options
    private final Options commandLineOptions = new Options();
    
    private static synchronized void init(Properties p) {
        fac = StatusdbUtils.getSessionFactory(p);
    }
    
    public StatusPersisterSubsystem()  {
        super("localdb", AgentInfo.AgentType.SERVICE);
        
        if (p == null) {
            p = BootstrapResourceUtils.getBootstrapSystemProperties(); 
        }
        commandLineOptions.addOption("u", "update", false, 
                "Start the Trending Ingest Module in update mode.");     
    }

    @Override
    public void build() {
        periodicTaskService.scheduleAgentPeriodicTask(new AgentPeriodicTask("statistics-publish", 
                () -> {
                    CacheStatistics stats = new CacheStatistics(fac, influxDbClientService.getInfluxDbClient());
                    if ( influxDbClientService.getInfluxDbClient() == null ) {
                        statusDataPersister.processEncodedData(getName(), stats.getStatData());
                    } 
                }
        ).withIsFixedRate(true).withPeriod(Duration.ofMinutes(1)));
    }
    
    @Override
    public void init() {
        init(p);
    }
    
    @Override
    public void startAgent() {
        // Make sure that migration of AgentState table has been made
        Session sess = fac.openSession();
        
        int nEntries = 0;
        boolean oldBridgeTableExists = true;
        try {
            sess.createSQLQuery("SELECT 1 FROM ccs_agentSt_innerStDe_b LIMIT 1").list();
        } catch (Exception ex) {
            // bridge table does not exist
            oldBridgeTableExists = false;
        }
        if (oldBridgeTableExists) {
            SQLQuery somethingToMigrate = sess.createSQLQuery("select count(*) from ccs_agentSt_innerStDe_b");
            nEntries = ((BigInteger)somethingToMigrate.uniqueResult()).intValue();
        }
        sess.close();

        if(nEntries > 0 ) {
            throw new RuntimeException("localdb cannot start because the MigrateAgentStates application has not been run yet. It is available for localdb version 3.0.x");
        }
        
        super.startAgent();
    }
    
    @Override
    public void processCommandLineOptions(String[] args) throws ParseException {
        CommandLineParser parser = new BasicParser();
        try {
            CommandLine line = parser.parse(commandLineOptions, args, true);
            
            if (line.hasOption("update")) {
                p.setProperty("hibernate.hbm2ddl.auto", "update");
            }

        } catch (Exception e) {
            e.printStackTrace();
            printHelp();
        }
    }

    @Override
    public void printHelp() {
        HelpFormatter formatter = new HelpFormatter();
        formatter.printHelp(100, "localdb", "", commandLineOptions, "", true);
    }
    
    
    
}
