package org.lsst.ccs.localdb.configdb;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CountDownLatch;
import org.lsst.ccs.Agent;
import org.lsst.ccs.AlertService;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.config.ConfigurationDescription;
import org.lsst.ccs.config.ConfigurationServiceException;
import org.lsst.ccs.config.ConfigurationView;
import org.lsst.ccs.config.ConfigurationDAO;
import org.lsst.ccs.config.PackCst;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.configdb.model.Description;
import org.lsst.ccs.localdb.configdb.model.HibernateDAO;
import org.lsst.ccs.localdb.configdb.model.ConfigurationInfoData;
import org.lsst.ccs.localdb.statusdb.AgentPresenceEvent;
import org.lsst.ccs.localdb.statusdb.LocalDBAlert;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * A CCS component that stores Configuration information in two different ways,
 * depending on an agent's chosen protocol.
 * <UL>
 * <LI/> Listening on the status bus
 * <LI/> Responding to configuration commands on the command bus
 * </UL>
 *
 * @author LSST CCS Team
 */
public class ConfigurationPersister implements ConfigurationDAO, HasLifecycle  {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.configdb");
    
    private ConfigurationFacade facade;
    
    /** A map of alive agents to their running description. */
    private final Map<String, Description> aliveDescriptions = new HashMap<>();
    
    /** Synchronization between Status and Command bus when the persister is in active mode for an agent. */
    private final Map<String, CountDownLatch> descriptionReady = new HashMap<>();
    
    @LookupField(strategy = Strategy.TOP)
    private Agent s;

    @LookupField(strategy = Strategy.TREE)
    private AlertService alertService;
    
    private final BlockingQueue<Object[]> ciQueue = new ArrayBlockingQueue<>(1000);
    
    /**
     * This thread serializes access to the configuration database for
     * {@code ConfigurationInfo} objects received on the status bus.
     */
    private final Thread statusConfigurationInfoSerializer;
    
    public ConfigurationPersister() {
        statusConfigurationInfoSerializer = new Thread(
                new ConfigurationInfoPersister()
        );
    }

    private class ConfigurationInfoPersister implements Runnable {

        @Override
        public void run() {
            while (true) {
                try {
                    Object[] o = ciQueue.take();
                    Object key = o[0];
                    if (key instanceof ConfigurationInfoData) {
                        ConfigurationInfoData cid = (ConfigurationInfoData) key;
                        String name = cid.getAgentDesc().getAgentName();
                        StatusConfigurationInfo sci = (StatusConfigurationInfo) o[1];
                        Description desc = aliveDescriptions.get(name);
                        if (desc == null) {
                            // Initialization phase
                            processFirstStatusConfigurationInfo(sci, cid);
                        } else {
                            if (descriptionReady.get(name) != null) {
                                // active mode
                                descriptionReady.get(name).countDown();
                                facade.registerStandaloneStatusConfigurationInfo(aliveDescriptions.get(name), cid);
                            } else {
                                // passive mode
                                log.fine("received configuration state at : " + sci.getTimeStamp() + " : " + sci.getConfigurationInfo().getConfigurationName());
                                facade.registerConfiguration(aliveDescriptions.get(name), sci.getConfigurationInfo(), cid);
                            }
                        }
                    } else if (key instanceof AgentPresenceEvent) {
                        AgentPresenceEvent ape = (AgentPresenceEvent) key;
                        AgentInfo agent = ape.getAgent();
                        if (ape.isConnecting()) {
                            String remote = agent.getAgentProperty(PackCst.IS_CONFIG_REMOTE);
                            if ("true".equals(remote)) {
                                log.finest("got connection event from " + agent.getName());
                                descriptionReady.put(agent.getName(), new CountDownLatch(1));
                            }
                        } else {
                            log.finest("got disconnection event from " + agent.getName());
                            Description desc = aliveDescriptions.remove(agent.getName());
                            if (desc != null) {
                                // setting a stop date to the current runs
                                facade.stopConfigurationRuns(desc, ape.getTime());
                            }
                        }
                    } else {
                        log.info("configuration persister received unexpected object " + key.getClass());
                    }
                } catch (InterruptedException ex) {
                        log.info("configuration info serializer thread interrupted", ex);
                        return;
                } catch(Exception ex) {
                    alertService.raiseAlert(LocalDBAlert.ConfigurationPersistenceException.getAlert(null, ex), AlertState.WARNING, LocalDBAlert.getFirstException(ex));
                }
            }
        }
    }
    
    @Override
    public void init() {
        s.setAgentProperty(PackCst.CONFIG_DB, "true");
    }
    
    @Override
    public void start() {
        facade = new ConfigurationFacade(new HibernateDAO(StatusdbUtils.getSessionFactory(null)));
        statusConfigurationInfoSerializer.start();
    }
    
    @Override
    public void shutdownNow() {
        statusConfigurationInfoSerializer.interrupt();
    }

    /** Read only operations. */
    /**
     * 
     * @param agentName
     * @param configDescription
     * @return
     * @throws ConfigurationServiceException 
     */
    @Command
    @Override
    public ConfigurationView loadConfiguration(String agentName, ConfigurationDescription configDescription) throws ConfigurationServiceException {
        try {
            return facade.loadConfiguration(descriptionWaitAndGet(agentName),configDescription);
        } catch (Exception ex) {
            throw new ConfigurationServiceException("error at persistence layer", ex);
        }
    }
    
    @Command
    @Override
    public ConfigurationView loadGlobalConfiguration(String agentName, String name, int version) {
        try {
            return facade.loadGlobalConfiguration(descriptionWaitAndGet(agentName), name, version);
        } catch (Exception ex) {
            throw new ConfigurationServiceException("error at persistence layer", ex);
        }
    }

    @Command
    @Override
    public Set<String> findAvailableConfigurationsForCategory(String agentName, String category) {
        try {
            return facade.findAvailableConfigurationsForCategory(descriptionWaitAndGet(agentName), category);
        } catch (Exception ex) {
            throw new ConfigurationServiceException("error at persistence layer", ex);
        }
    }

    @Command
    @Override
    public boolean isAvailable() {
        return true;
    }

    /** Read-write operations. */
    /**
     * @param agentName
     * @param configInfo
     * @return the resulted configuration versions.
     * @throws ConfigurationServiceException 
     */
    @Command
    @Override
    public ConfigurationDescription registerConfiguration(String agentName, ConfigurationInfo configInfo) throws ConfigurationServiceException {
        try {
            return facade.registerStandaloneConfiguration(descriptionWaitAndGet(agentName), configInfo);
        } catch (Exception ex) {
            throw new ConfigurationServiceException("error at persistence layer", ex);
        }
    }
    
    @Override
    public ConfigurationDescription saveChangesForCategoriesAs(String agentName, ConfigurationDescription configDesc, ConfigurationInfo configInfo) throws ConfigurationServiceException {
        return null;
        // empty implementation
    }
    
    private Description descriptionWaitAndGet(String agentName) {
        try {
            descriptionReady.get(agentName).await();
            return aliveDescriptions.get(agentName);
        } catch (Exception ex) {
            throw new RuntimeException("cannot get description for " + agentName, ex);
        }
    }
    
    private void processFirstStatusConfigurationInfo(StatusConfigurationInfo sci, ConfigurationInfoData cid) throws Exception {
        // Registering the description
        ConfigurationInfo ci = sci.getConfigurationInfo();
        String agentName = cid.getAgentDesc().getAgentName();
        Description desc = facade.registerDescription(cid.getAgentDesc(), ci.getAllParameterInfo());
        log.finest("registered description for " + agentName + " with id " + desc.getId());
        // Initiates the configuration runs
        facade.cleanupConfigurationRuns(desc, cid.getTime());
        facade.registerConfiguration(desc, ci, cid);
        log.finest("registered initial configuration info : " + ci.getConfigurationName() + " at " + sci.getTimeStamp());
        // Keeping the description in memory as long as the agent in on the buses.
        aliveDescriptions.put(agentName, desc);
    }
    
    public void submit(Object[] objs) {
        ciQueue.offer(objs);
    }
}
