package org.lsst.ccs.localdb.statusdb;

import java.util.ArrayList;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.hibernate.FlushMode;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.resource.transaction.spi.TransactionStatus;
import org.lsst.ccs.AlertService;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.framework.ClearAlertHandler;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Persists entities as a batch.
 *
 * @author LSST CCS Team
 */
public abstract class BatchPersister implements Runnable, ClearAlertHandler {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.statusdb");
    
    private final String persisterName = this.getClass().getSimpleName();
    private static final int CONSECUTIVE_MAX_CAPACITY_LIMIT = 15;
    private static final String MAX_CAPACITY_ALERT_ID = "MaxCapacityAlert";
    
    private final Queue<Object[]> rq = new ConcurrentLinkedQueue<>();
    
    private final AlertService alertService;
    
    private int maxCapacityCount = 0;
    
    protected final SessionFactory fac;
    
    public BatchPersister(SessionFactory fac, AlertService as) {
        this.alertService = as;
        alertService.addClearAlertHandler(this);
        this.fac = fac;
    }
    
    public SessionFactory getSessionFactory() {
        return fac;
    }
    
    @Override
    public ClearAlertCode canClearAlert(Alert alert) {
        if(alert.getAlertId().equals(persisterName+"_"+MAX_CAPACITY_ALERT_ID)) {
            maxCapacityCount = 0;
            return ClearAlertCode.CLEAR_ALERT;
        } else {
            return ClearAlertCode.UNKWNOWN_ALERT;
        }
    }
    
    @Override
    public void run() {
        
        ArrayList<Object[]> workingList = new ArrayList<>(1000);
        boolean empty = false;
        
        while (!empty) {
            // get at most 1000 objets from the queue
            for (int i = 0; i < 1000; i++) {
                Object[] d = rq.poll();
                if (d == null) {
                    maxCapacityCount = 0;
                    break;
                }
                workingList.add(d);
                if(i == 999) {
                    maxCapacityCount ++;
                }
            }
            
            if(maxCapacityCount == CONSECUTIVE_MAX_CAPACITY_LIMIT) {
                alertService.raiseAlert(new Alert(persisterName+"_"+MAX_CAPACITY_ALERT_ID, persisterName + " running at full capacity"), AlertState.WARNING, "current queue size : " + rq.size());
                maxCapacityCount = 0;
            }
            
            if (workingList.isEmpty()) {
                // Will sleep 1 second
                empty = true;
            } else {
                log.debug(persisterName + " start batch processing of " + workingList.size() + " entities.");
                Session sess = fac.openSession();
                sess.setFlushMode(FlushMode.COMMIT);
                Transaction tx = null;
                try {
                    tx = sess.beginTransaction();
                    for (Object[] o : workingList) {
                        try {
                            persist(o, sess);
                        } catch (Exception e) {
                            throw new Exception("Failed to persist object ("+o[1]+") "+o[0], e);
                        }
                    }
                    tx.commit();
                } catch (Exception ex) {
                    log.error("caught exception when persisting : " + ex, ex);
                    try {
                        if (tx != null && (
                                tx.getStatus() == TransactionStatus.ACTIVE
                                || tx.getStatus() == TransactionStatus.MARKED_ROLLBACK)) {
                            tx.rollback();
                        }
                    } catch (Exception rbEx) {
                        log.error("Rollback of transaction failed : " + rbEx, rbEx);
                    }
                } finally {
                    if (sess.isOpen()) {
                        sess.close();
                    }
                    workingList.clear();
                }
            }
        }
        log.debug(this.getClass().getSimpleName() + " leaving batch processing loop");
    }
    
    public void addToQueue(Object[] obj) {
        rq.add(obj);
    }
    
    /**
     * Called from inside an open transaction.
     *
     * @param obj
     * @param sess
     */
    public abstract void persist(Object[] obj, Session sess);
    
}
