package org.lsst.ccs.localdb.configdb;

import java.util.AbstractMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.CountDownLatch;
import org.hibernate.SessionFactory;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.config.ConfigurationDescription;
import org.lsst.ccs.config.ConfigurationServiceException;
import org.lsst.ccs.config.ConfigurationView;
import org.lsst.ccs.config.ConfigurationDAO;
import org.lsst.ccs.config.PackCst;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.configdb.model.Description;
import org.lsst.ccs.localdb.configdb.model.HibernateDAO;
import org.lsst.ccs.localdb.configdb.model.ConfigurationInfoData;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * A CCS component that stores Configuration information in two different ways,
 * depending on an agent's chosen protocol.
 * <UL>
 * <LI/> Listening on the status bus
 * <LI/> Responding to configuration commands on the command bus
 * </UL>
 *
 * @author LSST CCS Team
 */

public class ConfigurationPersister implements ConfigurationDAO, AgentPresenceListener, HasLifecycle  {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.configdb");
    private final ConfigurationFacade facade;
    
    /** A map of alive agents to their running description. */
    private final Map<String, Description> aliveDescriptions = new HashMap<>();
    
    /** Allows to synchronize between Status and Command bus when the persister is in active mode for an agent. */
    private final Map<String, CountDownLatch> descriptionReady = new HashMap<>();
    
    @LookupField(strategy = Strategy.TOP)
    private Agent s;
    
    private final BlockingQueue<Map.Entry<ConfigurationInfoData, StatusConfigurationInfo>> ciQueue = new ArrayBlockingQueue<>(1000);
    
    /**
     * This thread serializes access to the configuration database for
     * {@code ConfigurationInfo} objects received on the status bus.
     */
    private final Thread statusConfigurationInfoSerializer;
    
    /**
     * @param agent
     * @param fac The session factory.
     */
    public ConfigurationPersister(Agent agent, SessionFactory fac) {
        s = agent;
        s.setAgentProperty(PackCst.CONFIG_DB, "true");
        facade = new ConfigurationFacade(new HibernateDAO(fac));
        statusConfigurationInfoSerializer = new Thread(
                new ConfigurationInfoPersister()
        );
    }
    
    private class ConfigurationInfoPersister implements Runnable {
        @Override
        public void run() {
            try {
                while (true) {
                    Map.Entry<ConfigurationInfoData, StatusConfigurationInfo> e = ciQueue.take();
                    ConfigurationInfoData cid = e.getKey();
                    String name = cid.getAgentDesc().getAgentName();
                    StatusConfigurationInfo sci = e.getValue();                   
                    Description desc = aliveDescriptions.get(name);
                    if (desc == null) {
                        // Initialization phase
                        processFirstStatusConfigurationInfo(sci, cid);
                    } else {
                        if (descriptionReady.get(name) != null) {
                            // active mode
                            descriptionReady.get(name).countDown();
                        } else {
                            // passive mode
                            log.fine("received  configuration state at : " + sci.getTimeStamp() + " : " + sci.getConfigurationInfo().getConfigurationName());
                            facade.registerConfiguration(aliveDescriptions.get(name), sci.getConfigurationInfo(), cid);
                        }
                    }
                }
            } catch (InterruptedException ex) {
                log.info("configuration info serializer thread interrupted", ex);
                Thread.currentThread().interrupt();
            }
        }
    }
    
    @Override
    public void start() {
        statusConfigurationInfoSerializer.start();
        
    }
    
    @Override
    public void shutdownNow() {
        statusConfigurationInfoSerializer.interrupt();
    }

    /** Read only operations. */
    /**
     * 
     * @param agentName
     * @param configDescription
     * @return
     * @throws ConfigurationServiceException 
     */
    @Command
    @Override
    public ConfigurationView loadConfiguration(String agentName, ConfigurationDescription configDescription) throws ConfigurationServiceException {
        return facade.loadConfiguration(descriptionWaitAndGet(agentName),configDescription);
    }
    
    @Command
    @Override
    public ConfigurationView loadGlobalConfiguration(String agentName, String name, int version) {
        return facade.loadGlobalConfiguration(descriptionWaitAndGet(agentName), name, version);
    }

    @Command
    @Override
    public Set<String> findAvailableConfigurationsForCategory(String agentName, String category) {
        return facade.findAvailableConfigurationsForCategory(descriptionWaitAndGet(agentName), category);
    }

    @Command
    @Override
    public boolean isAvailable() {
        return true;
    }

    /** Read-write operations. */
    /**
     * @param agentName
     * @param configInfo
     * @return the resulted configuration versions.
     * @throws ConfigurationServiceException 
     */
    @Command
    @Override
    public ConfigurationDescription registerConfiguration(String agentName, ConfigurationInfo configInfo) throws ConfigurationServiceException {
        return facade.registerConfiguration(descriptionWaitAndGet(agentName), configInfo, null);
    }
    
    @Override
    public ConfigurationDescription saveChangesForCategoriesAs(String agentName, ConfigurationDescription configDesc, ConfigurationInfo configInfo) throws ConfigurationServiceException {
        return null;
        // empty implementation
    }
    
    private Description descriptionWaitAndGet(String agentName) {
        try {
            descriptionReady.get(agentName).await();
            return aliveDescriptions.get(agentName);
        } catch (Exception ex) {
            throw new RuntimeException("cannot get description for " + agentName, ex);
        }
    }
    
    private void processFirstStatusConfigurationInfo(StatusConfigurationInfo sci, ConfigurationInfoData cid) {
        // Registering the description
        ConfigurationInfo ci = sci.getConfigurationInfo();
        String agentName = cid.getAgentDesc().getAgentName();
        try {
            Description desc = facade.registerDescription(cid.getAgentDesc(), ci.getAllParameterInfo());
            log.finest("registered description for " + agentName + " with id " + desc.getId());
            // Initiates the configuration runs
            log.finest("registering initial configuration info : " + ci.getConfigurationName() + " at " + sci.getTimeStamp());
            facade.cleanupConfigurationRuns(desc, cid.getTime());
            facade.registerConfiguration(desc, ci, cid);
            // Keeping the description in memory as long as the agent in on the buses.
            aliveDescriptions.put(agentName, desc);
        } catch (Exception ex) {
            log.error("cannot register description or persist first ConfigurationInfo", ex);
        }
    }
    
    @Override
    public void connecting(AgentInfo agent) {
        String remote = agent.getAgentProperty(PackCst.IS_CONFIG_REMOTE);
        if("true".equals(remote)) {
            log.finest("got connection event from " + agent.getName());
            descriptionReady.put(agent.getName(), new CountDownLatch(1));
        }
    }
    
    @Override
    public void disconnecting(AgentInfo agent) {
        log.finest("got disconnection event from " + agent.getName());
        long time = System.currentTimeMillis();
        Description desc = aliveDescriptions.remove(agent.getName());
        if (desc != null) {
            // setting a stop date to the current runs
            facade.stopConfigurationRuns(desc, time);
        }
    }

    public void submit(ConfigurationInfoData data, StatusConfigurationInfo sci) {
        ciQueue.offer(new AbstractMap.SimpleEntry<ConfigurationInfoData, StatusConfigurationInfo>(data, sci));
    }
}
