package org.lsst.ccs.localdb.configdb;


import java.rmi.RemoteException;
import java.util.List;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.localdb.configdb.model.ConfigProfile;
import org.lsst.ccs.localdb.configdb.model.ConfigurationFacade;
import org.lsst.ccs.localdb.configdb.model.DBInterface;
import org.lsst.ccs.localdb.configdb.model.DeprecationListener;
import org.lsst.ccs.localdb.configdb.model.ParameterConfiguration;
import org.lsst.ccs.localdb.configdb.model.PersistenceLayerException;
import org.lsst.ccs.localdb.configdb.model.SubsystemDescription;

/**
 * A raw implementation of Configuration Service that delegates to a <TT>ConfigurationFacade</TT>
 * object.
 * @author bamade
 */
// Date: 06/06/12

public class ConfigurationServiceImpl implements ConfigurationService {
    ConfigurationFacade facade ;
    public static ConfigurationServiceImpl ON_DATABASE ;
    static {
        String clazzName = BootstrapResourceUtils.getBootstrapSystemProperties().getProperty("org.lsst.ccs.configDAO", "org.lsst.ccs.config.dao.hibernate.HibernateDAO");
        DBInterface dao = null;
        try {
            dao = (DBInterface) Class.forName(clazzName).newInstance();
            ON_DATABASE = new ConfigurationServiceImpl(new ConfigurationFacade(dao));
        } catch (Exception e) {
            throw new Error(e);
        }
    }

    public ConfigurationServiceImpl(ConfigurationFacade facade) {
        this.facade = facade ;
    }

    public ConfigurationFacade getFacade() {
        return facade ;
    }

    @Override
    public synchronized DescriptionResult registerSubsystemDescription(SubsystemDescription newDescription) throws RemoteException {
        final DescriptionResult res = new DescriptionResult();
        facade.setDeprecationListener(new DeprecationListener() {
            @Override
            public void subsystemDeprecating(SubsystemDescription description) {
                res.setDescriptionDeprecationEvent(new DescriptionDeprecationEvent(description, ""));
            }

            @Override
            public void configProfileDeprecating(ConfigProfile profile) {
                res.addProfileDeprecationEvent(new ProfileDeprecationEvent(profile, ""));
            }
        });
        try {
            SubsystemDescription description = facade.registerSubsystemDescription(newDescription);
            res.setResult(description);
            return res ;
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server registration fail", e);
        }
    }

    @Override
    public synchronized DescriptionResult deprecateSubsystemDescription(String subsystemName, String tag) throws RemoteException {
        final DescriptionResult res = new DescriptionResult();
        facade.setDeprecationListener(new DeprecationListener() {
            @Override
            public void subsystemDeprecating(SubsystemDescription description) {
                res.setDescriptionDeprecationEvent(new DescriptionDeprecationEvent(description, ""));
            }

            @Override
            public void configProfileDeprecating(ConfigProfile profile) {
                res.addProfileDeprecationEvent(new ProfileDeprecationEvent(profile, ""));
            }
        });
        try {
            SubsystemDescription deprecated = facade.deprecateSubsystemDescription(subsystemName, tag);
            res.setResult(deprecated);
            return res;
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server deprecation fail", e);
        } finally {
            facade.setDeprecationListener(null);
        }
    }

    @Override
    public SubsystemDescription getActiveSubsystemDescription(String name, String tag) throws RemoteException {

        try {
            return facade.getActiveSubsystemDescription(name, tag) ;
        } catch (PersistenceLayerException e) {
           throw new RemoteException("description query" , e) ;
        }
    }

    @Override
    public synchronized ProfileResult registerConfigProfile(ConfigProfile newProfile) throws RemoteException {
        final ProfileResult res = new ProfileResult();
        facade.setDeprecationListener(new DeprecationListener() {
            @Override
            public void subsystemDeprecating(SubsystemDescription description) {
            }

            @Override
            public void configProfileDeprecating(ConfigProfile profile) {
                res.setProfileDeprecationEvent(new ProfileDeprecationEvent(profile, ""));
            }
        });
        try {
            ConfigProfile profile = facade.registerConfigProfile(newProfile);
            res.setResult(profile) ;
            return res ;
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server registration fail", e);
        }
    }

    @Override
    public ProfileResult registerConfiguration(String subsystemName, String categoryName, String configurationName, String tag,
                                               String user, int level) throws RemoteException {
        SubsystemDescription description = getActiveSubsystemDescription(subsystemName, tag) ;

        
        ConfigProfile profile = facade.getActiveConfigProfile(subsystemName, description.getTag(), configurationName, categoryName);
            
        return registerConfigProfile(profile) ;
    }

    @Override
    public synchronized ProfileResult deprecateConfigProfile(String subsystemName, String descriptionName, String name, String tag) throws RemoteException {
        final ProfileResult res = new ProfileResult();
        facade.setDeprecationListener(new DeprecationListener() {
            @Override
            public void subsystemDeprecating(SubsystemDescription description) {
            }

            @Override
            public void configProfileDeprecating(ConfigProfile profile) {
                res.setProfileDeprecationEvent(new ProfileDeprecationEvent(profile, ""));
            }
        });
        try {
            ConfigProfile deprecated = facade.deprecateConfigProfile(subsystemName, descriptionName, name, tag);
            res.setResult(deprecated);
            return res;
        } catch (PersistenceLayerException exc) {
            throw new RemoteException("server deprecation fail", exc);
        } finally {
            facade.setDeprecationListener(null);
        }
    }

    @Override
    public ConfigProfile getActiveConfigProfile(String subsystemName, String descriptionName, String name, String tag) throws RemoteException {
        try {
            return facade.getActiveConfigProfile(subsystemName, descriptionName, name, tag) ;
        } catch (PersistenceLayerException e) {
           throw new RemoteException("configuration query", e) ;
        }
    }

    @Override
    public ParameterConfiguration modifyParmConf(ParameterConfiguration engineeringModeParm) throws RemoteException {
        try {
            ParameterConfiguration res = facade.engineerParmConfig(engineeringModeParm) ;
            return res ;
        } catch (PersistenceLayerException exc) {
            throw new RemoteException("parameter configuration fail :", exc);
        }
    }


    @Override
    public ConfigProfile getConfigRunningAt(String subsystemName, String descriptionName, String category, long date) throws RemoteException {
        try {
            return facade.getConfigRunningAt(subsystemName, descriptionName, category, date);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public String getActiveValueAt(String subsystemName, String descriptionName, String category, String parameterPath, long date) throws RemoteException {
        try {
            return facade.getActiveValueAt(subsystemName,descriptionName, category, parameterPath, date);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public ConfigProfile getConfigValidAt(String subsystemName, String name, String tag, long date) throws RemoteException {
        try {
            return facade.getConfigValidAt(subsystemName,name, tag, date);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public String getValueValidAt(String subsystemName,  String profileName, String profileTag, String parameterPath, long date) throws RemoteException {
        try {
            return facade.getValueValidAt(subsystemName, profileName, profileTag, parameterPath, date);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }


    @Override
    public ConfigProfile getPrevious(ConfigProfile current) throws RemoteException {
        try {
            return facade.getPrevious(current);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public ConfigProfile getNext(ConfigProfile current) throws RemoteException {
        try {
            return facade.getNext(current);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public SubsystemDescription getPrevious(SubsystemDescription current) throws RemoteException {
        try {
            return facade.getPrevious(current);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server get fail", e);
        }
    }

    @Override
    public List<?> simpleHQLRequest(String hqlString) throws RemoteException {
        try {
            return facade.simpleHQLRequest(hqlString);
        } catch (PersistenceLayerException e) {
            throw new RemoteException("server request fail", e);
        }
    }
}
