package org.lsst.ccs.localdb.statusdb;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.TimeUnit;
import org.hibernate.Query;
import org.hibernate.Session;
import org.lsst.ccs.Agent;
import org.lsst.ccs.bus.data.AgentInfo;
import org.lsst.ccs.bus.data.AgentInfo.AgentType;
import org.lsst.ccs.bus.data.Alert;
import org.lsst.ccs.bus.data.RaisedAlertSummary;
import org.lsst.ccs.bus.messages.StatusAlert;
import org.lsst.ccs.bus.messages.StatusClearedAlert;
import org.lsst.ccs.bus.messages.StatusConfigurationInfo;
import org.lsst.ccs.bus.messages.StatusMessage;
import org.lsst.ccs.bus.messages.StatusRaisedAlert;
import org.lsst.ccs.bus.messages.StatusRuntimeInfo;
import org.lsst.ccs.bus.messages.StatusStateChangeNotification;
import org.lsst.ccs.bus.states.AlertState;
import org.lsst.ccs.bus.states.StateBundle;
import org.lsst.ccs.commons.annotations.LookupField;
import org.lsst.ccs.commons.annotations.LookupField.Strategy;
import org.lsst.ccs.framework.HasLifecycle;
import org.lsst.ccs.localdb.configdb.ConfigurationPersister;
import org.lsst.ccs.localdb.dao.LocaldbFacade;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.localdb.statusdb.model.AgentState;
import org.lsst.ccs.localdb.statusdb.model.AlertData;
import org.lsst.ccs.localdb.statusdb.model.AlertDesc;
import org.lsst.ccs.localdb.statusdb.model.ClearedAlertData;
import org.lsst.ccs.localdb.configdb.model.ConfigurationInfoData;
import org.lsst.ccs.localdb.statusdb.model.RaisedAlertData;
import org.lsst.ccs.localdb.statusdb.model.StateChangeNotificationData;
import org.lsst.ccs.localdb.statusdb.model.StatusMessageData;
import org.lsst.ccs.messaging.AgentPresenceListener;
import org.lsst.ccs.messaging.BusMessageFilterFactory;
import org.lsst.ccs.messaging.StatusMessageListener;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * @author LSST CCS Team
 */
public class StatusPersister extends BatchPersister<Object[]> implements StatusMessageListener, AgentPresenceListener, HasLifecycle {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.statusdb");
    
    private final List<String> agentsToCheck = new CopyOnWriteArrayList<>();
    
    @LookupField(strategy=Strategy.TREE)
    private ConfigurationPersister configurationPersister;
    
    @LookupField(strategy=Strategy.TOP)
    private Agent subsys;
    
    /** Lock to synchronize StatusListener and AgentPresenceListener callback methods. */
    private final Object listenerLock = new Object();
    
    public StatusPersister() {
        super(1000, true);
    }
    
    @Override
    public void start() {
        subsys.getMessagingAccess().getAgentPresenceManager().addAgentPresenceListener(this);
        subsys.getMessagingAccess().addStatusMessageListener(this,
                BusMessageFilterFactory.messageOrigin(null).and(
                        BusMessageFilterFactory.messageClass(StatusAlert.class)
                                .or(BusMessageFilterFactory.messageClass(StatusStateChangeNotification.class)
                                        .or(BusMessageFilterFactory.messageClass(StatusConfigurationInfo.class))
                                        .and( bm -> bm.getOriginAgentInfo().getType().compareTo(AgentInfo.AgentType.WORKER) >=0
                                        ))
                ));
        subsys.getScheduler().scheduleWithFixedDelay(this, 0, 1, TimeUnit.SECONDS);
    }
    
    @Override
    public void onStatusMessage(StatusMessage msg) {
        synchronized(listenerLock) {            
            if (msg instanceof StatusAlert) {
                
                StatusAlert statusAlert = (StatusAlert)msg;
                RaisedAlertSummary alertSummary = statusAlert.getRaisedAlertSummary();
                // Case 1 : received a StatusRaisedAlert
                if (statusAlert instanceof StatusRaisedAlert) {
                    String alertId = ((Alert)statusAlert.getEncodedData()).getAlertId();
                    String alertDescription = ((StatusRaisedAlert)statusAlert).getCause();
                    AlertState currentSeverity = alertSummary.getRaisedAlert(alertId).getLatestAlertState();
                    RaisedAlertData arad = new RaisedAlertData(currentSeverity, alertDescription, msg.getTimeStamp());
                    addToQueue(new Object[]{arad, statusAlert, alertId});
                } else if (statusAlert instanceof StatusClearedAlert) {
                    // Case 2 : received a StatusClearedAlert
                    StatusClearedAlert sca = (StatusClearedAlert)statusAlert;
                    String[] clearedAlerts = sca.getClearAlertIds();
                    if (clearedAlerts != null) {
                        for (String clearedId : clearedAlerts) {
                            ClearedAlertData cad = new ClearedAlertData();
                            cad.setTime(msg.getTimeStamp());
                            addToQueue(new Object[]{cad, statusAlert,clearedId});
                        }
                    }
                }
            } else if (msg instanceof StatusStateChangeNotification) {
                StatusStateChangeNotification statusSCN = (StatusStateChangeNotification)msg;
                StateChangeNotificationData scnd = new StateChangeNotificationData();
                scnd.setTime(msg.getTimeStamp());
                addToQueue(new Object[]{scnd,statusSCN});
            } else if (msg instanceof StatusConfigurationInfo) {
                StatusConfigurationInfo sci = (StatusConfigurationInfo)msg;
                ConfigurationInfoData cid = new ConfigurationInfoData();
                cid.setTime(msg.getTimeStamp());
                addToQueue(new Object[]{cid, sci});
            } else if (msg instanceof StatusRuntimeInfo) {
                addToQueue(new Object[]{msg, null});
            }
        }
    }
    
    /** 
     * AgentPresenceListener callback methods.
     * When a agent connects, its alert entries in ActiveRaisedAlertData are removed.
     * @param agent
     */
    @Override
    public void connecting(AgentInfo agent) {
        long time = System.currentTimeMillis();
        synchronized(listenerLock) {
            if ( agent.getType().compareTo(AgentType.WORKER) >= 0 ) {
                addToQueue(new Object[]{new AgentPresenceEvent(agent, time, true)});
                agentsToCheck.add(agent.getName());
            }
        }
    }
    
    /**
     * The active alerts for the disconnecting agent are pushed to history.
     * @param agent 
     */
    @Override
    public void disconnecting(AgentInfo agent) {
        long time = System.currentTimeMillis();
        synchronized(listenerLock) {
            if ( agent.getType().compareTo(AgentType.WORKER) >= 0 ) {
                addToQueue(new Object[]{new AgentPresenceEvent(agent, time, false)});
            }
        }
    }
    
    private void pushActiveAlerts(String agentName, long agentStartTime, Session sess ) {
        agentsToCheck.remove(agentName);
        log.info("updating active alerts for agent : " + agentName);
        Query q = sess.createQuery("from RaisedAlertData rad where rad.agentDesc.agentName=:name and rad.active=true and rad.time<=:t");
        q.setString("name", agentName);
        q.setLong("t", agentStartTime);
        List<RaisedAlertData> res = q.list();
        for (RaisedAlertData rad : res) {
            rad.setActive(false);
        }
        sess.flush();
    }
    
    /**
     * Called from inside a transaction.
     * This method is called sequentially : LocaldbFacade needs not be thread safe.
     * TO-DO : is this the right strategy ?
     * @param obj
     * @param sess 
     */
    @Override
    public void persist(Object[] obj, Session sess) {
        if(obj[0] instanceof StatusMessageData) {
            // Operations common to every status message
            StatusMessageData msgData = (StatusMessageData)obj[0];
            StatusMessage statusMsg = (StatusMessage)obj[1];
            
            AgentInfo ai = statusMsg.getOriginAgentInfo();
            StateBundle state = statusMsg.getState();
            
            // Fetching agent description from local cache or create it and persist it.
            AgentDesc agDesc = LocaldbFacade.getAgentDesc(ai, sess);
            msgData.setAgentDesc(agDesc);
            
            // State reconstruction.
            AgentState stateData = LocaldbFacade.getAgentState(agDesc, state, sess);
            msgData.setAgentState(stateData);
            
            // Operations specific to sub classes of status messages.
            if (msgData instanceof AlertData) {
                // The object to be persisted
                AlertData alData = (AlertData)msgData;
                
                // The original status
                StatusAlert statusAlert = (StatusAlert)statusMsg;
                
                // The alertId associated to
                String alertId = (String)obj[2];
                
                // Fetching alert description from local cache or create and persist it.
                if (statusAlert instanceof StatusRaisedAlert) {
                    Alert al = ((StatusRaisedAlert)statusAlert).getRaisedAlert();
                    AlertDesc ad = LocaldbFacade.getAlertDescOrPersist(agDesc, al, sess);
                    alData.setAlertDesc(ad);
                    sess.persist(alData);
                    
                } else if (statusAlert instanceof StatusClearedAlert) {
                    AlertDesc ad = LocaldbFacade.getAlertDesc(agDesc, alertId, sess);
                    if (ad != null) {
                        alData.setAlertDesc(ad);
                        sess.persist(alData);
                    } else {
                        log.warn("could not find description for alert with id " + alertId);
                    }
                    // Cleared flag is updated on the instances of the active cleared alert.
                    Query q = sess.createQuery("from RaisedAlertData rad where"
                            + " rad.agentDesc.agentName=:name and rad.alertDesc.alertId =:id and rad.active=true")
                            .setString("id", alertId).setString("name", agDesc.getAgentName());
                    List<RaisedAlertData> res = q.list();
                    for (RaisedAlertData arad : res) {
                        arad.setClearingAlert((ClearedAlertData)alData);
                        sess.update(arad);
                    }
                    log.fine("cleared " + res.size() + " instances of " + agDesc.getAgentName()+"/"+alertId);
                }
            } else if (msgData instanceof StateChangeNotificationData) {
                StateChangeNotificationData scnd = (StateChangeNotificationData)msgData;
                
                StateBundle oldState = ((StatusStateChangeNotification)statusMsg).getOldState();
                scnd.setOldState(LocaldbFacade.getAgentState(agDesc, oldState, sess));
                
                if(scnd.getOldState().getId() == scnd.getNewState().getId() && !state.equals(oldState)) {
                    log.warn("configuration persister considers old state and new state equal ("+scnd.getOldState().getId()+"). Diff actually is :\n" + state.diffState(oldState));
                    alertService.raiseAlert(LocalDBAlert.WrongStatePersistence.addData("oldState", oldState).addData("newState", state).getAlert(null, null),AlertState.WARNING, "for agent " + ai.getName());
                }
                sess.persist(scnd);
            } else if (msgData instanceof ConfigurationInfoData) {
                configurationPersister.submit(obj);
            } 
        } else if (obj[0] instanceof AgentPresenceEvent) {
            configurationPersister.submit(obj);
            AgentPresenceEvent ape = (AgentPresenceEvent)obj[0];
            if(!ape.isConnecting()) {
                pushActiveAlerts(ape.getAgent().getName(), ape.getTime(), sess);
            }
        } else if (obj[0] instanceof StatusRuntimeInfo) {
            StatusRuntimeInfo sri = (StatusRuntimeInfo)obj[0];
            if(agentsToCheck.contains(sri.getOriginAgentInfo().getName())) {
                pushActiveAlerts(sri.getOriginAgentInfo().getName(), sri.getTimeStamp()-sri.getRuntimeInfo().getUptime()*1000, sess);
            }
        }
    }
}
