/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package org.lsst.ccs.localdb.statusdb;

import java.util.List;
import java.util.concurrent.CountDownLatch;
import org.apache.commons.cli.BasicParser;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.CommandLineParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Options;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.lsst.ccs.localdb.statusdb.model.DataDesc;
import org.lsst.ccs.localdb.statusdb.model.RawData;
import org.lsst.ccs.localdb.statusdb.server.DataChannel;
import org.lsst.ccs.localdb.statusdb.server.DataServer;
import org.lsst.ccs.localdb.statusdb.utils.StatusdbUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 *
 * @author turri
 */
public class CreateStatDataFromRaw {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.statusdb");
    
    private static int N_THREADS = 3;
    
    private static long tstart = 0;
    
    private static long tstop = System.currentTimeMillis();
    
    private static SessionFactory fac;
    private static TimeRangeBroker timeBroker;
    private static CountDownLatch latch;
    
    /**
     * @param args the command line arguments
     */
    public static void main(String[] args) throws Exception {
        
        // define the command line options
        Options commandLineOptions = new Options();
        // The help option
        commandLineOptions.addOption("h", "help", false, "Print the help message");

        // Partition the table on the time column
        commandLineOptions.addOption("n", "threads_number", true, "Number of parallel threads");
        commandLineOptions.getOption("threads_number").setArgName("THREADS_NUMBER");

        commandLineOptions.addOption("t1", "t_start", true, "time origin");
        commandLineOptions.getOption("t_start").setArgName("T_START");

        commandLineOptions.addOption("t2", "t_stop", true, "end time");
        commandLineOptions.getOption("t_stop").setArgName("T_STOP");
        
        commandLineOptions.addOption("r", "range", true, "time range");
        commandLineOptions.getOption("range").setArgName("RANGE");
        
        CommandLineParser parser = new BasicParser();
        CommandLine line = parser.parse(commandLineOptions, args, false);

        if (line.hasOption("help")) {
            HelpFormatter formatter = new HelpFormatter();
            formatter.printHelp(100, "CreateStatDataFromRaw", "", commandLineOptions, "", true);
            return;
        }

        if (line.hasOption("threads_number")) {
            N_THREADS = Integer.parseInt(line.getOptionValue("threads_number"));
        }
        latch = new CountDownLatch(N_THREADS);
        
        if (line.hasOption("t_start")) {
            tstart = Long.parseLong(line.getOptionValue("t_start"));
        }
        if (line.hasOption("t_stop")) {
            tstop = Long.parseLong(line.getOptionValue("t_stop"));
        }
        long range = 86400000L;
        if (line.hasOption("range")) {
            range = Long.parseLong(line.getOptionValue("range"));
        }
        if (range % 1800000L != 0) {
            throw new RuntimeException("range must be a multiple of 1800000 (30 min)");
        }
        
        timeBroker = new TimeRangeBroker(tstart, tstop, range);
        
        fac = StatusdbUtils.getSessionFactory(null);
        
        DataServer d = new DataServer();
        
        // Adding default statDescs upstream, because adding them concurrently 
        // causes ConstraintViolationExceptions.
        Session sess = fac.openSession();
        Transaction tx = sess.beginTransaction();
        try {
            List<DataDesc> l = sess.createQuery("from DataDesc").list();
            for (DataDesc dd : l) {
                StatusDataPersister.getStatDescs(dd, sess);
            }
            tx.commit();
        } catch (Exception ex) {
            throw new RuntimeException("Persisting default statistical data failed. Aborting", ex);
        } finally {
            if (sess.isOpen()) {
                sess.close();
            }
        }
        
        for (int i = 0; i<N_THREADS; i++) {
            new Thread(new StatDataCompleter("completer-"+i, d)).start();
        }
        latch.await();
        log.info("All done.");
    }
    
    private static class StatDataCompleter implements Runnable {

        private final DataServer ds;
        private final DataChannel.DataChannelList data;
        private final StatusDataPersister dataPersister = new StatusDataPersister();
        private final String name;
        
        StatDataCompleter(String name, DataServer ds) {
            this.name = name;
            this.ds = ds;
            data = ds.getChannels();
        }

        @Override
        public void run() {
            
            long start = 0, stop = 0;
            
            while ( start < tstop ) {
                long[] range = timeBroker.getNextRange();
                if (range == null) break;
                
                start = range[0];
                stop = range[1];
                log.info(name + " : processing range ["+start+", "+stop+"]");
                for ( DataChannel ch : data.list ) {
                    List<RawData> rawData = ds.getRawData(ch.getId(),start, stop);
                    
                    if (rawData.isEmpty()) {
                        continue;
                    }
                    log.info(name + " : processing data " + ch.getPathAsString() + ", count : " + rawData.size());
                    Session sess = fac.openSession();
                    try {
                        Transaction tx = sess.beginTransaction();
                        for (RawData rd : rawData) {
                            dataPersister.getStatAccumulator().persist(rd, sess);
                        }                    
                        dataPersister.getStatAccumulator().flush(sess);
                        tx.commit();
                    } catch (Exception e) {
                        log.severe("thread " + name, e);
                    } finally {
                        if (sess.isOpen()) {
                            sess.close();
                        }
                    }
                }
                log.info(name + " : done");
            }
            latch.countDown();
        }
        
    }
    
    /**
     * A way to distribute time ranges to process among threads.
     */
    private static class TimeRangeBroker {
        
        private final long width;
        private final long end;
        private long tstart;
        
        TimeRangeBroker(long start, long end, long width) {
            this.tstart = start;
            this.end = end;
            this.width = width;
        }
        
        /**
         * @param range the range asked for
         * @return 
         */
        synchronized long[] getNextRange() {
            long[] range = new long[2];
            if (tstart >= end) {
                return null;
            }
            range[0]=tstart;
            // Normalizing range to the day
            long tstop = (range[0]+width)/width*width;
            range[1] = tstop >= end ? end : tstop;
            tstart = range[1];
            return range;
        }
    }
}
