package org.lsst.ccs.localdb.configdb;

import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;
import org.lsst.ccs.bus.data.ConfigurationInfo;
import org.lsst.ccs.bus.data.ConfigurationParameterInfo;
import org.lsst.ccs.config.ConfigurationDescription;
import org.lsst.ccs.config.ConfigurationView;
import org.lsst.ccs.localdb.configdb.model.Configuration;
import org.lsst.ccs.localdb.configdb.model.ConfigurationParameterValue;
import org.lsst.ccs.localdb.configdb.model.ConfigurationRun;
import org.lsst.ccs.localdb.configdb.model.Description;
import org.lsst.ccs.localdb.configdb.model.GlobalConfiguration;
import org.lsst.ccs.localdb.configdb.model.HibernateDAO;
import org.lsst.ccs.localdb.statusdb.model.AgentDesc;
import org.lsst.ccs.localdb.configdb.model.ConfigurationInfoData;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Configuration database high level facade.
 * @author LSST CCS Team
 */
public class ConfigurationFacade {
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.localdb.configdb");
    
    private final HibernateDAO dbInterface;
    
    public ConfigurationFacade(HibernateDAO dbInterface) {
        this.dbInterface = dbInterface;
    }
    
    /**
     * Finds or registers in database the agent description corresponding to the
     * given {@link org.lsst.ccs.bus.data.ConfigurationInfo} object.
     *
     * @param ad the agent description
     * @param cpis the list of configuration parameter info.
     * @return the id of the corresponding {@code Description} entity
     */
    public Description registerDescription(AgentDesc ad, List<ConfigurationParameterInfo> cpis) throws Exception {
        dbInterface.begin();
        Description res = dbInterface.findDescriptionOrCreate(ad, cpis);
        dbInterface.end();
        return res;
    }
    
    public void stopConfigurationRuns(Description desc, long time) throws Exception {
        dbInterface.begin();
        Map<String, ConfigurationRun> activeRuns = dbInterface.getActiveConfigurationRuns(desc);
        for (ConfigurationRun cr : activeRuns.values()) {
            cr.setTstop(time);
        }
        dbInterface.end();
    }
    
    /**
     * Registration of configuration information coming from the Command bus.
     * There is no associated StatusConfigurationInfo yet.
     *
     * @param desc
     * @param ci
     * @return
     * @throws java.lang.Exception
     */
    public ConfigurationDescription registerStandaloneConfiguration(Description desc, ConfigurationInfo ci) throws Exception {
        dbInterface.begin();
        Map<String, ConfigurationRun> newRuns = registerRuns(desc, ci, ci.getTimestamp());
        
        ConfigurationDescription cd = new ConfigurationDescription(ci.getCategorySet());
        
        for (ConfigurationRun  cr : newRuns.values()) {
            cd.putTagForCategory(cr.getConfiguration().getCategory(), cr.getConfiguration().getConfigName(), cr.getConfiguration().getVersion());
        }
        dbInterface.end();
        return cd;
    }
    
    /**
     * This methods resolves the association between the ConfigurationInfo
     * registered previously via the command bus and the subsequent 
     *
     * @param desc the subsystem description
     * @param cid the ConfigurationInfoData to persist
     */
    public void registerStandaloneStatusConfigurationInfo(Description desc, ConfigurationInfoData cid) throws Exception {
        dbInterface.begin();
        Map<String, ConfigurationRun> currentRuns = dbInterface.getActiveConfigurationRuns(desc);
        cid.setConfigurationRuns(currentRuns);
        dbInterface.persist(cid);
        dbInterface.end();
    }
    
    /**
     * Registration of an incoming {@code ConfigurationInfo} object via the Status Bus. The
     * registration happens by comparing the received {@code ConfigurationInfo}
     * with the previous configuration state persisted in the current
     * {@code ConfigurationRun}
     *
     * @param desc
     * @param ci
     * @param cid the associated ConfigurationInfoData entity

      */
    public void registerConfiguration(Description desc, ConfigurationInfo ci, ConfigurationInfoData cid) throws Exception {
        long time = cid.getTime();
        cid.setDescription(desc);
        dbInterface.begin();
        Map<String, ConfigurationRun> newRuns = registerRuns(desc, ci, time);
        // Taking care of the global configuration

        cid.setConfigurationRuns(newRuns);
        dbInterface.persist(cid);
        log.fine("persisted configuration state at : " + cid.getTime() + " : " + cid.getConfigurationDescriptionString());
        dbInterface.end();
    }
    
    private Map<String, ConfigurationRun> registerRuns(Description desc, ConfigurationInfo ci, long time) {

        Map<String, ConfigurationRun> currentRuns = dbInterface.getActiveConfigurationRuns(desc);
        Map<String, ConfigurationRun> newRuns = new HashMap<>();
        Map<String, List<ConfigurationParameterInfo>> cpiMap = ConfigurationInfo.getParameterInfoGroupByCategory(ci.getAllParameterInfo());

        Map<String, Configuration> newConfigs = new HashMap<>();

        // First pass : finding the base configurations
        for (String cat : ci.getCategorySet()) {
            String nextTag = ci.getConfigNameForCategory(cat);
            Integer version = ci.getConfigVersion(cat);
            newConfigs.put(cat, dbInterface.getConfigurationOrCreate(desc, nextTag, cat, version, cpiMap.get(cat), time, currentRuns.get(cat)));
        }

        // Finding the globalConfiguration
        GlobalConfiguration gc = null;
        // The global configuration information is only set when the state is CONFIGURED (no run time changes)
        if (ci.getGlobalName() != null && !ci.hasChanges()) {
            gc = dbInterface.findGlobalConfigurationOrCreate(desc, ci.getGlobalName(), newConfigs);
        }

        // Second pass : create and populate the configuration runs
        for (String cat : ci.getCategorySet()) {
            Configuration baseConfig = newConfigs.get(cat);
            ConfigurationRun newRun = new ConfigurationRun(baseConfig, time);
            newRun.setGlobalConfiguration(gc);
            ConfigurationRun currentRun = currentRuns.get(cat);
            List<ConfigurationParameterInfo> changesForCat = cpiMap.get(cat)
                    .stream().filter(
                            (cpi) -> {
                                // Getting rid of final and non dirty parameters
                                return !cpi.isFinal() && cpi.isDirty();
                            }
                    ).collect(Collectors.toList());
            
            if (currentRun == null) {
                /** there is no previous run to compare with. */
                newRun = dbInterface.populateRunFromScratch(newRun, changesForCat);
            } else {
                if (time <= currentRun.getTstart()) {
                    if(time < currentRun.getTstart()) {
                        log.info("for " + desc.getAgentDesc().getAgentName() + ": received configuration info time ("+new Date(time)+") is previous to an already persisted configuration state at " + new Date(currentRun.getTstart()));
                    }
                    newRun = currentRun;
                } else {
                    newRun = dbInterface.populateRunFromPreviousRun(currentRun, newRun, changesForCat);
                }
            }

            newRuns.put(cat, newRun);
        }
        return newRuns;
    }
    
    public ConfigurationView loadConfiguration(Description desc, ConfigurationDescription configDesc) throws Exception {
        ConfigurationView view = new ConfigurationView(configDesc);
        dbInterface.begin();
        try {
            for (Map.Entry<String, String> tags : configDesc.getCategoryTags().entrySet()) {
                Configuration request = dbInterface.getConfigurationOrNull(desc, tags.getValue(), tags.getKey(), configDesc.getCategoryVersions().get(tags.getKey()));
                if (request == null) {
                    throw new RuntimeException("cannot find configuration \""+tags.getKey()+"\":\""+tags.getValue()+"\"("+configDesc.getCategoryVersions().get(tags.getKey())+")");
                }
                for(ConfigurationParameterValue cpv : request.getConfigurationParameterValues().values()) {
                    view.putParameterValue(cpv.getPath().getComponentName(), cpv.getPath().getParameterName(), cpv.getValue());
                }
            }
        } finally {
            dbInterface.end();
        }
        return view;
    }
    
    public ConfigurationView loadGlobalConfiguration(Description desc, String name, int version) throws Exception {
        dbInterface.begin();
        GlobalConfiguration gc = dbInterface.findGlobalConfiguration(desc, name, version);
        ConfigurationView res = null;
        if (gc != null) {
            ConfigurationDescription cd = new ConfigurationDescription(gc.getConfigurations().keySet());
            cd.setName(gc.getName(), gc.getVersion());
            res = new ConfigurationView(cd);
            for(Configuration c : gc.getConfigurations().values()) {
                cd.putTagForCategory(c.getCategory(), c.getCategory(), c.getVersion());
                for (ConfigurationParameterValue cpv : c.getConfigurationParameterValues().values()) {
                    res.putParameterValue(cpv.getPath().getComponentName(), cpv.getPath().getParameterName(), cpv.getValue());
                }
            }
        }
        dbInterface.end();
        return res;
    }
    
    // Read only operation
    public Set<String> findAvailableConfigurationsForCategory(Description desc, String category) throws Exception {
        Set<String> res = new HashSet<>();
        dbInterface.begin();
        res.addAll(dbInterface.findAvailableConfigurationsForCategory(desc, category));
        dbInterface.end();
        return res;
    }

    void cleanupConfigurationRuns(Description desc, long time) throws Exception {
        dbInterface.begin();
        dbInterface.cleanupConfigurationRuns(desc, time);
        dbInterface.end();
    }
}
