package org.lsst.ccs.localdb.statusdb;

import java.util.ArrayList;
import java.util.Queue;
import java.util.concurrent.ConcurrentLinkedQueue;
import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.Transaction;
import org.hibernate.resource.transaction.spi.TransactionStatus;
import static org.lsst.ccs.localdb.statusdb.StatusPersister.log;

/**
 * Persists entities as a batch.
 *
 * @author LSST CCS Team
 */
public abstract class BatchPersister implements Runnable {
    
    private final Queue<Object[]> rq = new ConcurrentLinkedQueue<>();
    
    private final SessionFactory fac;
    
    public BatchPersister(SessionFactory fac) {
        this.fac = fac;
    }
    
    @Override
    public void run() {
        
        ArrayList<Object[]> workingList = new ArrayList<>(1000);
        boolean empty = false;
        
        while (!empty) {
            // get at most 1000 objets from the queue
            for (int i = 0; i < 1000; i++) {
                Object[] d = rq.poll();
                if (d == null) {
                    break;
                }
                workingList.add(d);
            }
            
            if (workingList.isEmpty()) {
                // Will sleep 1 second
                empty = true;
            } else {
                log.debug(this.getClass().getSimpleName() + " start batch processing of " + workingList.size() + " entities.");
                Session sess = fac.openSession();
                Transaction tx = null;
                try {
                    tx = sess.beginTransaction();
                    for (Object[] o : workingList) {
                        try {
                            persist(o, sess);
                        } catch (Exception e) {
                            throw new Exception("Failed to persist object ("+o[1]+") "+o[0], e);
                        }
                    }
                    tx.commit();
                } catch (Exception ex) {
                    log.error("caught exception when persisting : " + ex, ex);
                    try {
                        if (tx != null && (
                                tx.getStatus() == TransactionStatus.ACTIVE
                                || tx.getStatus() == TransactionStatus.MARKED_ROLLBACK)) {
                            tx.rollback();
                        }
                    } catch (Exception rbEx) {
                        log.error("Rollback of transaction failed : " + rbEx, rbEx);
                    }
                } finally {
                    if (sess.isOpen()) {
                        sess.close();
                    }
                    workingList.clear();
                }
            }
        }
        log.debug(this.getClass().getSimpleName() + " leaving batch processing loop");
    }
    
    public void addToQueue(Object[] obj) {
        rq.add(obj);
    }
    
    /**
     * Called from inside an open transaction.
     *
     * @param obj
     * @param sess
     */
    public abstract void persist(Object[] obj, Session sess);
    
}
