package org.lsst.ccs.localdb.configdb;

import org.lsst.ccs.localdb.configdb.model.ConfigProfile;
import org.lsst.ccs.localdb.configdb.model.ParameterConfiguration;
import org.lsst.ccs.localdb.configdb.model.SubsystemDescription;

import java.rmi.Remote;
import java.rmi.RemoteException;
import java.util.List;

/**
 * Abstract services definition for Configuration Service.
 * This is using the semantics of RMI (such as <TT>Remote</TT>
 * or <TT>RemoteException</TT>) but can be used in other contexts.
 * @author bamade
 */
// Date: 06/06/12

public interface ConfigurationService extends Remote {
    /*
     TODO: remote deprecationListener
     todo: get active configuration for machine ?
     */
//////////////////////////////// METHODS FOR REGISTERING/DEPRECATING OBJECTS
    /**
     * Registers in the configuration service a new subsystem description (created through
     * <TT>Factories</TT> code.
     * @param newDescription do not use this object any more after the call
     * @return a data structure containing a reference to an immutable registered subsystem (use <TT>getResult()</TT>)
     *    a possible deprecated subsystem which has been replaced by the newDescription
     *     and a list of possible deprecated <TT>ConfigProfile</TT> that were referencing the obsolete subsystem
     * @throws RemoteException
     */
    DescriptionResult registerSubsystemDescription(SubsystemDescription newDescription) throws RemoteException;

    /**
     * Deprecates a registered subsystem.
     * @param subsystemName
     * @param tag
     * @return a data structure with the deprecated subsystem (or null if not found) and a list of <TT>ConfigProfile</TT>
     *         that may have been deprecated
     * @throws RemoteException
     */
    DescriptionResult deprecateSubsystemDescription(String subsystemName, String tag) throws RemoteException;

    /**
     * fetch the active Description with name and tag
     * @param name
     * @param tag
     * @return
     * @throws RemoteException
     */
     SubsystemDescription getActiveSubsystemDescription( String name, String tag) throws RemoteException ;

    /**
     * Registers a new <TT>ConfigProfile</TT> in the configuration service.
     * The object must have been created with <TT>Factories</TT> using a registered valid <TT>SubsystemDescription</TT>
     * @param newProfile do not use this object any more after the call
     * @return a data structure containing an immutable registered Objecta (use <TT>getResult()</TT>)
     *       and possibly a deprecated profile it replaces.
     * @throws RemoteException
     */
    ProfileResult registerConfigProfile(ConfigProfile newProfile) throws RemoteException;

    /**
     * typical remote service: a local properties structure is base for creating a new
     * ConfigProfile
     * @param subsystemName
     * @param categoryName 
     * @param configurationName
     * @param tag
     * @param user
     * @param level
     * @return
     * @throws RemoteException
     */
    public ProfileResult registerConfiguration(String subsystemName, String categoryName, String configurationName, String tag,
                                                 String user, int level) throws RemoteException;
    /**
     *  Deprecates a valid Configuration Profile.
     * @param name configuration name
     * @param tag
     * @return a data structure containing the deprecated object (or null if not found)
     * @throws RemoteException
     */
    ProfileResult deprecateConfigProfile(String subsystemName, String descriptionName, String name, String tag) throws RemoteException;

    /**
     * gets an active ConfigProfile with anme and tag
     * @param name
     * @param tag
     * @return null if no such configuration
     * @throws RemoteException
     */
     ConfigProfile getActiveConfigProfile(String subsystemName, String descriptionName, String name, String tag) throws RemoteException ;

    /**
     * Registers a modification to a <TT>ParameterConfiguration</TT> operated
     * through <TT>ConfigProfile.temporaryChangeConfigurationValue</TT>
     * @param engineeringModeParm  a <TT>ParameterConfiguration</TT> object which is registered in a registered
     *                             <TT>ConfigProfile</TT> object in engineering mode (see <TT>Factories.copyProfile</TT>
     *                             and register the resulting profile before calling this method)
     * @return its argument (can be reused for other calls of temporaryChange)
     * @throws RemoteException
     */
    //TODO change that! remot and local are different
    ParameterConfiguration modifyParmConf(ParameterConfiguration engineeringModeParm) throws RemoteException;



///////////////////////////////// METHODS LINKED TO A VALUE IN TIME

    /**
     * Gets a <TT>ConfigProfile</TT> which was (or is) valid at a precise moment.
     * <P>
     * NOTE: this method will not work on "dummy local services"
     * @param configName
     * @param configTag
     * @param date
     * @return null if not found
     * @throws RemoteException
     */
    ConfigProfile getConfigValidAt(String subsystemName, String configName, String configTag, long date) throws RemoteException;

    /**
     * Gets the value of a Parameter which was valid at a moment in time.
     * <P>
     * NOTE: this method will not work on "dummy local services"
     * <P>
     *     there is no need for a susbsystem name since at a moment in time there is only one subsystem known to a combintaiton of
     *     config name and tag.
     * @param configName
     * @param configTag
     * @param parameterPath
     * @param date
     * @return
     * @throws RemoteException
     */
    String getValueValidAt( String subsystemName, String configName, String configTag, String parameterPath, long date) throws RemoteException;

    /**
     * Returns the Configuration which was running for a given subsystem at a moment in time
     * <P>
     * NOTE: this method will not work on "dummy local services"
     * @param subsystemName
     * @param date
     * @return
     * @throws RemoteException
     */
    ConfigProfile getConfigRunningAt(String subsystemName, String descriptionName, String category, long date) throws RemoteException;

    /**
     * gets the value of a parameter for a subsystem which was active at a moment in time.
     * <P>
     * NOTE: this method will not work on "dummy local services"
     * @param subsystemName
     * @param parameterPath
     * @param date
     * @return
     * @throws RemoteException
     */
    String getActiveValueAt(String subsystemName, String descriptionName, String category, String parameterPath, long date) throws RemoteException ;


/////////////////////////////////// LINKED DATA

    /**
     * gets a (deprecated) <TT>ConfigProfile</TT> with same name and tag
     * as the argument.
     * @param current
     * @return
     * @throws RemoteException
     */
    ConfigProfile getPrevious(ConfigProfile current) throws RemoteException;

    /**
     * Gets a (deprecated or "alive") <TT>ConfigProfile</TT> with same name and tag
     * as the argument
     * @param current
     * @return
     * @throws RemoteException
     */
    ConfigProfile getNext(ConfigProfile current) throws RemoteException;

    /**
     * Gets the (deprecated) <TT>SubsystemDescription</TT> with same characteritics as the argument
     * @param current
     * @return
     * @throws RemoteException
     */
    SubsystemDescription getPrevious(SubsystemDescription current) throws RemoteException;

//////////////////////////////////// GENERAL METHOD

    /**
     * can send a simple HQL request to the underlying Database.
     * It is assumed that the request is only a "read" (a SELECT equivalent) -though this is not checked-
     * <P>
     * NOTE: this method will not work on "dummy local services"
     * @param hqlSelectString
     * @return
     * @throws RemoteException
     */
    List<?> simpleHQLRequest(String hqlSelectString) throws RemoteException;

}
